/*
 * Copyright (C) 2009-2011 Freescale Semiconductor, Inc.  All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation 
 * files (the "Software"), to deal in the Software without 
 * restriction, including without limitation the rights to use, copy, 
 * modify, merge, publish, distribute, sublicense, and/or sell copies 
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 * SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <linux/fb.h>
#include <linux/mxcfb.h>

#include "xf86.h"
#include "xf86_OSproc.h"

#include "mipointer.h"
#include "mibstore.h"
#include "micmap.h"
#include "colormapst.h"
#include "xf86cmap.h"
#include "exa.h"

/* for visuals */
#include "fb.h"
#include "fbdevhw.h"

#if GET_ABI_MAJOR(ABI_VIDEODRV_VERSION) < 6
#include "xf86Resources.h"
#include "xf86RAC.h"
#endif


#include "imx_type.h"

#if IMX_XVIDEO_ENABLE
#include "xf86xv.h"
#endif


#define IMX_NAME		"imx"
#define IMX_DRIVER_NAME		"imx"

#define IMX_VERSION_MAJOR	PACKAGE_VERSION_MAJOR
#define IMX_VERSION_MINOR	PACKAGE_VERSION_MINOR
#define IMX_VERSION_PATCH	PACKAGE_VERSION_PATCHLEVEL

#define IMX_VERSION_CURRENT \
	((IMX_VERSION_MAJOR << 20) |\
	(IMX_VERSION_MINOR << 10) | \
	 (IMX_VERSION_PATCH))



#if IMX_XVIDEO_ENABLE
/* for Xvideo */
extern int MXXVInitializeAdaptor(ScrnInfoPtr, XF86VideoAdaptorPtr **);
#endif

/* for EXA (X acceleration) */
extern void IMX_EXA_GetRec(ScrnInfoPtr pScrn);
extern void IMX_EXA_FreeRec(ScrnInfoPtr pScrn);
extern Bool IMX_EXA_PreInit(ScrnInfoPtr pScrn);
extern Bool IMX_EXA_ScreenInit(int scrnIndex, ScreenPtr pScreen);
extern Bool IMX_EXA_CloseScreen(int scrnIndex, ScreenPtr pScreen);
extern Bool IMX_EXA_GetPixmapProperties(PixmapPtr pPixmap, void** pPhysAddr, int* pPitch);

/* for X extension */
extern void IMX_EXT_Init();


enum { IMX_ROTATE_NONE=0, IMX_ROTATE_CW=270, IMX_ROTATE_UD=180, IMX_ROTATE_CCW=90 };


/* -------------------------------------------------------------------- */

/*
 * This is intentionally screen-independent.  It indicates the binding
 * choice made in the first PreInit.
 */
static int pix24bpp = 0;

typedef enum {
	IMX_TYPE_FB,
	IMX_TYPE_FB_EPDC,	/* for mx50 */
	IMX_TYPE_FB_UNKNOWN
} IMXTypeFB;

static IMXTypeFB
imxGetFrameBufferType(struct fb_fix_screeninfo* pFixInfo)
{
	if (0 == strcmp("mxc_epdc_fb", pFixInfo->id)) {
		return IMX_TYPE_FB_EPDC;
	}
#if 0
	if (0 == strcmp("DISP3 BG", pFixInfo->id)) {
		return IMX_TYPE_FB;
	}
	return IMX_TYPE_FB_UNKNOWN;
#else
	return IMX_TYPE_FB;
#endif
}

/* Supported "chipsets" */
static SymTabRec imxChipsets[] = {
    { 0, "i.MX5x Z160" },
    {-1, NULL }
};

/* Supported options */
typedef enum {
	OPTION_FBDEV,
	OPTION_FORMAT_EPDC,
	OPTION_NOACCEL,
	OPTION_ACCELMETHOD,
	OPTION_ROTATE
} IMXOpts;

#define	OPTION_STR_FBDEV	"fbdev"
#define	OPTION_STR_FORMAT_EPDC	"FormatEPDC"
#define	OPTION_STR_NOACCEL	"NoAccel"
#define	OPTION_STR_ACCELMETHOD	"AccelMethod"
#define	OPTION_STR_ROTATE	"Rotate"

static const OptionInfoRec imxOptions[] = {
	{ OPTION_FBDEV,		OPTION_STR_FBDEV,	OPTV_STRING,	{0},	FALSE },
	{ OPTION_FORMAT_EPDC,	OPTION_STR_FORMAT_EPDC,	OPTV_STRING,	{0},	FALSE },
	{ OPTION_NOACCEL,	OPTION_STR_NOACCEL,	OPTV_BOOLEAN,	{0},	FALSE },
	{ OPTION_ACCELMETHOD,	OPTION_STR_ACCELMETHOD,	OPTV_STRING,	{0},	FALSE },
	{ OPTION_ROTATE,	OPTION_STR_ROTATE,	OPTV_STRING,	{0},	FALSE },
	{ -1,			NULL,			OPTV_NONE,	{0},	FALSE }
};

/* -------------------------------------------------------------------- */

static IMXPtr
imxGetRec(ScrnInfoPtr pScrn)
{
	IMXPtr fPtr = NULL;

	if (NULL == pScrn->driverPrivate) {
	
		pScrn->driverPrivate = xnfcalloc(sizeof(IMXRec), 1);
		if (NULL == pScrn->driverPrivate) {
			xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			   	"unable to allocate driver private memory\n");
			return NULL;
		}

		fPtr = IMXPTR(pScrn);
		fPtr->useAccel = FALSE;

		IMX_EXA_GetRec(pScrn);
	}

	return fPtr;
}

static void
imxFreeRec(ScrnInfoPtr pScrn)
{
	if (NULL != pScrn->driverPrivate) {
		IMX_EXA_FreeRec(pScrn);
		xfree(pScrn->driverPrivate);
		pScrn->driverPrivate = NULL;
	}
}

/* -------------------------------------------------------------------- */

static Bool
imxPreInitEPDC(ScrnInfoPtr pScrn, IMXPtr fPtr, int fd, char* strFormat, char* strRotate)
{
	Bool result = TRUE;

	/* Get frame buffer variable screen info */
	struct fb_var_screeninfo fbVarScreenInfo;
	if (-1 == ioctl(fd,FBIOGET_VSCREENINFO,(void*)(&fbVarScreenInfo))) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			   "FBIOGET_VSCREENINFO: %s\n", strerror(errno));
		return FALSE;
	}

	/* Find the requested EPDC format and change if requested */
	if (NULL != strFormat) {
		if (0 == xf86NameCmp(strFormat, "RGB565")) {
			fbVarScreenInfo.grayscale = 0;
			fbVarScreenInfo.bits_per_pixel = 16;
		}
		else if (0 == xf86NameCmp(strFormat, "Y8")) {
			fbVarScreenInfo.grayscale = GRAYSCALE_8BIT;
			fbVarScreenInfo.bits_per_pixel = 8;
		}
		else if (0 == xf86NameCmp(strFormat, "Y8INV")) {
			fbVarScreenInfo.grayscale = GRAYSCALE_8BIT_INVERTED;
			fbVarScreenInfo.bits_per_pixel = 8;
		}
		else {
			xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
				"\"%s\" is not a valid value for Option \"%s\"\n", strFormat, OPTION_STR_FORMAT_EPDC);
			xf86DrvMsg(pScrn->scrnIndex, X_INFO,
				"valid options are \"RGB565\", \"Y8\" and \"Y8INV\"\n");
			return FALSE;
		}
	}

	/* Get the screen rotation */
	fbVarScreenInfo.rotate = FB_ROTATE_UR;
	if (NULL != strRotate) {
		if (0 == xf86NameCmp(strRotate, "UR")) {
			fbVarScreenInfo.rotate = FB_ROTATE_UR;
			xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
				"EPDC frame buffer output oriented up-right\n");
		} else if (0 == xf86NameCmp(strRotate, "CW")) {
			fbVarScreenInfo.rotate = FB_ROTATE_CW;
			xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
				"rotating EPDC frame buffer output clockwise\n");
		} else if (0 == xf86NameCmp(strRotate, "CCW")) {
			fbVarScreenInfo.rotate = FB_ROTATE_CCW;
			xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
				"rotating EPDC frame buffer output counter-clockwise\n");
		} else if (0 == xf86NameCmp(strRotate, "UD")) {
			fbVarScreenInfo.rotate = FB_ROTATE_UD;
			xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
				"rotating EPDC frame buffer output upside-down\n");
		} else {
			xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
				"\"%s\" is not a valid value for Option \"Rotate\"\n", strRotate);
		}
	}

	/* It is required to initialize the device so */
	/* use force activation just in case nothing changed */
	fbVarScreenInfo.activate = FB_ACTIVATE_FORCE;
	if (-1 == ioctl(fd,FBIOPUT_VSCREENINFO,(void*)(&fbVarScreenInfo))) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			   "FBIOPUT_VSCREENINFO: %s\n", strerror(errno));
		return FALSE;
	}

	return TRUE;
}

static Bool
imxPreInit(ScrnInfoPtr pScrn, int flags)
{
	int default_depth, fbbpp;
	const char *s;
	int type;

	/* Do not auto probe */
	if (flags & PROBE_DETECT) {
		return FALSE;
	}

	/* Check the number of entities, and fail if it isn't one. */
	if (pScrn->numEntities != 1) {
		return FALSE;
	}

	/* Use the current monitor set in xorg.conf. */
	/* X needs this to be set. */
	pScrn->monitor = pScrn->confScreen->monitor;

	/* Allocate driver private data associated with screen. */
	IMXPtr fPtr = imxGetRec(pScrn);
	if (NULL == fPtr) {
		return FALSE;
	}

	fPtr->pEnt = xf86GetEntityInfo(pScrn->entityList[0]);

	/* Access the name of the frame buffer device */
	char* fbdevName = xf86FindOptionValue(fPtr->pEnt->device->options, OPTION_STR_FBDEV);
	if (NULL == fbdevName) {
		xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
			"Option \"%s\" missing\n", OPTION_STR_FBDEV);
		return FALSE;
	}

	/* Open the frame buffer device */
	int fd = open(fbdevName,O_RDWR,0);
	if (fd == -1) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			"open %s: %s\n", fbdevName, strerror(errno));
		return FALSE;
	}

	/* Get frame buffer fixed screen info */
	struct fb_fix_screeninfo fbFixScreenInfo;
	if (-1 == ioctl(fd,FBIOGET_FSCREENINFO,(void*)(&fbFixScreenInfo))) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			   "FBIOGET_FSCREENINFO: %s\n", strerror(errno));
		close(fd);
		return FALSE;
	}

	/* Special case for EPDC driver */
	if (IMX_TYPE_FB_EPDC == imxGetFrameBufferType(&fbFixScreenInfo)) {

		/* Get the format for the EPDC */
		char* strFormat = xf86FindOptionValue(fPtr->pEnt->device->options, OPTION_STR_FORMAT_EPDC);

		/* Get the rotation for the EPDC */
		char* strRotate = xf86FindOptionValue(fPtr->pEnt->device->options, OPTION_STR_ROTATE);

		/* Perform pre-init on EPDC */
		if (!imxPreInitEPDC(pScrn, fPtr, fd, strFormat, strRotate)) {
			close(fd);
			return FALSE;
		}
	}
	close(fd);

	/* Open device */
	if (!fbdevHWInit(pScrn,NULL,fbdevName))
		return FALSE;
	default_depth = fbdevHWGetDepth(pScrn,&fbbpp);
	if (!xf86SetDepthBpp(pScrn, default_depth, default_depth, fbbpp,
			     Support24bppFb | Support32bppFb | SupportConvert32to24 | SupportConvert24to32))
		return FALSE;
	xf86PrintDepthBpp(pScrn);

	/* Get the depth24 pixmap format */
	if (pScrn->depth == 24 && pix24bpp == 0)
		pix24bpp = xf86GetBppFromDepth(pScrn, 24);

	/* Color weight */
	if (pScrn->depth > 8) {
		rgb zeros = { 0, 0, 0 };
		if (!xf86SetWeight(pScrn, zeros, zeros))
			return FALSE;
	}

	/* Visual init */
	if (!xf86SetDefaultVisual(pScrn, -1))
		return FALSE;

	/* We don't currently support DirectColor at > 8bpp */
	if (pScrn->depth > 8 && pScrn->defaultVisual != TrueColor) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "requested default visual"
			   " (%s) is not supported at depth %d\n",
			   xf86GetVisualName(pScrn->defaultVisual), pScrn->depth);
		return FALSE;
	}

	{
		Gamma zeros = {0.0, 0.0, 0.0};

		if (!xf86SetGamma(pScrn,zeros)) {
			return FALSE;
		}
	}

	pScrn->progClock = TRUE;
	pScrn->rgbBits   = 8;
	pScrn->chipset   = IMX_DRIVER_NAME;
	pScrn->videoRam  = fbdevHWGetVidmem(pScrn);

	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "hardware: %s (video memory:"
		   " %dkB)\n", fbdevHWGetName(pScrn), pScrn->videoRam/1024);

	/* Handle options */
	xf86CollectOptions(pScrn, NULL);
	if (!(fPtr->Options = xalloc(sizeof(imxOptions))))
		return FALSE;
	memcpy(fPtr->Options, imxOptions, sizeof(imxOptions));
	xf86ProcessOptions(pScrn->scrnIndex, fPtr->pEnt->device->options, fPtr->Options);

	/* NoAccel option */
	fPtr->useAccel = TRUE;
	if (xf86ReturnOptValBool(fPtr->Options, OPTION_NOACCEL, FALSE)) {
		fPtr->useAccel = FALSE;
	}

	/* AccelMethod option */
	if (fPtr->useAccel) {
		s = xf86GetOptValString(fPtr->Options, OPTION_ACCELMETHOD);
		if ((NULL != s) && (0 != xf86NameCmp(s, "EXA"))) {
			fPtr->useAccel = FALSE;
		}
	} 


	/* Select video modes */
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "checking modes against framebuffer device...\n");
	fbdevHWSetVideoModes(pScrn);

	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "checking modes against monitor...\n");
	{
		DisplayModePtr mode, first = mode = pScrn->modes;
		
		if (mode != NULL) do {
			mode->status = xf86CheckModeForMonitor(mode, pScrn->monitor);
			mode = mode->next;
		} while (mode != NULL && mode != first);

		xf86PruneDriverModes(pScrn);
	}

	if (NULL == pScrn->modes)
		fbdevHWUseBuildinMode(pScrn);
	pScrn->currentMode = pScrn->modes;

	/* First approximation, may be refined in ScreenInit */
	pScrn->displayWidth = pScrn->virtualX;

	xf86PrintModes(pScrn);

	/* Set display resolution */
	xf86SetDpi(pScrn, 0, 0);

	/* Load bpp-specific modules */
	switch ((type = fbdevHWGetType(pScrn)))
	{
	case FBDEVHW_PLANES:
                xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                          "plane mode is not supported by the imx driver\n");
		return FALSE;
		break;
	case FBDEVHW_PACKED_PIXELS:
		switch (pScrn->bitsPerPixel)
		{
		case 8:
		case 16:
		case 24:
		case 32:
			break;
		default:
			xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			"unsupported number of bits per pixel: %d",
			pScrn->bitsPerPixel);
			return FALSE;
		}
		break;
	case FBDEVHW_INTERLEAVED_PLANES:
               /* Not supported yet, don't know what to do with this */
               xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                          "interleaved planes are not yet supported by the "
			  "imx driver\n");
		return FALSE;
	case FBDEVHW_TEXT:
               /* This should never happen ...
                * we should check for this much much earlier ... */
               xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                          "text mode is not supported by the fbdev driver\n");
		return FALSE;
       case FBDEVHW_VGA_PLANES:
               /* Not supported yet */
               xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                          "EGA/VGA planes are not yet supported by the imx "
			  "driver\n");
               return FALSE;
       default:
               xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                          "unrecognised imx hardware type (%d)\n", type);
               return FALSE;
	}
	if (xf86LoadSubModule(pScrn, "fb") == NULL) {
		imxFreeRec(pScrn);
		return FALSE;
	}

	/* Perform EXA pre-init */
	if (fPtr->useAccel) {

		if (!IMX_EXA_PreInit(pScrn)) {
			imxFreeRec(pScrn);
			return FALSE;
		}
	}
	
	return TRUE;
}

static Bool
imxCloseScreen(int scrnIndex, ScreenPtr pScreen)
{
	IMX_EXA_CloseScreen(scrnIndex, pScreen);

	ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
	IMXPtr fPtr = IMXPTR(pScrn);

	fbdevHWRestore(pScrn);
	fbdevHWUnmapVidmem(pScrn);
	pScrn->vtSema = FALSE;

	pScreen->CloseScreen = fPtr->CloseScreen;
	return (*pScreen->CloseScreen)(scrnIndex, pScreen);
}


static Bool
imxScreenInit(int scrnIndex, ScreenPtr pScreen, int argc, char **argv)
{
	ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
	IMXPtr fPtr = IMXPTR(pScrn);
	VisualPtr visual;
	int init_picture = 0;
	int ret, flags;
	int type;

#if DEBUG
	ErrorF("\tbitsPerPixel=%d, depth=%d, defaultVisual=%s\n"
	       "\tmask: %x,%x,%x, offset: %d,%d,%d\n",
	       pScrn->bitsPerPixel,
	       pScrn->depth,
	       xf86GetVisualName(pScrn->defaultVisual),
	       pScrn->mask.red,pScrn->mask.green,pScrn->mask.blue,
	       pScrn->offset.red,pScrn->offset.green,pScrn->offset.blue);
#endif

	if (NULL == (fPtr->fbmem = fbdevHWMapVidmem(pScrn))) {
	        xf86DrvMsg(scrnIndex,X_ERROR,"mapping of video memory"
			   " failed\n");
		return FALSE;
	}
	fPtr->fboff = fbdevHWLinearOffset(pScrn);

	fbdevHWSave(pScrn);

	if (!fbdevHWModeInit(pScrn, pScrn->currentMode)) {
		xf86DrvMsg(scrnIndex,X_ERROR,"mode initialization failed\n");
		return FALSE;
	}
	fbdevHWSaveScreen(pScreen, SCREEN_SAVER_ON);
	fbdevHWAdjustFrame(scrnIndex,0,0,0);

	/* mi layer */
	miClearVisualTypes();
	if (pScrn->bitsPerPixel > 8) {
		if (!miSetVisualTypes(pScrn->depth, TrueColorMask, pScrn->rgbBits, TrueColor)) {
			xf86DrvMsg(scrnIndex,X_ERROR,"visual type setup failed"
				   " for %d bits per pixel [1]\n",
				   pScrn->bitsPerPixel);
			return FALSE;
		}
	} else {
		if (!miSetVisualTypes(pScrn->depth,
				      miGetDefaultVisualMask(pScrn->depth),
				      pScrn->rgbBits, pScrn->defaultVisual)) {
			xf86DrvMsg(scrnIndex,X_ERROR,"visual type setup failed"
				   " for %d bits per pixel [2]\n",
				   pScrn->bitsPerPixel);
			return FALSE;
		}
	}
	if (!miSetPixmapDepths()) {
	  xf86DrvMsg(scrnIndex,X_ERROR,"pixmap depth setup failed\n");
	  return FALSE;
	}

	/* FIXME: this doesn't work for all cases, e.g. when each scanline
		has a padding which is independent from the depth (controlfb) */
	pScrn->displayWidth = fbdevHWGetLineLength(pScrn) /
			      (pScrn->bitsPerPixel / 8);

	if (pScrn->displayWidth != pScrn->virtualX) {
		xf86DrvMsg(scrnIndex, X_INFO,
			   "Pitch updated to %d after ModeInit\n",
			   pScrn->displayWidth);
	}

	fPtr->fbstart = fPtr->fbmem + fPtr->fboff;

	switch ((type = fbdevHWGetType(pScrn)))
	{
	case FBDEVHW_PACKED_PIXELS:
		switch (pScrn->bitsPerPixel) {
		case 8:
		case 16:
		case 24:
		case 32:
			ret = fbScreenInit(pScreen, fPtr->fbstart,
					   pScrn->virtualX, pScrn->virtualY,
					   pScrn->xDpi, pScrn->yDpi,
					   pScrn->displayWidth,
					   pScrn->bitsPerPixel);
			init_picture = 1;
			break;
	 	default:
			xf86DrvMsg(scrnIndex, X_ERROR,
				   "internal error: invalid number of bits per"
				   " pixel (%d) encountered in"
				   " imxScreenInit()\n", pScrn->bitsPerPixel);
			ret = FALSE;
			break;
		}
		break;
	case FBDEVHW_INTERLEAVED_PLANES:
		/* This should never happen ...
		* we should check for this much much earlier ... */
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: interleaved planes are not yet "
			   "supported by the imx driver\n");
		ret = FALSE;
		break;
	case FBDEVHW_TEXT:
		/* This should never happen ...
		* we should check for this much much earlier ... */
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: text mode is not supported by the "
			   "imx driver\n");
		ret = FALSE;
		break;
	case FBDEVHW_VGA_PLANES:
		/* Not supported yet */
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: EGA/VGA Planes are not yet "
			   "supported by the imx driver\n");
		ret = FALSE;
		break;
	default:
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: unrecognised hardware type (%d) "
			   "encountered in imxScreenInit()\n", type);
		ret = FALSE;
		break;
	}
	if (!ret)
		return FALSE;

	if (pScrn->bitsPerPixel > 8) {
		/* Fixup RGB ordering */
		visual = pScreen->visuals + pScreen->numVisuals;
		while (--visual >= pScreen->visuals) {
			if ((visual->class | DynamicClass) == DirectColor) {
				visual->offsetRed   = pScrn->offset.red;
				visual->offsetGreen = pScrn->offset.green;
				visual->offsetBlue  = pScrn->offset.blue;
				visual->redMask     = pScrn->mask.red;
				visual->greenMask   = pScrn->mask.green;
				visual->blueMask    = pScrn->mask.blue;
			}
		}
	}

	/* must be after RGB ordering fixed */
	if (init_picture && !fbPictureInit(pScreen, NULL, 0))
		xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
			   "Render extension initialisation failed\n");

#if 0
	if (IMX_ROTATE_NONE != fPtr->rotate) {
	  xf86DrvMsg(scrnIndex, X_INFO, "using driver rotation; disabling "
			                "XRandR\n");
	  xf86DisableRandR();
	  if (pScrn->bitsPerPixel == 24)
	    xf86DrvMsg(scrnIndex, X_WARNING, "rotation might be broken at 24 "
                                             "bits per pixel\n");
	}
#endif

	xf86SetBlackWhitePixels(pScreen);

	/* INITIALIZE ACCELERATION BEFORE INIT FOR BACKING STORE AND SOFTWARE CURSOR */ 
	if (fPtr->useAccel) {

		if (!IMX_EXA_ScreenInit(scrnIndex, pScreen)) {

			fPtr->useAccel = FALSE;

		} else {

			/* If acceleration was enabled, then initialize the extension. */
			IMX_EXT_Init();
		}

	}

	/* note if acceleration is in use */
	if (fPtr->useAccel) {

		xf86DrvMsg(pScrn->scrnIndex, X_INFO, "IMX EXA acceleration setup successful\n");

	} else {

		xf86DrvMsg(pScrn->scrnIndex, X_INFO, "No acceleration in use\n");
	}
	miInitializeBackingStore(pScreen);
	xf86SetBackingStore(pScreen);

	/* software cursor */
	miDCInitialize(pScreen, xf86GetPointerScreenFuncs());

	/* colormap */
	switch ((type = fbdevHWGetType(pScrn)))
	{
	/* XXX It would be simpler to use miCreateDefColormap() in all cases. */
	case FBDEVHW_PACKED_PIXELS:
		if (!miCreateDefColormap(pScreen)) {
			xf86DrvMsg(scrnIndex, X_ERROR,
                                   "internal error: miCreateDefColormap failed "
				   "in imxScreenInit()\n");
			return FALSE;
		}
		break;
	case FBDEVHW_INTERLEAVED_PLANES:
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: interleaved planes are not yet "
			   "supported by the imx driver\n");
		return FALSE;
	case FBDEVHW_TEXT:
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: text mode is not supported by "
			   "the imx driver\n");
		return FALSE;
	case FBDEVHW_VGA_PLANES:
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: EGA/VGA planes are not yet "
			   "supported by the imx driver\n");
		return FALSE;
	default:
		xf86DrvMsg(scrnIndex, X_ERROR,
		           "internal error: unrecognised imx hardware type "
			   "(%d) encountered in imxScreenInit()\n", type);
		return FALSE;
	}
	flags = CMAP_PALETTED_TRUECOLOR;
	if(!xf86HandleColormaps(pScreen, 256, 8, fbdevHWLoadPaletteWeak(), 
				NULL, flags))
		return FALSE;

	xf86DPMSInit(pScreen, fbdevHWDPMSSetWeak(), 0);

	pScreen->SaveScreen = fbdevHWSaveScreenWeak();

	/* Wrap the current CloseScreen function */
	fPtr->CloseScreen = pScreen->CloseScreen;
	pScreen->CloseScreen = imxCloseScreen;

#if IMX_XVIDEO_ENABLE
	{
	    XF86VideoAdaptorPtr *ptr;

	    int n = xf86XVListGenericAdaptors(pScrn,&ptr);
	    if (n) {
		xf86XVScreenInit(pScreen,ptr,n);
	    }
	}
#endif

	return TRUE;
}

Bool
IMXGetPixmapProperties(
	PixmapPtr pPixmap,
	void** pPhysAddr,
	int* pPitch)
{
	/* Initialize values to be returned. */
	*pPhysAddr = NULL;
	*pPitch = 0;

	/* Is there a pixmap? */
	if (NULL == pPixmap) {
		return FALSE;
	}

	/* Access screen associated with this pixmap. */
	ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];

	/* Check if the screen associated with this pixmap has IMX driver. */
	if (0 != strcmp(IMX_DRIVER_NAME, pScrn->driverName)) {
		return FALSE;
	}

	/* Access driver specific content. */
	IMXPtr fPtr = IMXPTR(pScrn);

	/* Cannot process if not accelerating. */
	if (!fPtr->useAccel) {
		return FALSE;
	}

	/* If we get here, then query EXA portion of driver. */
	return IMX_EXA_GetPixmapProperties(pPixmap, pPhysAddr, pPitch);
}

static Bool
imxProbe(DriverPtr drv, int flags)
{
	/* For now, just bail out for PROBE_DETECT. */
	if (flags & PROBE_DETECT) {
		return FALSE;
	}

	/* Load fbdevhw support */
	if (!xf86LoadDrvSubModule(drv, "fbdevhw")) {
	    return FALSE;
	}

	/* Find all of the device sections in the config */
	GDevPtr *sections;
	int nsects;
	nsects = xf86MatchDevice(IMX_NAME, &sections);
	if (nsects <= 0) {
		return FALSE;
	}

	Bool foundScreen = FALSE;
	int i;
	for (i = 0; i < nsects; i++) {

		/* Get required device name from xorg.conf */
		char* dev = xf86FindOptionValue(sections[i]->options, OPTION_STR_FBDEV);
		if (NULL == dev) {
			xf86Msg(X_WARNING, "Option \"%s\" missing in Section with Identifier \"%s\"\n", OPTION_STR_FBDEV, sections[i]->identifier);
	 		continue;
      		}

		/* Open device */
		int fd = open(dev, O_RDWR, 0);
		if (fd <= 0) {
			xf86Msg(X_WARNING, "Could not open '%s': %s\n",
				dev, strerror(errno));
			continue;
		}

		/* Read fixed screen info from fb device. */
		struct fb_fix_screeninfo info;
		if (ioctl(fd, FBIOGET_FSCREENINFO, &info)) {
			xf86Msg(X_WARNING, "Unable to read hardware info "
					"from %s: %s\n", dev, strerror(errno));
			close(fd);
			continue;
		}
		close(fd);

		/* Make sure that this is a imx driver */
		if (IMX_TYPE_FB_UNKNOWN == imxGetFrameBufferType(&info)) {
			xf86Msg(X_WARNING, "%s is not an imx device: %s\n", dev, info.id);
			continue;
		}

		int entity = xf86ClaimFbSlot(drv, 0, sections[i], TRUE);
		ScrnInfoPtr pScrn =
			xf86ConfigFbEntity(NULL, 0, entity, NULL, NULL, NULL, NULL);

		xf86Msg(X_INFO, "Add screen %p\n", pScrn);

		/* Setup the hooks for the screen. */
		if (pScrn) {
			foundScreen = TRUE;
			
			pScrn->driverVersion = IMX_VERSION_CURRENT;
			pScrn->driverName    = IMX_DRIVER_NAME;
			pScrn->name          = IMX_NAME;
			pScrn->Probe         = imxProbe;
			pScrn->PreInit       = imxPreInit;
			pScrn->ScreenInit    = imxScreenInit;
			pScrn->SwitchMode    = fbdevHWSwitchModeWeak();
			pScrn->AdjustFrame   = fbdevHWAdjustFrameWeak();
			pScrn->EnterVT       = fbdevHWEnterVTWeak();
			pScrn->LeaveVT       = fbdevHWLeaveVTWeak();
			pScrn->ValidMode     = fbdevHWValidModeWeak();
			
			xf86DrvMsg(pScrn->scrnIndex, X_INFO,
				   "using %s\n", dev ? dev : "default device");
		}
	}

	xfree(sections);
	return foundScreen;
}

/* -------------------------------------------------------------------- */

static const OptionInfoRec *
imxAvailableOptions(int chipid, int busid)
{
	return imxOptions;
}

static void
imxIdentify(int flags)
{
	xf86PrintChipsets(IMX_NAME, "Driver for Freescale IMX processors", imxChipsets);
}

_X_EXPORT DriverRec imxDriver = {
	IMX_VERSION_CURRENT,
	IMX_DRIVER_NAME,
	imxIdentify,
	imxProbe,
	imxAvailableOptions,
	NULL,
	0,
	NULL,
};

MODULESETUPPROTO(imxSetup);

static XF86ModuleVersionInfo imxVersRec =
{
	IMX_DRIVER_NAME,
	MODULEVENDORSTRING,
	MODINFOSTRING1,
	MODINFOSTRING2,
	XORG_VERSION_CURRENT,
	IMX_VERSION_MAJOR, IMX_VERSION_MINOR, IMX_VERSION_PATCH,
	ABI_CLASS_VIDEODRV,
	ABI_VIDEODRV_VERSION,
	NULL,
	{0,0,0,0}
};

_X_EXPORT XF86ModuleData imxModuleData = { &imxVersRec, imxSetup, NULL };

pointer
imxSetup(pointer module, pointer opts, int *errmaj, int *errmin)
{
	static Bool setupDone = FALSE;

	if (!setupDone) {
		setupDone = TRUE;
		xf86AddDriver(&imxDriver, module, HaveDriverFuncs);
		return (pointer)1;
	} else {
		if (errmaj) *errmaj = LDR_ONCEONLY;
		return NULL;
	}
}

