/**************************************************************************
 * 
 * Copyright 2007 Tungsten Graphics, Inc., Cedar Park, Texas.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL TUNGSTEN GRAPHICS AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 **************************************************************************/

/* Authors:  Keith Whitwell <keith@tungstengraphics.com>
 */
#include "imports.h"
#include "macros.h"

#define CLIP_PRIVATE
#include "clip/clip_context.h"

#define CLIP_PIPE_PRIVATE
#include "clip/clip_pipe.h"



struct wide_stage {
   struct clip_pipe_stage stage;

   GLuint hw_data_offset;
   GLuint psize_offset;
   GLbitfield point_sprite_coords; /* gen tex coords for which units? */
   GLboolean point_sprite_invert;
   GLfloat half_line_width;
   GLfloat half_point_size;
   GLfloat half_point_size_min, half_point_size_max;
};



static INLINE struct wide_stage *wide_stage( struct clip_pipe_stage *stage )
{
   return (struct wide_stage *)stage;
}





static void tri( struct clip_pipe_stage *next,
		 struct vertex_header *v0,
		 struct vertex_header *v1,
		 struct vertex_header *v2 )
{
   struct prim_header tmp;

   tmp.det = 1.0;
   tmp.v[0] = v0;
   tmp.v[1] = v1;
   tmp.v[2] = v2;
   next->tri( next, &tmp );
}

static void quad( struct clip_pipe_stage *next,
		  struct vertex_header *v0,
		  struct vertex_header *v1,
		  struct vertex_header *v2,
		  struct vertex_header *v3 )
{
   /* XXX: Need to disable tri-stipple
    */
   tri( next, v0, v1, v3 );
   tri( next, v2, v0, v3 );
}

static void wide_line( struct clip_pipe_stage *stage,
		       struct prim_header *header )
{
   struct wide_stage *wide = wide_stage(stage);
   GLuint hw_data_offset = wide->hw_data_offset;
   GLfloat half_width = wide->half_line_width;

   struct vertex_header *v0 = dup_vert(stage, header->v[0], 0);
   struct vertex_header *v1 = dup_vert(stage, header->v[0], 1);
   struct vertex_header *v2 = dup_vert(stage, header->v[1], 2);
   struct vertex_header *v3 = dup_vert(stage, header->v[1], 3);

   GLfloat *pos0 = (GLfloat *)&(v0->data[hw_data_offset]);
   GLfloat *pos1 = (GLfloat *)&(v1->data[hw_data_offset]);
   GLfloat *pos2 = (GLfloat *)&(v2->data[hw_data_offset]);
   GLfloat *pos3 = (GLfloat *)&(v3->data[hw_data_offset]);
   
   GLfloat dx = FABSF(pos0[0] - pos2[0]);
   GLfloat dy = FABSF(pos0[1] - pos2[1]);
   
#if 0
   _mesa_printf("line (%f, %f) .. (%f, %f)  half_width = %f\n",
                pos0[0], pos0[1], pos2[0], pos2[1], half_width);
#endif
   /*
    * Draw wide line as a quad (two tris) by "stretching" the line along
    * X or Y.
    * XXX For AA lines, the quad corners have to be computed in a
    * more sophisticated way.
    */

   /* need to tweak coords in several ways to be conformant here */

   if (dx > dy) {
      /* x-major line */
      /* adjust Y positions */
      pos0[1] = pos0[1] - half_width - 0.125f;
      pos1[1] = pos1[1] + half_width - 0.125f;
      pos2[1] = pos2[1] - half_width - 0.125f;
      pos3[1] = pos3[1] + half_width - 0.125f;
      /* adjust X positions */
      if (pos0[0] < pos2[0]) {
         /* left to right line */
         pos0[0] -= 0.5f;
         pos1[0] -= 0.5f;
         pos2[0] -= 0.5f;
         pos3[0] -= 0.5f;
      }
      else {
         /* right to left line */
         pos0[0] += 0.5f;
         pos1[0] += 0.5f;
         pos2[0] += 0.5f;
         pos3[0] += 0.5f;
      }
   }
   else {
      /* y-major line */
      /* adjust X positions */
      pos0[0] = pos0[0] - half_width + 0.125f;
      pos1[0] = pos1[0] + half_width + 0.125f;
      pos2[0] = pos2[0] - half_width + 0.125f;
      pos3[0] = pos3[0] + half_width + 0.15f;
      /* adjust Y positions */
      if (pos0[1] < pos2[1]) {
         /* top to bottom line */
         pos0[1] -= 0.5f;
         pos1[1] -= 0.5f;
         pos2[1] -= 0.5f;
         pos3[1] -= 0.5f;
      }
      else {
         /* bottom to top line */
         pos0[1] += 0.5f;
         pos1[1] += 0.5f;
         pos2[1] += 0.5f;
         pos3[1] += 0.5f;
      }
   }

#if 0
   _mesa_printf("wide line quad (%g, %g) (%g, %g) (%g, %g) (%g, %g)\n",
          pos0[0], pos0[1],
          pos1[0], pos1[1],
          pos2[0], pos2[1],
          pos3[0], pos3[1]);
#endif

   quad( stage->next, v0, v1, v2, v3 );
}


static void make_wide_point( struct clip_pipe_stage *stage,
			     const struct vertex_header *vin,
			     struct vertex_header *v[],
			     GLfloat half_size )
{
   struct wide_stage *wide = wide_stage(stage);
   GLuint hw_data_offset = wide->hw_data_offset;
   GLfloat *pos[4];
   float left_adj, right_adj;

   v[0] = dup_vert(stage, vin, 0);
   pos[0] = (GLfloat *)&(v[0]->data[hw_data_offset]);
   
   /* Found by trial and error on psb hw vs indirect rendering.
    * Probably not totally correct:
    */
   pos[0][0] = pos[0][0] - 0.125;
   pos[0][1] = pos[0][1] - 0.125;

   v[1] = dup_vert(stage, v[0], 1);
   v[2] = dup_vert(stage, v[0], 2);
   v[3] = dup_vert(stage, v[0], 3);

   pos[1] = (GLfloat *)&(v[1]->data[hw_data_offset]);
   pos[2] = (GLfloat *)&(v[2]->data[hw_data_offset]);
   pos[3] = (GLfloat *)&(v[3]->data[hw_data_offset]);
   
//   _mesa_printf("point %f %f, %f\n", pos[0][0], pos[0][1], half_size);
   left_adj = -half_size + 0.25f;
   right_adj = half_size + 0.25f;

   pos[0][0] += left_adj;
   pos[0][1] -= half_size;

   pos[1][0] += left_adj;
   pos[1][1] += half_size;

   pos[2][0] += right_adj;
   pos[2][1] -= half_size;

   pos[3][0] += right_adj;
   pos[3][1] += half_size;
   
}

/**
 * Replace a vertex's texcoord with 'val' for the units which are 
 * specified in the 'coordUnits' mask.
 */
static INLINE void set_texcoord( struct vertex_fetch *vf,
                                 struct vertex_header *v,
                                 const GLfloat *val,
                                 GLbitfield coordUnits )
{ 
   GLubyte *dst = (GLubyte *)v;
   const struct vf_attr *a = vf->attr;
   const GLuint attr_count = vf->attr_count;
   GLuint j;

   /* XXX: precompute which attributes need to be set.
    */
   for (j = 1; j < attr_count; j++) {
      GLuint attr = a[j].attrib;
      if (attr >= VF_ATTRIB_TEX0 &&
	  attr <= VF_ATTRIB_TEX7 &&
          (coordUnits & (1 << (attr - VF_ATTRIB_TEX0))))
	 a[j].insert[4-1]( &a[j], dst + a[j].vertoffset, val );
   }
}




/* If there are lots of sprite points (and why wouldn't there be?) it
 * would probably be more sensible to change hardware setup to
 * optimize this rather than doing the whole thing in software like
 * this.
 */
static void sprite_point( struct vertex_fetch *vf,
			  struct vertex_header *v[4],
                          GLboolean invert,
                          GLbitfield coordUnits )
{
   static const GLfloat tex00[4] = { 0, 0, 0, 1 };
   static const GLfloat tex01[4] = { 0, 1, 0, 1 };
   static const GLfloat tex11[4] = { 1, 1, 0, 1 };
   static const GLfloat tex10[4] = { 1, 0, 0, 1 };

   if (invert) {
      set_texcoord( vf, v[0], tex01, coordUnits );
      set_texcoord( vf, v[1], tex00, coordUnits );
      set_texcoord( vf, v[2], tex11, coordUnits );
      set_texcoord( vf, v[3], tex10, coordUnits );
   }
   else {
      set_texcoord( vf, v[0], tex00, coordUnits );
      set_texcoord( vf, v[1], tex01, coordUnits );
      set_texcoord( vf, v[2], tex10, coordUnits );
      set_texcoord( vf, v[3], tex11, coordUnits );
   }
}



/**
 * Return the vertex's point size, divided by two.
 */
static GLfloat get_vert_psiz( struct clip_pipe_stage *stage,
                              struct prim_header *header )
{
   struct wide_stage *wide = wide_stage(stage);
   const GLubyte *vert = (GLubyte *)header->v[0];
   float sz;
   if (wide->psize_offset)
      sz = (*(GLfloat *)( vert + wide->psize_offset )) * .5;
   else
      sz = wide->half_point_size;
   sz = CLAMP(sz, wide->half_point_size_min, wide->half_point_size_max);
   return sz;
}


/* All possible wide/sprite psiz/state combinations:
 */
static void wide_point_const( struct clip_pipe_stage *stage,
			      struct prim_header *header )
{
   struct wide_stage *wide = wide_stage(stage);
   struct vertex_header *v[4];

   make_wide_point(stage, header->v[0], v, wide->half_point_size );
   quad( stage->next, v[0], v[1], v[2], v[3] );
}

static void wide_point_psiz( struct clip_pipe_stage *stage,
			     struct prim_header *header )
{
   struct vertex_header *v[4];
   make_wide_point(stage, header->v[0], v, get_vert_psiz(stage, header) );
   quad( stage->next, v[0], v[1], v[2], v[3] );
}


static void sprite_point_const( struct clip_pipe_stage *stage,
				struct prim_header *header )
{
   struct vertex_fetch *vf = stage->pipe->clip->vb.vf;
   struct wide_stage *wide = wide_stage(stage);
   struct vertex_header *v[4];

   make_wide_point(stage, header->v[0], v, wide->half_point_size );
   sprite_point(vf, v, wide->point_sprite_invert, wide->point_sprite_coords);
   quad( stage->next, v[0], v[1], v[2], v[3] );
}

static void sprite_point_psiz( struct clip_pipe_stage *stage,
			     struct prim_header *header )
{
   struct vertex_fetch *vf = stage->pipe->clip->vb.vf;
   struct wide_stage *wide = wide_stage(stage);
   struct vertex_header *v[4];

   make_wide_point(stage, header->v[0], v, get_vert_psiz(stage, header) );
   sprite_point(vf, v, wide->point_sprite_invert, wide->point_sprite_coords);
   quad( stage->next, v[0], v[1], v[2], v[3] );
}





static void wide_begin( struct clip_pipe_stage *stage )
{
   struct wide_stage *wide = wide_stage(stage);
   struct clip_context *draw = stage->pipe->clip;
   const struct vf_attr *attr
      = stage->pipe->clip->vb.vf->lookup[VF_ATTRIB_POINTSIZE];

   if (attr)
      wide->psize_offset = attr->vertoffset;
   else
      wide->psize_offset = 0;


   if (draw->vb_state.clipped_prims)
      wide->hw_data_offset = 16;
   else
      wide->hw_data_offset = 0;	

   wide->half_point_size = draw->state.point_size * .5;
   wide->half_line_width = draw->state.line_width * .5;

   wide->half_point_size_min = 0.5 * MAX2(draw->state.point_size_min, 1.0);
   wide->half_point_size_max = 0.5 * MIN2(draw->state.point_size_max, 16.0);

   if (draw->state.line_passthrough) {
      wide->stage.line = clip_passthrough_line;
   }
   else {
      wide->stage.line = wide_line;
   }

   if (draw->state.point_sprite) {
      if (draw->state.use_vertex_pointsize)
	 wide->stage.point = sprite_point_const;
      else
	 wide->stage.point = sprite_point_psiz;
      wide->point_sprite_invert = draw->state.point_sprite_invert;
      wide->point_sprite_coords = draw->state.point_sprite;
   }
   else if (draw->state.use_vertex_pointsize)
      wide->stage.point = wide_point_psiz;
   else if (draw->state.point_size != 1.0) 
      wide->stage.point = wide_point_const;
   else
      wide->stage.point = clip_passthrough_point;

   
   stage->next->begin( stage->next );
}



static void wide_end( struct clip_pipe_stage *stage )
{
   stage->next->end( stage->next );
}

struct clip_pipe_stage *clip_pipe_wide( struct clip_pipeline *pipe )
{
   struct wide_stage *wide = CALLOC_STRUCT(wide_stage);

   clip_pipe_alloc_tmps( &wide->stage, 4 );

   wide->stage.pipe = pipe;
   wide->stage.next = NULL;
   wide->stage.begin = wide_begin;
   wide->stage.point = wide_point_const;
   wide->stage.line = wide_line;
   wide->stage.tri = clip_passthrough_tri;
   wide->stage.reset_tmps = clip_pipe_reset_tmps;
   wide->stage.end = wide_end;

   return &wide->stage;
}
