#!/bin/sh
#
# OEM common libraries
#

OEMLIB=${OEMLIB:-'/usr/share/volatile'}

_apt_install()
{
    local packages="$*"

    for p in $packages; do
        python /usr/share/volatile/aptInstall --debug -p "${p}" -a "${APT_OPTIONS}"
    done
}

apt_install()
{
    local APT_OPTIONS="--yes --allow-downgrades --allow-remove-essential --allow-unauthenticated -o apt::install-recommends=true install"
    _apt_install "$*"
}

apt_install_oobe()
{
    while true
    do
        printf "%s\n" "$1" >> /var/lib/volatile-apt-installed
        shift || break
    done
}

apt_reinstall()
{
    local APT_OPTIONS="--yes --allow-downgrades --allow-remove-essential --allow-unauthenticated -o apt::install-recommends=true install --reinstall"
    _apt_install "$*"
}

apt_purge()
{
    apt-get --yes --allow-downgrades --allow-remove-essential --allow-unauthenticated purge "$*"
}

apt_reinstall_chroot_multiversion_packages()
{
    local packages="$(python /usr/share/volatile/aptInstall -o list_pkgs_name)"
    
    for pkg in $packages; do
        if dpkg-query -l "$pkg"; then
            apt_reinstall "$pkg"
        fi
    done
}

vercomp()
{
    # compare version
    # 0 $1 is equal to   $2
    # 1 $1 is newer than $2
    # 2 $2 is newer than $1

    local newer=$(printf "%s\n%s" "$1" "$2" | sort -Vr | head -1)

    if [ "$1" = "$newer" ]; then
        [ "$2" = "$newer" ] &&  \
            echo 0 || echo 1
    else
        echo 2
    fi
}

remove_boot_entry ()
{
    local efi_distributor="$*"

    for bootnum in $(efibootmgr | grep '^Boot[0-9]' | fgrep -i " $efi_distributor" | cut -b5-8); do
        efibootmgr -b "$bootnum" -B
    done
}

add_boot_entry ()
{
    local device_node="$1"
    shift
    local partition_number="$1"
    shift
    local efi_distributor="$*"

    # Delete old entries from ubuntu distributor.
    remove_boot_entry ubuntu

    # Delete old entries from the same distributor.
    remove_boot_entry "$efi_distributor"

    efibootmgr -c -d "$device_node" -p "$partition_number" -l "\\EFI\\BOOT\\BOOTX64.EFI" -L "$efi_distributor"
}

#keep some basic debug info in /var/log
run_oem_hwinfo()
{
    local EXTRALOG="/var/log/oem-hwinfo.log"

    #add oem-specific bug info.
    printf "===Ubuntu Preload Image Buildstamp===\n" > $EXTRALOG
    cat /etc/buildstamp >> $EXTRALOG
    printf "\n===BIOS Info (vendor/version/date)===\n" >> $EXTRALOG
    cat /sys/class/dmi/id/bios_vendor /sys/class/dmi/id/bios_version /sys/class/dmi/id/bios_date | tr '\n' ' ' >> $EXTRALOG
    printf "\n" >> $EXTRALOG
    printf "\n===Board Info (vendor/name/serial/version)===\n" >> $EXTRALOG
    cat /sys/class/dmi/id/board_vendor /sys/class/dmi/id/board_name /sys/class/dmi/id/board_serial /sys/class/dmi/id/board_version | tr '\n' ' ' >> $EXTRALOG
    printf "\n" >> $EXTRALOG
    printf "\n===Product Info (name/serial/uuid/version)===\n" >> $EXTRALOG
    cat /sys/class/dmi/id/product_name /sys/class/dmi/id/product_serial /sys/class/dmi/id/product_uuid /sys/class/dmi/id/product_version | tr '\n' ' ' >> $EXTRALOG
    printf "\n" >> $EXTRALOG
    printf "\n===ID===\n" >> $EXTRALOG
    lspci -x | awk 'NR==4 {print $14 $15 $16 $17}' >> $EXTRALOG
    printf "\n===Kernel version===\n" >> $EXTRALOG
    uname -a >> $EXTRALOG
    printf "\n===/proc/cmdline===\n" >> $EXTRALOG
    cat /proc/cmdline >> $EXTRALOG
    printf "\n===CPU model===\n" >> $EXTRALOG
    sed -rn 's/model name[ \t]+: (.+)/\1/p' /proc/cpuinfo | head -1  >> $EXTRALOG
    printf "\n===Volatile-task-core version===\n" >> $EXTRALOG
    dpkg-query -W -f='${Version}\n' volatile-task-core >> $EXTRALOG
    printf "\n===Volatile-task-platform version===\n" >> $EXTRALOG
    dpkg-query -W -f='${Version}\n' volatile-task-platform >> $EXTRALOG
    printf "\n===/etc/lsb-release===\n"  >> $EXTRALOG
    cat /etc/lsb-release >> $EXTRALOG
    printf "\n===hwclock===\n" >> $EXTRALOG
    hwclock >> $EXTRALOG
    printf "\n===date===\n" >> $EXTRALOG
    date >> $EXTRALOG
    printf "\n===memory===\n" >> $EXTRALOG
    free >> $EXTRALOG
    printf "\n===xrandr===\n" >> $EXTRALOG
    xrandr -q --verbose >> $EXTRALOG
    printf "\n===lspci===\n">> $EXTRALOG
    lspci -nnvv >> $EXTRALOG
    printf "\n===lsusb===\n">> $EXTRALOG
    lsusb >> $EXTRALOG

    if check_vga '8086' ; then
        #probably I+A config, add some intel info
        printf "\n=== Intel driver info===\n"  >> $EXTRALOG
        modinfo "$(lsmod |grep "^i915" | awk '{print $1}')" >> $EXTRALOG
        intel_reg_dumper >> $EXTRALOG
    fi
    
    acpidump > /var/log/acpidump.out 2>/dev/null
}

get_project_name()
{
    echo "stella"
}

get_project_series()
{
    echo "columbia"
}

get_project_kver()
{
    local kver=$(file "$(readlink -f /vmlinuz)" | awk -F- '{print $2}')

    echo "$kver"
}

get_product()
{
    local dmi_product_name="/sys/class/dmi/id/product_name"

    [ -f $dmi_product_name ] && cat $dmi_product_name
}

get_platform()
{
    local dmi_board_name="/sys/class/dmi/id/board_name"

    [ -f $dmi_board_name ] && cat $dmi_board_name
}

# import project customized library
[ -f $OEMLIB/oem-$(get_project_name).sh ] && \
    . $OEMLIB/oem-$(get_project_name).sh
