/* Copyright 2011 Canonical, Ltd. This software is licensed under the GNU
 * Lesser General Public License version 3 or later (see the file COPYING).
 */

#ifndef UTOUCH_QML_GESTUREAREA_H_
#define UTOUCH_QML_GESTUREAREA_H_

#include <QDeclarativeItem>

#include "gesture.h"

class GestureArea;
class GestureEvent;

/**
 * A property type that conveys the initial and current value of a variant.
 */

class GestureProperty : public QObject {
  Q_OBJECT
  /**
   * This property holds the initial value of the variant.
   */
  Q_PROPERTY(QVariant initial READ initial NOTIFY initialChanged)

  /**
   * This property holds the current value of the variant.
   */
  Q_PROPERTY(QVariant current READ current NOTIFY currentChanged)

 public:
  explicit GestureProperty(QObject* parent = 0) : QObject(parent) {}

  QVariant initial() const { return initial_; }
  QVariant current() const { return current_; }

 signals:
  void initialChanged();
  void currentChanged();

 private:
  void set(QVariant new_value) {
    initial_ = new_value;
    emit initialChanged();
    current_ = new_value;
    emit currentChanged();
  }
  void update(QVariant new_value) {
    if (current_ != new_value) {
      current_ = new_value;
      emit currentChanged();
    }
  }

  QVariant initial_;
  QVariant current_;

  friend class GestureArea;
  friend class UTouchPinchArea;
  friend class UTouchRotateArea;

  Q_DISABLE_COPY(GestureProperty)
};

class GestureArea : public QDeclarativeItem {
  Q_OBJECT
  /**
   * This property holds the subscription for the item.
   *
   * When the subscription is modified, the old subscription is deleted and a
   * new subscription is created.
   */
  Q_PROPERTY(Gesture* subscription READ subscription CONSTANT)

  /**
   * This property holds whether the item accepts gesture events.
   *
   * By default, this property is true.
   */
  Q_PROPERTY(bool enabled READ enabled WRITE set_enabled NOTIFY enabledChanged)

  /**
   * This property holds the position of the centroid of the touches
   * comprising the gesture.
   *
   * For TouchScreen devices, the centroid is mapped to item coordinates. For
   * all other devices, the centroid is left in device coordinates.
   */
  Q_PROPERTY(GestureProperty* centroid READ centroid CONSTANT)

  /**
   * This property holds the focus point of the gesture.
   *
   * For TouchScreen devices, the focus point is the centroid of the touches
   * when the gesture begins. For other devices, the focus point is the
   * location of the pointer when the gesture begins.
   *
   * The focus point is always mapped to item coordinates.
   */
  Q_PROPERTY(QPointF focus READ focus NOTIFY focusChanged)

 public:
  enum Primitive {
    kDrag,
    kPinch,
    kRotate,
    kTap
  };

  GestureArea() {}    /* To satisfy QML */
  GestureArea(QDeclarativeItem* parent, Primitive primitive);
  virtual ~GestureArea();

  Primitive primitive() const { return primitive_; }
  Gesture* subscription() { return &subscription_; }
  bool enabled() const { return enabled_; }
  void set_enabled(bool new_value);
  GestureProperty* centroid() { return &centroid_; }
  QPointF focus() const { return focus_; }
  WId window_id() const { return window_id_; }
  virtual bool IsGestureEventHandled(GestureEvent* event);
  virtual void HandleGestureUpdateEvent(bool end, GestureEvent* event);

 signals:
  void enabledChanged();
  void focusChanged();

 protected:
  virtual QVariant itemChange(GraphicsItemChange change, const QVariant& value);

 private:
  void BuildSubscription(Gesture* gesture);

  Primitive primitive_;
  WId window_id_;
  Gesture subscription_;
  bool enabled_;
  GestureProperty centroid_;
  QPointF focus_;

  Q_DISABLE_COPY(GestureArea)

 private slots:
  void SubscriptionChanged();
  void GeisInitialized();
};

QML_DECLARE_TYPE(GestureArea)
QML_DECLARE_TYPE(GestureProperty)

#endif  // UTOUCH_QML_GESTUREAREA_H_
