/**
 * @file geis_xcb_backend_sub_table.c
 * @brief GEIS XCB subscription table
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "geis_config.h"
#include "geis_xcb_backend_sub_table.h"

#include "geis_logging.h"
#include <stdlib.h>


typedef struct GeisXcbBackendSubTableEntry
{
  xcb_window_t     window_id;
  uint16_t         device_id;
  XcbBackendToken  token;
} GeisXcbBackendSubTableEntry;

struct GeisXcbBackendSubTable
{
  GeisSize                     size;
  GeisSize                     count;
  GeisXcbBackendSubTableEntry *data;
};

static const int table_expansion_factor = 2;


/*
 * A pseudoconstructor for the sub table entry.
 */
static inline void
_set_table_entry(GeisXcbBackendSubTableEntry *entry,
                 xcb_window_t                 window_id,
                 uint16_t                     device_id,
                 XcbBackendToken              token)
{
  entry->window_id = window_id;
  entry->device_id = device_id;
  entry->token = token;
}


static inline void
_copy_table_entry(GeisXcbBackendSubTableEntry *lhs,
                  GeisXcbBackendSubTableEntry *rhs)
{
  lhs->window_id = rhs->window_id;
  lhs->device_id = rhs->device_id;
  lhs->token = rhs->token;
}

                          
/*
 * Creates a new XCB subscription table.
 */
GeisXcbBackendSubTable
geis_xcb_backend_sub_table_new()
{
  GeisXcbBackendSubTable table = calloc(1, sizeof(struct GeisXcbBackendSubTable));
  return table;
}


/*
 * Destroys an XCB subscription table.
 */
void
geis_xcb_backend_sub_table_delete(GeisXcbBackendSubTable table)
{
  free(table->data);
  free(table);
}


/*
 * Adds a new entry to an XCB subscription table.
 */
void
geis_xcb_backend_sub_table_insert(GeisXcbBackendSubTable table,
                                  xcb_window_t           window_id,
                                  uint16_t               device_id,
                                  XcbBackendToken        token)
{
  if (table->count == table->size)
  {
    GeisSize new_size = table->size ? table->size * table_expansion_factor : 4;
    GeisXcbBackendSubTableEntry *new_data = realloc(table->data,
                                  sizeof(GeisXcbBackendSubTableEntry) * new_size);
    if (new_data)
    {
      table->size = new_size;
      table->data = new_data;
    }
  }
  _set_table_entry(&table->data[table->count], window_id, device_id, token);
  ++table->count;
}


/*
 * Removes all entries matching a given token from an XCB subscription table.
 */
void
geis_xcb_backend_sub_table_remove_token(GeisXcbBackendSubTable table,
                                        XcbBackendToken        token)
{
  GeisSize i = 0;
  while (i < table->count)
  {
    if (table->data[i].token == token)
    {
      GeisSize j = i;
      for (j = i+1; j < table->count; ++j)
      {
	_copy_table_entry(&table->data[j-1], &table->data[j]);
      }
      --table->count;
    }
    else
    {
      ++i;
    }
  }
}


/*
 * Executes a function for all entires in an XCB subscription table matching
 * given criteria.
 */
void
geis_xcb_backend_sub_table_foreach(GeisXcbBackendSubTable          table,
                                   xcb_window_t                    window_id,
                                   uint16_t                        device_id,
                                   GeisXcbBackendSubTableFunction  function,
                                   void                           *context)
{
  GeisSize i;
  for (i = 0; i < table->count; ++i)
  {
    if (table->data[i].window_id == window_id
     && table->data[i].device_id == device_id)
    {
      function(table->data[i].token, context);
    }
  }
}


