/**
 * @file geis_instance.c
 * 
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.	 See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_instance.h"

#include "geis_logging.h"
#include <stdlib.h>


/**
 * Creates a new GeisInstance for an X11 windowId.
 *
 * @param[in] An X11 windowId.
 *
 * @returns a new GeisInstance or NULL on failure.
 */
GeisInstance
geis_instance_new(uint32_t window_id)
{
  GeisInstance instance = calloc(1, sizeof(struct _GeisInstance));
  if (!instance)
  {
    geis_error("allocating GeisInstance");
    return NULL;
  }
  instance->window_id = window_id;
  return instance;
}


/**
 * Checks the GeisInstance and indicates, to the best of its ability, if it is
 * valid or not.
 *
 * @returns a nonzero (true) value if the @p instance is valid, 0 (false)
 * otherwise.
 */
int
geis_instance_is_valid(GeisInstance instance)
{
  return (instance != 0 && instance->window_id != 0);
}


GeisStatus
geis_instance_subscribe(GeisInstance         instance,
                        GeisXcb              xcb,
                        GeisInputDeviceId   *input_list,
                        const char*         *gesture_list,
                        GeisGestureFuncs    *funcs,
                        void                *cookie)
{
  GeisStatus result = GEIS_UNKNOWN_ERROR;
  instance->gesture_funcs    = funcs;
  instance->gesture_cookie   = cookie;
  if (input_list == GEIS_ALL_INPUT_DEVICES)
  {
    result = geis_xcb_subscribe(xcb, instance, 0, gesture_list);
  }
  else
  {
    GeisInputDeviceId *device_id = input_list;
    for (; *device_id; ++device_id)
    {
      GeisStatus a_result = geis_xcb_subscribe(xcb, instance,
                                               (uint16_t)*device_id,
                                               gesture_list);
      if (a_result == GEIS_STATUS_SUCCESS)
      {
	result = a_result;
      }
    }
  }
  return result;
}


uint32_t
geis_instance_get_window_id(GeisInstance instance)
{
  return instance->window_id;
}


void
geis_instance_free(GeisInstance instance)
{
  free(instance);
}
