/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens {
     
  public class RhythmboxScope : SimpleScope
  { 
    private RhythmboxCollection collection;
    private bool db_ready;
    private FileMonitor rb_xml_monitor;

    public RhythmboxScope ()
    {
      base ();

      scope = new Unity.Scope ("/com/canonical/unity/scope/rhythmbox");
      scope.search_in_global = true;
      scope.activate_uri.connect (activate);
      scope.sources.add_option ("rhythmbox", _("Rhythmbox"), null);

      base.initialize ();
      collection = new RhythmboxCollection ();
      db_ready = false;
    }

    protected override int num_results_without_search { get {return 100; } }
    protected override int num_results_global_search { get { return 20; } }
    protected override int num_results_lens_search { get { return 50; } }

    private async void prepare_rhythmbox_play_queue (string[] main_exec)
    {
      Thread.create<void> (() =>
      {
        try
        {
          Process.spawn_command_line_sync ("rhythmbox-client --pause");
          Process.spawn_command_line_sync ("rhythmbox-client --clear-queue");
          Process.spawn_sync (null, main_exec, null, SpawnFlags.SEARCH_PATH,
                              null, null);
          Process.spawn_command_line_sync ("rhythmbox-client --next");
          // make sure we're playing
          Process.spawn_command_line_sync ("rhythmbox-client --play");
        }
        catch (Error err)
        {
          warning ("%s", err.message);
        }
        Idle.add (prepare_rhythmbox_play_queue.callback);
      }, false);
      yield;
    }
    /**
     * Tells banshee to play the selected uri(s)
     */
    public Unity.ActivationResponse activate (string uri)
    {
      string[] exec = {"rhythmbox-client"};

      try 
      {
        if (Uri.parse_scheme (uri) == "album")
        {
            // this will be a bit crazy, let's do it in a separate thread
            exec += "--enqueue";
            string album_key = uri.substring (8);
            foreach (unowned string track_uri in collection.get_album_tracks (album_key))
            {
                exec += track_uri;
            }
            exec += null;
            prepare_rhythmbox_play_queue.begin (exec);
            return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
        }
        else
        {
            exec += "--play-uri";
            exec += uri;
        }

        exec += null;

        debug ("Spawning rb '%s'", string.joinv (" ", exec));
        Process.spawn_async (null, 
                 exec,
                 null, 
                 SpawnFlags.SEARCH_PATH,
                 null, 
                 null);
    
        return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
      } catch (SpawnError error) {
        warning ("Failed to launch URI %s", uri);
        return new Unity.ActivationResponse (Unity.HandledType.NOT_HANDLED);
      }
    }

    public override async void perform_search (LensSearch search,
                                               SearchType search_type, 
                                               owned List<FilterParser> filters,
                                               int max_results = -1,
                                               Cancellable? cancellable = null)
    {
      int category_override = -1;
      if (search_type == SearchType.GLOBAL)
      {
        category_override = Category.MUSIC;
        // the lens shouldn't display anything for empty search
        if (is_search_empty (search)) return;
      }

      if (!db_ready)
      {
        // parse the DB lazily
        var tdb_path = Path.build_filename (Environment.get_user_cache_dir (),
                                            "rhythmbox", "album-art",
                                            "store.tdb");
        collection.parse_metadata_file (tdb_path);

        var xml_path = Path.build_filename (Environment.get_user_data_dir (),
                                            "rhythmbox", "rhythmdb.xml");
        collection.parse_file (xml_path);
        if (rb_xml_monitor == null)
        {
          // re-parse the file if it changes
          File xml_file = File.new_for_path (xml_path);
          try
          {
            rb_xml_monitor = xml_file.monitor (FileMonitorFlags.NONE);
            rb_xml_monitor.changed.connect (() => { db_ready = false; });
          }
          catch (Error err)
          {
            warning ("%s", err.message);
          }
        }

        db_ready = true;
      }

      collection.search (search, search_type, filters,
                         max_results, category_override);
    }
  }
}
