# ubuntuone.storageprotocol.tests.test_bytesproducer -
#     BytesMessageProducer tests
#
# Author: Facundo Batista <facundo@canonical.com>
#
# Copyright (C) 2009 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License version 3,
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for directory content serialization/unserialization."""

from __future__ import with_statement

from unittest import TestCase
from cStringIO import StringIO

from twisted.internet import defer, reactor
from twisted.trial.unittest import TestCase as TwistedTestCase

from ubuntuone.storageprotocol import client, request, protocol_pb2

# puts a lower limit, easier for testing
request.MAX_PAYLOAD_SIZE = 3

class FakeRequest(object):
    """Fake Request class that is handy for tests."""

    def __init__(self):
        self.messages = []
        self.cancelled = False

    def sendMessage(self, message):
        """Store the message in own list."""
        name = protocol_pb2.Message.DESCRIPTOR \
                  .enum_types_by_name['MessageType'] \
                  .values_by_number[message.type].name
        self.messages.append(name)


class TestProducingState(TestCase):
    """Test for filename validation and normalization."""

    def setUp(self):
        fh = StringIO()
        fh.write("abcde")
        fh.seek(0)
        req = FakeRequest()
        self.bmp = client.BytesMessageProducer(req, fh, 0)

    def test_start(self):
        """It starts not producing anything."""
        self.assertFalse(self.bmp.producing)

    def test_resume_from_init(self):
        """Produce after a Resume coming from init."""
        self.bmp.resumeProducing()
        self.assertTrue(self.bmp.producing)

    def test_resume_from_pause(self):
        """Produce after a Resume coming from pause."""
        self.bmp.pauseProducing()
        self.bmp.resumeProducing()
        self.assertTrue(self.bmp.producing)

    def test_resume_from_stop(self):
        """Produce after a Resume coming from stop."""
        self.bmp.stopProducing()
        self.bmp.resumeProducing()
        self.assertTrue(self.bmp.producing)

    def test_resume_and_pause(self):
        """Pause after a Resume."""
        self.bmp.resumeProducing()
        self.bmp.pauseProducing()
        self.assertFalse(self.bmp.producing)

    def test_resume_and_stop(self):
        """Stop after a Resume."""
        self.bmp.resumeProducing()
        self.bmp.stopProducing()
        self.assertFalse(self.bmp.producing)


class TestGenerateData(TwistedTestCase):
    """Test for data generation."""

    timeout = 1

    def setUp(self):
        fh = StringIO()
        fh.write("abcde")
        fh.seek(0)
        self.req = FakeRequest()
        self.bmp = client.BytesMessageProducer(self.req, fh, 0)

    def test_start(self):
        """It starts not producing anything."""
        self.assertEqual(self.req.messages, [])

    def test_generate(self):
        """Generate all data after a resume."""
        d = defer.Deferred()
        self.bmp.resumeProducing()

        def check():
            """Check that the messages are ok."""
            self.assertEqual(self.req.messages, ["BYTES", "BYTES", "EOF"])
            d.callback(True)

        reactor.callLater(.1, check)
        return d

    def test_no_double_EOF(self):
        """Don't send EOF after finished."""
        d = defer.Deferred()
        self.bmp.resumeProducing()

        def resume_again():
            """Resume again, after it finished."""
            # we should have received everything, clean that and re-send resume
            self.assertEqual(self.req.messages, ["BYTES", "BYTES", "EOF"])
            self.req.messages[:] = []

            self.bmp.resumeProducing()
            reactor.callLater(.1, check)

        def check():
            """Check that the messages are ok."""
            self.assertEqual(self.req.messages, [])
            d.callback(True)

        reactor.callLater(.1, resume_again)
        return d


