#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for Sync."""

from __future__ import with_statement

import logging
import os
import shutil
import unittest

import dbus
from dbus.mainloop.glib import DBusGMainLoop

from contrib.testing.testcase import (
    FakeVolumeManager,
    BaseTwistedTestCase,
    MementoHandler, DummyClass
)

from ubuntuone.syncdaemon.dbus_interface import DBusInterface
from ubuntuone.syncdaemon.filesystem_manager import FileSystemManager
from ubuntuone.syncdaemon.main import Main
from ubuntuone.syncdaemon.sync import FSKey, Sync, SyncStateMachineRunner
from ubuntuone.syncdaemon.volume_manager import Share
from ubuntuone.syncdaemon.event_queue import EventQueue

DBusInterface.test = True

class FSKeyTestCase(BaseTwistedTestCase):
    """ Base test case for FSKey """

    def setUp(self):
        """ Setup the test """
        unittest.TestCase.setUp(self)
        self.test_dir = self.mktemp("test_dir")
        self.shares_dir = os.path.join(self.test_dir, 'shares')
        os.makedirs(self.shares_dir)
        self.root_dir = os.path.join(self.test_dir, 'root')
        os.makedirs(self.root_dir)
        self.fsmdir = os.path.join(self.test_dir, "fsmdir")
        self.partials_dir = os.path.join(self.test_dir, "partials")
        os.makedirs(self.partials_dir)
        self.fsm = FileSystemManager(self.fsmdir, self.partials_dir,
                                     FakeVolumeManager(self.root_dir))
        self.eq = EventQueue(self.fsm)
        self.fsm.register_eq(self.eq)
        self.share = self.create_share('share', 'share_name',
                                            self.fsm, self.shares_dir)
        self.share_path = self.share.path

    def tearDown(self):
        """ Clean up the test dir"""
        self.eq.shutdown()
        shutil.rmtree(self.test_dir)

    @staticmethod
    def create_share(share_id, share_name, fsm, shares_dir,
                     access_level='Modify'):
        """ creates a share """
        share_path = os.path.join(shares_dir, share_name)
        os.makedirs(share_path)
        share = Share(path=share_path, volume_id=share_id,
                      access_level=access_level)
        fsm.vm.add_share(share)
        return share


class FSKeyTests(FSKeyTestCase):
    """Test FSKey methods."""

    def test_get_mdid(self):
        """simple tests for get_mdid"""
        path = os.path.join(self.share.path, 'path')
        mdid = self.fsm.create(path, "share", node_id='uuid1')
        key = FSKey(self.fsm, path=path)
        self.assertEquals(mdid, key.get_mdid())
        key = FSKey(self.fsm, share_id='share', node_id='uuid1')
        self.assertEquals(mdid, key.get_mdid())
        key = FSKey(self.fsm, mdid=mdid)
        self.assertEquals(mdid, key.get_mdid())
        # with bad keys
        key = FSKey(self.fsm, share='share', node_id='uuid1')
        self.assertRaises(KeyError, key.get_mdid)
        # 1 valid and 1 invalid key
        key = FSKey(self.fsm, share='share', mdid=mdid)
        self.assertRaises(KeyError, key.get_mdid)

    def test_set(self):
        """test that changes to the key are keeped in _changes until sync"""
        path = os.path.join(self.share.path, 'path')
        self.fsm.create(path, "share", node_id='uuid1')
        key = FSKey(self.fsm, path=path)
        key.set(local_hash='a_hash')
        self.assertEquals('a_hash', key._changes['local_hash'])
        key.set(server_hash='a_hash_1')
        self.assertEquals('a_hash_1', key._changes['server_hash'])
        key.sync()
        self.assertEquals({}, key._changes)

    def test_sync(self):
        """test sync of the changes to the fsm"""
        path = os.path.join(self.share.path, 'path')
        mdid = self.fsm.create(path, "share", node_id='uuid1')
        key = FSKey(self.fsm, path=path)
        # change it
        key.set(local_hash='local_hash', server_hash='server_hash')
        # sync!
        key.sync()
        # get the mdobj and check the values
        mdobj = self.fsm.get_by_mdid(mdid)
        self.assertEquals('server_hash', mdobj.server_hash)
        self.assertEquals('local_hash', mdobj.local_hash)
        self.assertEquals('share', mdobj.share_id)
        self.assertEquals('uuid1', mdobj.node_id)

    def test_mdid_reset(self):
        """Test for mdid reset after a deletion"""
        # create a node
        path = os.path.join(self.share.path, 'path')
        self.fsm.create(path, "share", node_id='uuid1')
        key = FSKey(self.fsm, path=path)
        # fake a conflict and delete the metadata
        key.move_to_conflict()
        key.delete_metadata()
        # create the node again
        self.fsm.create(path, "share", node_id='uuid1')
        key.set(server_hash="")
        key.set(local_hash="")
        # fail no more!
        key.sync()
        # now check the delete_file method
        # fake a conflict and delete the metadata
        key.delete_file()
        # create the node again
        self.fsm.create(path, "share", node_id='uuid1')
        key.set(server_hash="")
        key.set(local_hash="")
        # fail no more!
        key.sync()

    def test_changes_reset(self):
        """Test for _changes reset after a deletion"""
        # create a node
        path = os.path.join(self.share.path, 'path')
        self.fsm.create(path, "share", node_id='uuid1')
        key = FSKey(self.fsm, path=path)
        # set a value and delete the metadata
        key.set(server_hash="server_hash")
        key.delete_metadata()
        # create the node again
        self.fsm.create(path, "share", node_id='uuid1')
        key.sync()
        self.assertNotEquals('server_hash',
                             self.fsm.get_by_path(path).server_hash)
        # now check the delete_file method
        key = FSKey(self.fsm, path=path)
        key.set(server_hash="server_hash1")
        # fake a conflict and delete the metadata
        key.delete_file()
        # create the node again
        self.fsm.create(path, "share", node_id='uuid1')
        key.sync()
        self.assertNotEquals('server_hash1',
                             self.fsm.get_by_path(path).server_hash)


class TestSync(BaseTwistedTestCase):
    """Test for Sync."""

    def setUp(self):
        """Init."""
        BaseTwistedTestCase.setUp(self)
        self.root = self.mktemp('root')
        self.shares = self.mktemp('shares')
        self.data = self.mktemp('data')
        self.partials_dir = self.mktemp('partials_dir')
        self.handler = MementoHandler()
        self.handler.setLevel(logging.ERROR)
        self.main = Main(root_dir=self.root,
                         shares_dir=self.shares,
                         data_dir=self.data,
                         partials_dir=self.partials_dir,
                         host='localhost', port=0,
                         dns_srv=False, ssl=False,
                         disable_ssl_verify=True,
                         realm='fake.realm',
                         mark_interval=60,
                         handshake_timeout=2,
                         glib_loop=DBusGMainLoop(set_as_default=True))
        logging.getLogger('ubuntuone.SyncDaemon').addHandler(self.handler)
        dbus.service.BusName.__del__ = lambda _: None

    def tearDown(self):
        """Clean up."""
        self.main.shutdown()
        shutil.rmtree(self.root)
        shutil.rmtree(self.shares)
        shutil.rmtree(self.data)
        for record in self.handler.records:
            exc_info = getattr(record, 'exc_info', None)
            if exc_info is not None:
                raise exc_info[0], exc_info[1], exc_info[2]
        BaseTwistedTestCase.tearDown(self)

    def test_deleting_open_files_is_no_cause_for_despair(self):
        """test_deleting_open_files_is_no_cause_for_despair."""
        def cb(_):
            d0 = self.main.wait_for('HQ_HASH_NEW')
            f = file(self.root + '/a_file', 'w')
            print >> f, 'hola'
            os.unlink(self.root + '/a_file')
            f.close()
            print >> file(self.root + '/b_file', 'w'), 'chau'
            return d0
        d = self.main.wait_for('SYS_LOCAL_RESCAN_DONE')
        self.main.start()
        d.addCallback(cb)
        return d

    def test_stomp_deleted_file_is_no_cause_for_despair_either(self):
        """test_stomp_deleted_file_is_no_cause_for_despair_either."""
        def cb(_):
            d0 = self.main.wait_for('HQ_HASH_NEW')
            f = file(self.root + '/a_file', 'w')
            print >> f, 'hola'
            os.unlink(self.root + '/a_file')
            f.close()
            print >> file(self.root + '/b_file', 'w'), 'chau'
            return d0
        d = self.main.wait_for('SYS_LOCAL_RESCAN_DONE')
        self.main.start()
        d.addCallback(cb)
        return d

    def test_handle_AQ_DOWNLOAD_DOES_NOT_EXIST(self):
        """handle_AQ_DOWNLOAD_DOES_NOT_EXIST."""
        sync = Sync(main=self.main)
        self.called = False

        def faked_nothing(ssmr, event, params, *args):
            """Wrap SSMR.nothing to test."""
            self.called = True

        SyncStateMachineRunner.nothing = faked_nothing
        kwargs = dict(share_id='share_id', node_id='node_id')
        sync.handle_AQ_DOWNLOAD_DOES_NOT_EXIST(**kwargs)
        self.assertTrue(self.called, 'nothing was called')


class SyncStateMachineRunnerTestCase(BaseTwistedTestCase):
    """Tests for the SyncStateMachineRunner."""

    def setUp(self):
        """Init."""
        BaseTwistedTestCase.setUp(self)
        self.ssmr = SyncStateMachineRunner(fsm=None, main=None,
                                           key=DummyClass(), logger=None)

    def tearDown(self):
        """Clean up."""
        self.ssmr = None
        BaseTwistedTestCase.tearDown(self)

    def test_delete_file(self):
        """delete_file can be called with or without the server hash."""
        self.ssmr.delete_file(event='AQ_DOWNLOAD_ERROR', params=None)

        self.ssmr.delete_file(event='AQ_DOWNLOAD_ERROR', params=None,
                              server_hash='')
