#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# «recovery_backend» - Backend Manager.  Handles backend service calls
#
# Copyright (C) 2009, urecovery Inc.
#           (C) 2008 Canonical Ltd.
#
# Author:
#  - Mario Limonciello <Mario_Limonciello@urecovery.com>
#
# This is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this application; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##################################################################################

import logging, os, os.path, signal, sys, re

import gobject
import dbus
import dbus.service
import dbus.mainloop.glib
import atexit
import tempfile
import subprocess
import tarfile
import shutil
import datetime
import distutils.dir_util
import stat
import zipfile
from hashlib import md5

from urecovery.common import (DOMAIN, LOCALEDIR, UP_FILENAMES,
                                  walk_cleanup, create_new_uuid, white_tree,
                                  fetch_output,
                                  DBUS_BUS_NAME, DBUS_INTERFACE_NAME,
                                  RestoreFailed, CreateFailed,
                                  PermissionDeniedByPolicy)
from urecovery.thread import ProgressByPulse, ProgressBySize

#Translation support
from gettext import gettext as _
from gettext import bindtextdomain, textdomain

class Backend(dbus.service.Object):
    '''Backend manager.

    This encapsulates all services calls of the backend. It
    is implemented as a dbus.service.Object, so that it can be called through
    D-BUS as well (on the /RecoveryMedia object path).
    '''

    #
    # D-BUS control API
    #

    def __init__(self):
        dbus.service.Object.__init__(self)

        #initialize variables that will be used during create and run
        self.bus = None
        self.main_loop = None
        self._timeout = False
        self.dbus_name = None
        
        # cached D-BUS interfaces for _check_polkit_privilege()
        self.dbus_info = None
        self.polkit = None
        self.progress_thread = None
        self.enforce_polkit = True

        #Enable translation for strings used
        bindtextdomain(DOMAIN, LOCALEDIR)
        textdomain(DOMAIN)

    def run_dbus_service(self, timeout=None, send_usr1=False):
        '''Run D-BUS server.

        If no timeout is given, the server will run forever, otherwise it will
        return after the specified number of seconds.

        If send_usr1 is True, this will send a SIGUSR1 to the parent process
        once the server is ready to take requests.
        '''
        dbus.service.Object.__init__(self, self.bus, '/RecoveryMedia')
        self.main_loop = gobject.MainLoop()
        self._timeout = False
        if timeout:
            def _quit():
                """This function is ran at the end of timeout"""
                self.main_loop.quit()
                return True
            gobject.timeout_add(timeout * 1000, _quit)

        # send parent process a signal that we are ready now
        if send_usr1:
            os.kill(os.getppid(), signal.SIGUSR1)

        # run until we time out
        while not self._timeout:
            if timeout:
                self._timeout = True
            self.main_loop.run()

    @classmethod
    def create_dbus_server(cls, session_bus=False):
        '''Return a D-BUS server backend instance.

        Normally this connects to the system bus. Set session_bus to True to
        connect to the session bus (for testing).

        '''
        backend = Backend()
        dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
        if session_bus:
            backend.bus = dbus.SessionBus()
            backend.enforce_polkit = False
        else:
            backend.bus = dbus.SystemBus()
        try:
            backend.dbus_name = dbus.service.BusName(DBUS_BUS_NAME, backend.bus)
        except dbus.exceptions.DBusException, msg:
            logging.error("Exception when spawning dbus service")
            logging.error(msg)
            return None
        return backend

    #
    # Internal methods
    #

    def _reset_timeout(self):
        '''Reset the D-BUS server timeout.'''

        self._timeout = False

    def _check_polkit_privilege(self, sender, conn, privilege):
        '''Verify that sender has a given PolicyKit privilege.

        sender is the sender's (private) D-BUS name, such as ":1:42"
        (sender_keyword in @dbus.service.methods). conn is
        the dbus.Connection object (connection_keyword in
        @dbus.service.methods). privilege is the PolicyKit privilege string.

        This method returns if the caller is privileged, and otherwise throws a
        PermissionDeniedByPolicy exception.
        '''
        if sender is None and conn is None:
            # called locally, not through D-BUS
            return
        if not self.enforce_polkit:
            # that happens for testing purposes when running on the session
            # bus, and it does not make sense to restrict operations here
            return

        # get peer PID
        if self.dbus_info is None:
            self.dbus_info = dbus.Interface(conn.get_object('org.freedesktop.DBus',
                '/org/freedesktop/DBus/Bus', False), 'org.freedesktop.DBus')
        pid = self.dbus_info.GetConnectionUnixProcessID(sender)

        # query PolicyKit
        if self.polkit is None:
            self.polkit = dbus.Interface(dbus.SystemBus().get_object(
                'org.freedesktop.PolicyKit1', '/org/freedesktop/PolicyKit1/Authority', False),
                'org.freedesktop.PolicyKit1.Authority')
        try:
            # we don't need is_challenge return here, since we call with AllowUserInteraction
            (is_auth, unused, details) = self.polkit.CheckAuthorization(
                    ('unix-process', {'pid': dbus.UInt32(pid, variant_level=1),
                        'start-time': dbus.UInt64(0, variant_level=1)}),
                    privilege, {'': ''}, dbus.UInt32(1), '', timeout=600)
        except dbus.DBusException, msg:
            if msg.get_dbus_name() == \
                                    'org.freedesktop.DBus.Error.ServiceUnknown':
                # polkitd timed out, connect again
                self.polkit = None
                return self._check_polkit_privilege(sender, conn, privilege)
            else:
                raise

        if not is_auth:
            logging.debug('_check_polkit_privilege: sender %s on connection %s pid %i is not authorized for %s: %s',
                    sender, conn, pid, privilege, str(details))
            raise PermissionDeniedByPolicy(privilege)

    #
    # Internal API for calling from Handlers (not exported through D-BUS)
    #

    def request_mount(self, recovery, sender=None, conn=None):
        '''Attempts to mount the recovery partition

           If successful, return mntdir.
           If we find that it's already mounted elsewhere, return that mount
           If unsuccessful, return an empty string
        '''
        #Work around issues sending a UTF-8 directory over dbus
        recovery = recovery.encode('utf8')

        #In this is just a directory
        if os.path.isdir(recovery):
            return recovery

        #check for an existing mount
        command = subprocess.Popen(['mount'], stdout=subprocess.PIPE)
        output = command.communicate()[0].split('\n')
        for line in output:
            processed_line = line.split()
            if len(processed_line) > 0 and processed_line[0] == recovery:
                return processed_line[2]

        #if not already, mounted, produce a mount point
        mntdir = tempfile.mkdtemp()
        mnt_args = ['mount', '-r', recovery, mntdir]
        if ".iso" in recovery:
            mnt_args.insert(1, 'loop')
            mnt_args.insert(1, '-o')
        else:
            self._check_polkit_privilege(sender, conn,
                                                'com.urecovery.recoverymedia.create')
        command = subprocess.Popen(mnt_args,
                                 stdout=subprocess.PIPE,
                                 stderr=subprocess.PIPE)
        output = command.communicate()
        ret = command.wait()
        if ret is not 0:
            os.rmdir(mntdir)
            if ret == 32:
                try:
                    mntdir = output[1].strip('\n').split('on')[1].strip(' ')
                except IndexError:
                    mntdir = ''
                    logging.warning("IndexError when operating on output string")
            else:
                mntdir = ''
                logging.warning("Unable to mount recovery partition")
                logging.warning(output)
        else:
            atexit.register(self._unmount_drive, mntdir)
        return mntdir

    def _unmount_drive(self, mnt):
        """Unmounts something mounted at a particular mount point"""
        if os.path.exists(mnt):
            ret = subprocess.call(['umount', mnt])
            if ret is not 0:
                print >> sys.stderr, "Error unmounting %s" % mnt
            try:
                os.rmdir(mnt)
            except OSError, msg:
                print >> sys.stderr, "Error cleaning up: %s" % str(msg)

    def start_sizable_progress_thread(self, input_str, mnt, w_size):
        """Initializes the extra progress thread, or resets it
           if it already exists'"""
        self.progress_thread = ProgressBySize(input_str, mnt, w_size)
        self.progress_thread.progress = self.report_progress
        self.progress_thread.start()

    def stop_progress_thread(self):
        """Stops the extra thread for reporting progress"""
        self.progress_thread.join()

    def start_pulsable_progress_thread(self, input_str):
        """Starts the extra thread for pulsing progress in the UI"""
        self.progress_thread = ProgressByPulse(input_str)
        self.progress_thread.progress = self.report_progress
        self.progress_thread.start()
    #
    # Client API (through D-BUS)
    #
    @dbus.service.method(DBUS_INTERFACE_NAME,
        in_signature = '', out_signature = '', sender_keyword = 'sender',
        connection_keyword = 'conn')
    def request_exit(self, sender=None, conn=None):
        """Closes the backend and cleans up"""
        self._check_polkit_privilege(sender, conn, 'com.urecovery.recoverymedia.request_exit')
        self._timeout = True
        self.main_loop.quit()

    @dbus.service.method(DBUS_INTERFACE_NAME,
        in_signature = 'sss', out_signature = '', sender_keyword = 'sender',
        connection_keyword = 'conn')
    def create_ubuntu(self, recovery, version, iso, sender=None, conn=None):
        """Creates Ubuntu compatible recovery media"""
        self._reset_timeout()
        self._check_polkit_privilege(sender, conn,
                                                'com.urecovery.recoverymedia.create')

        #create temporary workspace
        tmpdir = tempfile.mkdtemp()
        atexit.register(walk_cleanup, tmpdir)

        #mount the recovery partition
        mntdir = self.request_mount(recovery, sender, conn)

        if not os.path.exists(os.path.join(mntdir, '.disk', 'info')):
            print >> sys.stderr, \
                "recovery partition is missing critical ubuntu files."
            raise CreateFailed("Recovery partition is missing critical Ubuntu files.")

        #Arg list
        genisoargs = ['genisoimage',
            '-o', iso,
            '-input-charset', 'utf-8',
            '-no-emul-boot',
            '-boot-load-size', '4',
            '-boot-info-table',
            '-pad',
            '-r',
            '-J',
            '-joliet-long',
            '-N',
            '-hide-joliet-trans-tbl',
            '-cache-inodes',
            '-l',
            '-publisher', 'urecovery Inc.',
            '-V', 'urecovery Ubuntu Reinstallation Media',
            '-m', '*.exe',
            '-m', '*.sys',
            '-m', 'syslinux',
            '-m', 'syslinux.cfg',
            '-m', os.path.join(mntdir, 'isolinux'),
            '-m', os.path.join(mntdir, 'bto_version')]

        #Renerate UUID
        os.mkdir(os.path.join(tmpdir, '.disk'))
        os.mkdir(os.path.join(tmpdir, 'casper'))
        self.start_pulsable_progress_thread(_('Regenerating UUID / Rebuilding initramfs'))
        (old_initrd,
         old_uuid) = create_new_uuid(os.path.join(mntdir, 'casper'),
                        os.path.join(mntdir, '.disk'),
                        os.path.join(tmpdir, 'casper'),
                        os.path.join(tmpdir, '.disk'))
        self.stop_progress_thread()
        genisoargs.append('-m')
        genisoargs.append(os.path.join('.disk', old_uuid))
        genisoargs.append('-m')
        genisoargs.append(os.path.join('casper', old_initrd))

        #if we're grub based, generate a grub image
        grub_root = os.path.join(mntdir,'boot', 'grub', 'i386-pc')
        if os.path.exists(grub_root):
            os.makedirs(os.path.join(tmpdir, 'boot', 'grub'))
            if os.path.exists(os.path.join(mntdir, 'boot', 'grub', 'eltorito.img')):
                genisoargs.append('-m')
                genisoargs.append(os.path.join(mntdir, 'boot', 'grub', 'eltorito.img'))
                shutil.copy(os.path.join(mntdir, 'boot', 'grub', 'eltorito.img'),
                            os.path.join(tmpdir, 'boot', 'grub', 'eltorito.img'))
            else:
                if not os.path.exists(os.path.join(grub_root, 'core.img')):
                    raise CreateFailed("The target requested GRUB support, but core.img is missing.")
                if not os.path.exists(os.path.join(grub_root, 'cdboot.img')):
                    raise CreateFailed("The target requested GRUB support, but cdboot.img is missing.")

                self.start_pulsable_progress_thread(_('Building GRUB core image'))
                with open(os.path.join(tmpdir, 'boot', 'grub', 'eltorito.img'), 'w') as wfd:
                    for fname in ('cdboot.img', 'core.img'):
                        with open(os.path.join(grub_root, fname), 'r') as rfd:
                            wfd.write(rfd.read())                    
                self.stop_progress_thread()
            genisoargs.append('-m')
            genisoargs.append(os.path.join(mntdir,'boot/boot.catalog'))
            genisoargs.append('-b')
            genisoargs.append('boot/grub/eltorito.img')
            genisoargs.append('-c')
            genisoargs.append('boot/boot.catalog')

        #isolinux based
        else:
            genisoargs.append('-b')
            genisoargs.append('isolinux/isolinux.bin')
            genisoargs.append('-c')
            genisoargs.append('isolinux/boot.catalog')

            #if we have ran this from a USB key, we might have syslinux which will
            #break our build
            if os.path.exists(os.path.join(mntdir, 'syslinux')):
                shutil.copytree(os.path.join(mntdir, 'syslinux'), os.path.join(tmpdir, 'isolinux'))
                if os.path.exists(os.path.join(tmpdir, 'isolinux', 'syslinux.cfg')):
                    shutil.move(os.path.join(tmpdir, 'isolinux', 'syslinux.cfg'), os.path.join(tmpdir, 'isolinux', 'isolinux.cfg'))
            else:
                #Copy boot section for ISO to somewhere writable
                shutil.copytree(os.path.join(mntdir, 'isolinux'), os.path.join(tmpdir, 'isolinux'))

        #Make the image EFI compatible if necessary
        if os.path.exists(os.path.join(mntdir, 'boot', 'grub', 'efi.img')):
            efi_genisoimage = subprocess.Popen(['genisoimage','-help'],
                                                stdout=subprocess.PIPE,
                                                stderr=subprocess.PIPE)
            results = efi_genisoimage.communicate()[1]
            if 'efi' in results:
                genisoargs.append('-eltorito-alt-boot')
                genisoargs.append('-efi-boot')
                genisoargs.append('boot/grub/efi.img')
                genisoargs.append('-no-emul-boot')
            else:
                import apt.cache
                cache = apt.cache.Cache()
                version = cache['genisoimage'].installed.version
                del cache
                raise CreateFailed("The target image requested EFI support, but genisoimage %s doesn't support EFI.  \
You will need to create this image on a system with a newer genisoimage." % version)


        #Directories to install
        genisoargs.append(tmpdir + '/')
        genisoargs.append(mntdir + '/')

        #ISO Creation
        seg1 = subprocess.Popen(genisoargs,
                              stdout=subprocess.PIPE,
                              stderr=subprocess.PIPE)
        retval = seg1.poll()
        output = ""
        while (retval is None):
            stdout = seg1.stderr.readline()
            if stdout != "":
                output = stdout
            if output:
                progress = output.split()[0]
                if (progress[-1:] == '%'):
                    self.report_progress(_('Building ISO'), progress[:-1])
            retval = seg1.poll()
        if retval is not 0:
            print >> sys.stderr, genisoargs
            print >> sys.stderr, output.strip()
            print >> sys.stderr, seg1.stderr.readlines()
            print >> sys.stderr, seg1.stdout.readlines()
            print >> sys.stderr, \
                "genisoimage exited with a nonstandard return value."
            raise CreateFailed("ISO Building exited unexpectedly:\n%s" %
                               output.strip())

    @dbus.service.signal(DBUS_INTERFACE_NAME)
    def report_progress(self, progress_str, percent):
        '''Report ISO build progress to UI.
        '''
        return True
