/***************************************************************************
                             th-parsetter.c
                             --------------
    begin                : Sat Mar 26 2007
    copyright            : (C) 2007 by Tim-Philipp Müller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/* Munges the pixel-aspect-ratio field in the caps */

#include "th-parsetter.h"

enum {
  PROP_PAR = 1
};

#define DEFAULT_PAR_NUM    0
#define DEFAULT_PAR_DENOM  1

GST_BOILERPLATE (ThParSetter, th_par_setter, GstBaseTransform, GST_TYPE_BASE_TRANSFORM);

static GstFlowReturn th_par_setter_transform_ip (GstBaseTransform * bt,
    GstBuffer * buf);
static GstFlowReturn th_par_setter_prepare_output_buffer (GstBaseTransform * bt,
    GstBuffer * inbuf, gint size, GstCaps * caps, GstBuffer ** p_outbuf);
static GstCaps * th_par_setter_transform_caps (GstBaseTransform * trans,
    GstPadDirection dir, GstCaps * caps);
static gboolean th_par_setter_transform_size (GstBaseTransform * trans,
    GstPadDirection dir, GstCaps * caps, guint size, GstCaps * othercaps,
    guint * othersize);

static GstStaticPadTemplate parsetter_sink_factory =
    GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-rgb; video/x-raw-yuv"));

static GstStaticPadTemplate parsetter_src_factory =
    GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-rgb; video/x-raw-yuv"));

static const GstElementDetails parsetter_details =
GST_ELEMENT_DETAILS ("Pixel-Aspect Ratio Setter",
    "Filter/Effect/Video",
    "Force a particular pixel aspect ratio on a video stream",
    "Tim-Philipp Müller <tim centricular net>");

static void
th_par_setter_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details (element_class, &parsetter_details);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&parsetter_sink_factory));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&parsetter_src_factory));
}

static void
th_par_setter_set_property (GObject * obj, guint prop_id, const GValue * val,
    GParamSpec * pspec)
{
  ThParSetter *setter = TH_PAR_SETTER (obj);

  switch (prop_id) {
    case PROP_PAR: {
      gint num = gst_value_get_fraction_numerator (val);
      gint denom = gst_value_get_fraction_denominator (val);
      if (denom == 0)
        g_warning ("Trying to set invalid pixel-aspect-ratio %d/%d", num, denom);
      else
        gst_value_set_fraction (&setter->par, num, denom);
      break;
    }
    default:
      break;
  }
}

static void
th_par_setter_class_init (ThParSetterClass *klass)
{
  GstBaseTransformClass *basetrans_class = GST_BASE_TRANSFORM_CLASS (klass);
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = th_par_setter_set_property;

  g_object_class_install_property (gobject_class, PROP_PAR,
      gst_param_spec_fraction ("pixel-aspect-ratio", "Pixel Aspect Ratio",
          "Pixel Aspect Ratio to force on the video stream (0/1 = no change)",
          0, 1, G_MAXINT, 1, DEFAULT_PAR_NUM, DEFAULT_PAR_DENOM,
          G_PARAM_WRITABLE));

  basetrans_class->prepare_output_buffer =
      GST_DEBUG_FUNCPTR (th_par_setter_prepare_output_buffer);
  basetrans_class->transform_caps =
      GST_DEBUG_FUNCPTR (th_par_setter_transform_caps);
  basetrans_class->transform_ip =
      GST_DEBUG_FUNCPTR (th_par_setter_transform_ip);
  basetrans_class->transform_size =
      GST_DEBUG_FUNCPTR (th_par_setter_transform_size);
}

static void
th_par_setter_init (ThParSetter *setter, ThParSetterClass *klass)
{
  /* kids: technically, we should have a corresponding g_value_unset() in a
   * finalize function, but we know we can skip that because there aren't any
   * deep allocs in the value structure in this case */
  g_value_init (&setter->par, GST_TYPE_FRACTION);
  gst_value_set_fraction (&setter->par, DEFAULT_PAR_NUM, DEFAULT_PAR_DENOM);

  gst_base_transform_set_in_place (GST_BASE_TRANSFORM (setter), TRUE);
}

static GstFlowReturn
th_par_setter_prepare_output_buffer (GstBaseTransform * basetrans,
    GstBuffer * inbuf, gint size, GstCaps * caps, GstBuffer ** p_outbuf)
{
  ThParSetter *setter = TH_PAR_SETTER (basetrans);

  if (gst_value_get_fraction_numerator (&setter->par) == 0) {
    *p_outbuf = gst_buffer_ref (inbuf);
    /* don't touch caps */
  } else {
    /* ref to make sure make_metadata_writable creates a subbuffer */
    gst_buffer_ref (inbuf);
    *p_outbuf = gst_buffer_make_metadata_writable (inbuf);
    gst_buffer_set_caps (*p_outbuf,
        GST_PAD_CAPS (GST_BASE_TRANSFORM_SRC_PAD (basetrans)));
  }
  return GST_FLOW_OK;
}

static gboolean
th_par_setter_transform_size (GstBaseTransform * trans, GstPadDirection dir,
    GstCaps * caps, guint size, GstCaps * othercaps, guint * othersize)
{
  *othersize = size;
  return TRUE;
}

static GstFlowReturn
th_par_setter_transform_ip (GstBaseTransform * trans, GstBuffer * buf)
{
  /* nothing to do, we set the new caps in prepare_output_buffer */
  return GST_FLOW_OK;
}

static GstCaps *
th_par_setter_transform_caps (GstBaseTransform * trans, GstPadDirection dir,
    GstCaps * caps)
{
  ThParSetter *setter = TH_PAR_SETTER (trans);
  GstCaps *ret;
  guint i;

  ret = gst_caps_copy (caps);

  for (i = 0; i < gst_caps_get_size (ret); ++i) {
    GstStructure *s;

    s = gst_caps_get_structure (ret, i);

    if (dir == GST_PAD_SINK) {
      gst_structure_remove_field (s, "pixel-aspect-ratio");
      gst_structure_set_value (s, "pixel-aspect-ratio", &setter->par);
    } else {
      GValue par_range = { 0, };

      g_value_init (&par_range, GST_TYPE_FRACTION_RANGE);
      gst_value_set_fraction_range_full (&par_range, 1, G_MAXINT, G_MAXINT, 1);
      gst_structure_set_value (s, "pixel-aspect-ratio", &par_range);
      g_value_unset (&par_range);
    }
  }

  GST_LOG_OBJECT (setter, "transformed %" GST_PTR_FORMAT " to %" GST_PTR_FORMAT,
      caps, ret);

  return ret;
}

#if 0
static gboolean
th_par_setter_set_caps (GstBaseTransform *basetrans, GstCaps *caps)
{
  ThParSetter *setter = TH_PAR_SETTER (basetrans);
  GstStructure *s;

  s = gst_caps_get_structure (caps, 0);
  if (!gst_structure_get_int (s, "width", &sink->width) ||
      !gst_structure_get_int (s, "height", &sink->height)) {
    return FALSE;
  }

  if (!gst_structure_get_fraction (s, "pixel-aspect-ratio", &sink->par_n,
      &sink->par_d)) {
    sink->par_n = 1;
    sink->par_d = 1;
  }

  GST_INFO_OBJECT (setter, "width x height = %d x %d", sink->width, sink->height);
  GST_INFO_OBJECT (setter, "pixel-aspect-ratio = %d/%d", sink->par_d, sink->par_n);
  return TRUE;
}
#endif

