# telepathy-butterfly - an MSN connection manager for Telepathy
#
# Copyright (C) 2006-2007 Ali Sabil <ali.sabil@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import logging
import weakref
from string import ascii_letters, digits

import dbus
import telepathy
from telepathy.interfaces import CHANNEL_TYPE_CALL

from tpufa.channel.text import UfATextChannel
from tpufa.channel.call import UfACallChannel
from tpufa.handle import UfAHandleFactory

__all__ = ['UfAChannelManager']

logger = logging.getLogger('UfA.ChannelManager')

_ASCII_ALNUM = ascii_letters + digits

class UfAChannelManager(telepathy.server.ChannelManager):
    __text_channel_id = 1
    __call_channel_id = 1
    __ft_channel_id = 1

    def __init__(self, connection, protocol):
        telepathy.server.ChannelManager.__init__(self, connection)

        self.set_requestable_channel_classes(protocol.requestable_channels)

        self.implement_channel_classes(telepathy.CHANNEL_TYPE_TEXT, self._get_text_channel)
        self.implement_channel_classes(telepathy.CHANNEL_TYPE_CALL, self._get_call_channel)

    def _get_text_channel(self, props, conversation=None):
        _, surpress_handler, handle = self._get_type_requested_handle(props)

        logger.debug('New text channel')

        path = "TextChannel%d" % self.__text_channel_id
        self.__text_channel_id += 1

        # Normal 1-1 chat
        if handle.get_type() == telepathy.HANDLE_TYPE_CONTACT:
            channel = UfATextChannel(self._conn, self, conversation, handle, props,
                object_path=path)
        else:
            raise telepathy.NotImplemented('Only contacts are allowed')

        return channel

    def _get_call_channel(self, props, call=None):
        _, surpress_handler, handle = self._get_type_requested_handle(props)

        if handle.get_type() != telepathy.HANDLE_TYPE_CONTACT:
            raise telepathy.NotImplemented('Only contacts are allowed')

        contact = handle.contact

        logger.debug('New call channel')

        path = "CallChannel/%d" % self.__call_channel_id
        call = "ufacall%d" % self.__call_channel_id
        self.__call_channel_id += 1

        return UfACallChannel(self._conn, self, call, handle, props, object_path=path)

    def existing_channel(self, props):
        type, _, handle = self._get_type_requested_handle(props)
        if type in self._channels:
            for otherHandle in self._channels[type]:
                if not self._conn.comparePhoneNumbers(otherHandle.name, handle.name):
                    continue
                if len(self._channels[type][otherHandle]) > 0:
                    return self._channels[type][otherHandle][-1]

    def call_channel_for_connid(self, connid):
        type = telepathy.CHANNEL_TYPE_CALL
        if type in self._channels:
            for handle in self._channels[type]:
                for channel in self._channels[type][handle]:
                    if channel._androidCallId == connid: 
                        return channel
