# vim: tabstop=4 expandtab shiftwidth=4 softtabstop=4
#!/usr/bin/env python3
# -*- encoding=utf8 -*-
#
# Author 2011 Hsin-Yi Chen
#
# This is a free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.
#
# This software is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this software; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA
import gi
gi.require_version('UDisks', '2.0')
from gi.repository import UDisks

import syslog
import subprocess
from ubiquity import misc

# Devices include disks and partitions
class UdisksDevices():
    def __init__(self):
        self.label = ''
        self.filesystem = ''
        self.device_file = ''
        self.parent_device = ''
        self.uuid = ''
        self.partition_number = 0
        self.disk_vendor = ''
        self.disk_model = ''
        self.disk_size = 0

class Udisks():
    def __init__(self):
        self._devices_list = []
        self._disks = []

    @property
    def disks(self):
        if not self._disks:
            self._disks = self.find_disks()
        return self._disks

    def devices(self):

        if len(self._devices_list) > 0:
            return self._devices_list

        self.udisks = UDisks.Client.new_sync(None)
        manager = self.udisks.get_object_manager()
        for obj in manager.get_objects():
            loop = obj.get_loop()
            block = obj.get_block()
            partition = obj.get_partition()
            drive = obj.get_drive()

            # Ignore loop or Non-block device or ReadOnly Device
            # Ignore the device is not expected to be partitionable.(non HintPartitionable)
            # such as floppy drives, optical drives and LVM logical volumes.
            if loop or \
               not block or \
               not block.get_cached_property("HintPartitionable").get_boolean() or \
               block.get_cached_property("ReadOnly").get_boolean():
                continue

            device_path = block.get_cached_property("Device").get_bytestring().decode('utf-8')
            # Ignore ramdisk /dev/ram[0-15], which Drive property is '/'
            if device_path.startswith('/dev/ram'):
                continue
            syslog.syslog("block is %s" % device_path)

            # Ignore USB or Removable Disks or partition
            drive_name = block.get_cached_property('Drive').get_string()
            if drive_name != '/': 
                drive = self.udisks.get_object(drive_name).get_drive()
                if drive:
                    connection_bus = drive.get_cached_property('ConnectionBus').get_string()
                    is_removable = drive.get_cached_property('Removable').get_boolean()
                    if connection_bus == 'usb' :
                        syslog.syslog("Ignore block is %s" % device_path)
                        continue
                    # The MMC device is removable when in SDIO devices
                    elif connection_bus != 'sdio' and is_removable:
                        syslog.syslog("Ignore block is %s" % device_path)
                        continue

            # instance a new device
            dev = UdisksDevices()

            # get general info from block
            dev.label = block.get_cached_property('IdLabel').get_string()
            dev.device_file = block.get_cached_property('PreferredDevice').get_bytestring().decode('utf-8')
            dev.filesystem = block.get_cached_property('IdType').get_string()
            dev.uuid = block.get_cached_property('IdUUID').get_string()
            # System devices are devices that require additional permissions to access.
            dev.is_system = block.get_cached_property('HintSystem').get_boolean()

            if partition:
                # device is a partition, the number is bigger than zero
                dev.partition_number = partition.get_cached_property('Number').get_uint32()

                # From the Table property, we could get the parent device
                parent = partition.get_cached_property('Table').get_string()
                parent_block = self.udisks.get_object(parent).get_block()
                dev.parent_device = parent_block.get_cached_property('PreferredDevice').get_bytestring().decode('utf-8')
                # We get the disk size from his parent while not use the partition size
                dev.disk_size = parent_block.get_cached_property('Size').get_uint64()
            else:
                # device is a disk, it should have a Drive object, but for NVMe or eMMC there aren't Drive object
                if drive_name != '/'and drive:
                    dev.disk_vendor = drive.get_cached_property('Vendor').get_string()
                    dev.disk_model = drive.get_cached_property('Model').get_string()
                    dev.disk_size = drive.get_cached_property('Size').get_uint64()

                # For NVMe, the device is nvme0n1
                # For eMMC, the device is mmcblk0
                # But for general device, it's sda, sdb, sdc, no number at last.
                # so if the last character is digit, we 
                # elif device_path.startswith('/dev/nvme') or device_path.startswith('/dev/mmcblk'):
                elif device_path[-1].isdigit():
                    dev.disk_size = block.get_cached_property('Size').get_uint64()
                    # use smartctl method to get Vendor & Model, which is from dell-recovery
                    with misc.raised_privileges():
                        getdisk_subp = subprocess.Popen(['smartctl', '-d', 'scsi', '--all', device_path], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
                        output = getdisk_subp.communicate()[0]
                        for prop in output.decode('utf-8').split('\n'):
                            if prop.startswith('Vendor:'):
                                dev.vendor = prop.split(':')[1].strip()
                            elif prop.startswith('Product:'):
                                dev.disk_model = prop.split(':')[1].strip()
                            else:
                                continue

            # We need at least 4G for recovery partition, so we will ignore disks when the size is not enough.
            if dev.disk_size / 1000000000 <= 4:
                syslog.syslog("Ignore block is %s, cause of no enough space!" % device_path)
                continue
            self._devices_list.append(dev)
        return self._devices_list

    def find_partition(self, labels):
        for dev in self.devices():
            if dev.label == labels:
                return dev

    def find_disks(self):
        """get disks in this machine"""
        disks = []

        for dev in self.devices():
            # Just check disks, so skip Partitions
            if dev.partition_number > 0:
                continue

            # if we made it this far, add it
            devicefile = dev.device_file
            devicemodel = dev.disk_model
            devicevendor = dev.disk_vendor
            devicesize = dev.disk_size
            devicesize_gb = "%i" % (devicesize / 1000000000)
            disks.append([devicefile, devicesize, "%s GB %s %s (%s)" % (devicesize_gb, devicevendor, devicemodel, devicefile)])

        # If multiple candidates were found, record in the logs
        if len(disks) == 0:
            raise RuntimeError("Unable to find and candidate hard disks to install to.")
        if len(disks) > 1:
            disks.sort()
        return disks

    def disable_swap(self):
        """Disables any swap partitions in use"""
        from ubiquity import misc
        if misc.execute_root('swapoff', '-a') is False:
            raise RuntimeError("Error when disabling swap")

    def find_factory_rp_stats(self, label):
        """Uses udisks to find the RP of a system and return stats on it
           Only use this method during bootstrap."""
        recovery = {}
        dev = self.find_partition(label)
        if not dev:
            return None
        recovery["label" ] = label
        recovery["device"] = dev.device_file
        recovery["fs"    ] = dev.filesystem
        recovery["slave" ] = dev.parent_device
        recovery["number"] = dev.partition_number
        recovery["uuid"]   = dev.uuid
        recovery["size_gb"] = dev.disk_size / 1000000000
        return recovery
