/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "PSHSensor.h"
#include "Session.h"
#include "SessionList.h"
#include "string.h"
#include "Log.h"

static SessionList &mSessionList(SessionList::getInstance());
static unsigned char cmdBuf[CMD_PARAM_MAX_SIZE + 3];

PSHSensor::PSHSensor(const struct sensor_device_t &device, const sensor_additional_information_t &information, const psh_private_data_t &data, PSHService *service)
        :Sensor(device, information), mPSHService(service), lastTimestamp(-1)
{
        memcpy(&priv_data, &data, sizeof(psh_private_data_t));
}

PSHSensor::~PSHSensor()
{

}

bool PSHSensor::startStreaming(Session * session, int64_t samplingPeriodUs, int64_t maxReportLatencyUs)
{
        bool result, params_changed = false;

        if ((samplingPeriodUs != bestBatchParams.batchDelay) || (maxReportLatencyUs != bestBatchParams.batchTimeout)) {
                params_changed = true;
        }

        result = Sensor::startStreaming(session, samplingPeriodUs, maxReportLatencyUs);
        if (!result)
                return result;

        if (params_changed) {
                ALOGD_IF(DEBUG, "%s line: %d sensor: %s samplingPeriodUs: %lld maxReportLatencyUs: %lld",
                         __FUNCTION__, __LINE__, device.name, samplingPeriodUs, maxReportLatencyUs);
                result = streamingOperationImpl(CMD_START_STREAMING);
        }

        return result;
}

bool PSHSensor::stopStreaming(Session * session)
{
        bool result = Sensor::stopStreaming(session);
        if (!result)
                return result;
        if ((bestBatchParams.batchDelay != -1) && (bestBatchParams.batchTimeout != -1)) {
                ALOGD_IF(DEBUG, "%s line: %d sensor: %s samplingPeriodUs: %lld maxReportLatencyUs: %lld",
                         __FUNCTION__, __LINE__, device.name, bestBatchParams.batchDelay, bestBatchParams.batchTimeout);
                result = streamingOperationImpl(CMD_START_STREAMING);
        } else {
                ALOGD_IF(DEBUG, "%s line: %d sensor: %s", __FUNCTION__, __LINE__, device.name);
                result = streamingOperationImpl(CMD_STOP_STREAMING);
        }
        return result;
}

bool PSHSensor::flushStreaming(Session * session)
{
        if (!Sensor::flushStreaming(session))
                return false;

        mPSHService->lockFlushQueue();

        mPSHService->addToFlushQueue(device.handle, session);
        bool result = streamingOperationImpl(CMD_FLUSH_STREAMING);
        if (!result)
                mPSHService->removeFromFlushQueue();

        mPSHService->unlockFlushQueue();

        ALOGD_IF(DEBUG, "%s line: %d sensor: %s result: %d", __FUNCTION__, __LINE__, device.name, result);

        return result;
}

bool PSHSensor::setProperty(Session * session, void * property, size_t size)
{
        if (size > CMD_PARAM_MAX_SIZE - 1) {
                ALOGE("%s line: %d Invalid property size: %d, maximum is: %d",
                      __FUNCTION__, __LINE__, size, CMD_PARAM_MAX_SIZE - 1);
                return false;
        }

        memset(cmdBuf, 0, CMD_PARAM_MAX_SIZE + 3);

        cmdBuf[0] = 0; // tran_id
        cmdBuf[1] = CMD_SET_PROPERTY;
        cmdBuf[2] = priv_data.id; // sensor_id
        cmdBuf[3] = size;
        memcpy(cmdBuf + 4, property, size);

        size_t resp_size;
        void * resp_buf;
        if (!mPSHService->sendControlCommand(cmdBuf, size + 4, &resp_buf, &resp_size)) {
                ALOGE("%s line: %d sendControlCommand failed!", __FUNCTION__, __LINE__);
                return false;
        }

        bool result = false;
        int left = resp_size;
        struct psh_cmd_resp *resp;
        struct psh_cmd_resp_ack *resp_ack;
        while (left > 0) {
                ALOGD_IF(DEBUG, "%s line: %d result: %d", __FUNCTION__, __LINE__, result);
                resp = reinterpret_cast<struct psh_cmd_resp *>(reinterpret_cast<char *>(resp_buf) + (resp_size - left));
                resp_ack = reinterpret_cast<struct psh_cmd_resp_ack *>(resp->buf);
                left -= sizeof(struct psh_cmd_resp) + resp->data_len;
                if (resp->type != RESP_CMD_ACK) {
                        ALOGE("%s line: %d Unexcepted response type: %u", __FUNCTION__, __LINE__, resp->type);
                        continue;
                }
#if 0 // Currently PSH firmware does not reponse the correct sensor id
                if (resp->sensor_id != cmdBuf.sensor_id) {
                        ALOGE("%s line: %d Response sensor id(%u) does not match the command(%u)",
                              __FUNCTION__, __LINE__, resp->sensor_id, cmdBuf.sensor_id);
                        continue;
                }
#endif
                if (resp->data_len < sizeof(struct psh_cmd_resp_ack)) {
                        ALOGE("%s line: %d Invalid response size: %d excepted: %d",
                              __FUNCTION__, __LINE__, resp->data_len, sizeof(struct psh_cmd_resp_ack));
                        continue;
                }

                if (resp_ack->ret != 0 ) {
                        ALOGE("%s line: %d Response error code: %d",
                              __FUNCTION__, __LINE__, resp_ack->ret);
                        continue;
                }

                if (resp_ack->cmd_id != CMD_SET_PROPERTY) {
                        ALOGE("%s line: %d Command ID not matched! Response: %d, Sent: %d",
                              __FUNCTION__, __LINE__, resp_ack->cmd_id, CMD_SET_PROPERTY);
                        continue;
                }

                result = true;
        }

        return result;
}

size_t PSHSensor::getProperty(Session * session, void * command, size_t commandSize, void **property)
{
        if (commandSize > CMD_PARAM_MAX_SIZE - 1) {
                ALOGE("%s line: %d Invalid command size: %d, maximum is: %d",
                      __FUNCTION__, __LINE__, commandSize, CMD_PARAM_MAX_SIZE - 1);
                return 0;
        }

        *property = NULL;
        memset(cmdBuf, 0, CMD_PARAM_MAX_SIZE + 3);

        cmdBuf[0] = 0; // tran_id
        cmdBuf[1] = CMD_GET_PROPERTY;
        cmdBuf[2] = priv_data.id; // sensor_id
        cmdBuf[3] = commandSize;
        memcpy(cmdBuf + 4, command, commandSize);

        size_t resp_size;
        void * resp_buf;
        if (!mPSHService->sendControlCommand(cmdBuf, commandSize + 4, &resp_buf, &resp_size)) {
                ALOGE("%s line: %d sendControlCommand failed!", __FUNCTION__, __LINE__);
                return 0;
        }

        // Todo: implement the function
        size_t size = 0;
        int left = resp_size;
        struct psh_cmd_resp *resp;
        struct psh_cmd_resp_ack *resp_ack;
        while (left > 0) {
                resp = reinterpret_cast<struct psh_cmd_resp *>(reinterpret_cast<char *>(resp_buf) + (resp_size - left));
                resp_ack = reinterpret_cast<struct psh_cmd_resp_ack *>(resp->buf);
                left -= sizeof(struct psh_cmd_resp) + resp->data_len;
                if (resp->type != RESP_CMD_ACK) {
                        ALOGE("%s line: %d Unexcepted response type: %u", __FUNCTION__, __LINE__, resp->type);
                        continue;
                }
#if 0 // Currently PSH firmware does not reponse the correct sensor id
                if (resp->sensor_id != cmdBuf.sensor_id) {
                        ALOGE("%s line: %d Response sensor id(%u) does not match the command(%u)",
                              __FUNCTION__, __LINE__, resp->sensor_id, cmdBuf.sensor_id);
                        continue;
                }
#endif
                if (resp->data_len < sizeof(struct psh_cmd_resp_ack)) {
                        ALOGE("%s line: %d Invalid response size: %d excepted: %d",
                              __FUNCTION__, __LINE__, resp->data_len, sizeof(struct psh_cmd_resp_ack));
                        continue;
                }

                if (resp_ack->ret != 0 ) {
                        ALOGE("%s line: %d Response error code: %d",
                              __FUNCTION__, __LINE__, resp_ack->ret);
                        continue;
                }

                if (resp_ack->cmd_id != CMD_GET_PROPERTY) {
                        ALOGE("%s line: %d Command ID not matched! Response: %d, Sent: %d",
                              __FUNCTION__, __LINE__, resp_ack->cmd_id, CMD_GET_PROPERTY);
                        continue;
                }

                size = resp->data_len - sizeof(struct psh_cmd_resp_ack);
                if (size > 0) {
                        *property = new unsigned char[size];
                        memcpy(*property, resp_ack->extra, size);
                }
        }

        return size;
}

size_t PSHSensor::convertToStreaming(const char * buf, size_t bufSize)
{
        if (mStreamingSize < bufSize) {
                if (mStreaming != NULL)
                        delete [] mStreaming;
                mStreaming = new char[bufSize];
                mStreamingSize = bufSize;
        }
        memcpy(mStreaming, buf, bufSize);
        return bufSize;
}

void PSHSensor::handleEvents(int fd)
{

}

void PSHSensor::handleEvents(const char * buf, size_t bufSize)
{
        size_t eventsCount = 0;
        size_t streamingCount = 0;
        std::map<Session *, BatchParams>::iterator it;

        mSessionList.lockSessions();
        lockbatchParams(); // lock batchParams for it may be changed in main thread
        for (it = batchParams.begin(); it != batchParams.end(); it++) {
                Session *session = it->first;
                if (session->getCategory() == ANDROID_STANDARD) {
                        if (eventsCount == 0) {
                                eventsCount = convertToEvent(buf, bufSize);
                                for (int i = 0; i < eventsCount; i++) { // Workround for timestamp disorder.
                                        if (mEvents[i].timestamp <= lastTimestamp)
                                                mEvents[i].timestamp = lastTimestamp + 100000;
                                        lastTimestamp = mEvents[i].timestamp;
                                }
                        }
                        session->dispatchEvents(eventsCount * sizeof(struct sensors_event_t), mEvents);
                } else if (session->getCategory() == AWARE_SERVICE) {
                        if (streamingCount == 0)
                                streamingCount = convertToStreaming(buf, bufSize);
                        session->dispatchEvents(streamingCount, mStreaming);
                }
        }
        unlockbatchParams();
        mSessionList.unlockSessions();
}

struct psh_streaming_operation_cmd {
        struct psh_cmd header;
        unsigned short data_rate;
        unsigned short buffer_delay;
        unsigned short bit_cfg;
} __attribute__ ((packed));

bool PSHSensor::streamingOperationImpl(unsigned char cmd_id)
{
        int dataRate = bestBatchParams.batchDelay > 0 ? (1000000 + bestBatchParams.batchDelay - 1) / bestBatchParams.batchDelay : 1;
        int maxReportLatencyMs = bestBatchParams.batchTimeout / 1000;
        psh_streaming_operation_cmd cmd;
        size_t resp_size;
        void * resp_buf;

        if (cmd_id == CMD_START_STREAMING) {
                cmd.data_rate = dataRate;
                cmd.buffer_delay = maxReportLatencyMs;
                cmd.bit_cfg = (device.flags & SENSOR_FLAG_WAKE_UP) ? 1 : 0;
        } else if (cmd_id == CMD_STOP_STREAMING) {
                cmd_id = CMD_START_STREAMING; // work round: CMD_STOP_STREAMING not worked in firmware currently
                cmd.data_rate = 0;
                cmd.buffer_delay = 0;
                cmd.bit_cfg = 0;
        } else { // flush
                cmd.data_rate = 4;
                cmd.buffer_delay = 0;
                cmd.bit_cfg = 0;
        }
        cmd.header.cmd_id = cmd_id;
        cmd.header.tran_id = 0;
        cmd.header.sensor_id = priv_data.id;

        if (!mPSHService->sendControlCommand(&cmd, sizeof(cmd), &resp_buf, &resp_size))
                return false;

        if (resp_size < sizeof(struct psh_cmd_resp)) {
                ALOGE("%s line: %d Invalid response size: %d", resp_size);
                return false;
        }


        bool result = false;
        int left = resp_size;
        struct psh_cmd_resp *resp;
        struct psh_cmd_resp_ack *resp_ack;
        while (left > 0) {
                ALOGD_IF(DEBUG, "%s line: %d result: %d", __FUNCTION__, __LINE__, result);
                resp = reinterpret_cast<struct psh_cmd_resp *>(reinterpret_cast<char *>(resp_buf) + (resp_size - left));
                resp_ack = reinterpret_cast<struct psh_cmd_resp_ack *>(resp->buf);
                left -= sizeof(struct psh_cmd_resp) + resp->data_len;
                if (resp->type != RESP_CMD_ACK) {
                        ALOGE("%s line: %d Unexcepted response type: %u", __FUNCTION__, __LINE__, resp->type);
                        continue;
                }
#if 0 // Currently PSH firmware does not reponse the correct sensor id
                if (resp->sensor_id != cmd.header.sensor_id) {
                        ALOGE("%s line: %d Response sensor id(%u) does not match the command(%u)",
                              __FUNCTION__, __LINE__, resp->sensor_id, cmd.header.sensor_id);
                        continue;
                }
#endif
                if (resp->data_len < sizeof(struct psh_cmd_resp_ack)) {
                        ALOGE("%s line: %d Invalid response size: %d excepted: %d",
                              __FUNCTION__, __LINE__, resp->data_len, sizeof(struct psh_cmd_resp_ack));
                        continue;
                }

                if (resp_ack->ret != 0 ) {
                        ALOGE("%s line: %d Response error code: %d",
                              __FUNCTION__, __LINE__, resp_ack->ret);
                        continue;
                }

                if (resp_ack->cmd_id != cmd.header.cmd_id) {
                        ALOGE("%s line: %d Command ID not matched! Response: %d, Sent: %d",
                              __FUNCTION__, __LINE__, resp_ack->cmd_id, cmd.header.cmd_id);
                        continue;
                }

                result = true;
        }

        return result;
}
