/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "PSHFaceDetection.h"
#include "Log.h"

typedef struct _SFacePosition
{
        // The normalized (0 to 1) horizontal position in the scene of the upper-left corner of the first face rectangle.
        float left;
        // The normalized (0 to 1) vertical position in the scene of the upper-left corner of the first face rectangle.
        float top;
        // The normalized (0 to 1) horizontal position in the scene of the lower-right corner of the first face rectangle.
        float right;
        // The normalized (0 to 1) vertical position in the scene of the lower-right of the first face rectangle.
        float bottom;
}SFacePosition;

struct fd_data
{
        // The time in nanosecond at which the frame was captured
        uint64_t timestamp;
        // Number of faces which were detected
        uint32_t faces_nr;
        // Array which holds the position of the faces detected
        SFacePosition faces[0];
}__attribute__ ((packed));

PSHFaceDetection::PSHFaceDetection(const struct sensor_device_t &device, const sensor_additional_information_t &information, const psh_private_data_t &data, PSHService *service)
        :PSHSensor(device, information, data, service)
{

}

PSHFaceDetection::~PSHFaceDetection()
{

}

size_t PSHFaceDetection::convertToEvent(const char * buf, size_t bufSize)
{
        size_t count = 0, remaining_size = bufSize;
        const char *p = buf;
        const struct fd_data *faced;
        SFacePosition *pos;

        while (remaining_size > 0) {
                if (remaining_size < sizeof(struct fd_data)) {
                        ALOGE("%s line: %d invalid remaining_size: %u", __FUNCTION__, __LINE__, remaining_size);
                        return count;
                }

                faced = reinterpret_cast<const struct fd_data *>(p);
                if (remaining_size < sizeof(struct fd_data) + faced->faces_nr * sizeof(SFacePosition)) {
                        ALOGE("%s line: %d invalid remaining_size: %u", __FUNCTION__, __LINE__, remaining_size);
                        return count;
                }

                reallocEventsBuffer(count + 1);

                mEvents[count].timestamp = faced->timestamp;
                for (int i = 0; i < 4; i++) {
                        if (i < faced->faces_nr) {
                                mEvents[count].data[4 * i + 0] = faced->faces[i].left;
                                mEvents[count].data[4 * i + 1] = faced->faces[i].top;
                                mEvents[count].data[4 * i + 2] = faced->faces[i].right;
                                mEvents[count].data[4 * i + 3] = faced->faces[i].bottom;
                        } else {
                                for (int j = 0; j < 4; j++)
                                        mEvents[count].data[4 * i + j] = -1; // invalid value;
                        }
                }

                count++;
                remaining_size -= sizeof(struct fd_data) + faced->faces_nr * sizeof(SFacePosition);
                p += sizeof(struct fd_data) + faced->faces_nr * sizeof(SFacePosition);
        }

        return count;
}
