/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <string.h>
#include "SensorEnumerator.h"
#include "psh/PSHService.h"
#include "ish/ISHService.h"
#include "iio/IIOService.h"
#include "SensorTypeMap.h"
#include "Log.h"

#define SENSOR_CONFIG_FILE "/etc/sensors_config.xml"
#define SENSOR_XML_ROOT "sensor_hub_config"
#define GRAVITY_EARTH           (9.80665f)
#define M_PI 3.14159265358979323846 /* pi */

SensorEnumerator::SensorEnumerator()
        :mStringBuffer(StringBuffer::getInstance())
{
}

bool SensorEnumerator::enumerateSensors(std::map<int, Sensor *> &sensorList, std::vector<PlatformService *> &serviceList)
{
        xmlDocPtr doc;
        xmlNodePtr root;
        Sensor * sensor;
        struct sensor_device_t device;
        sensor_additional_information_t information;

        doc = xmlReadFile(SENSOR_CONFIG_FILE, NULL, XML_PARSE_NOBLANKS);
        if (doc==NULL) {
                ALOGE("%s line: %d XML Document \"%s\" does not parsed successfully.", __FUNCTION__, __LINE__, SENSOR_CONFIG_FILE);
                return false;
        }

        root=xmlDocGetRootElement(doc);
        if (root==NULL) {
                ALOGE("%s line: %d Empty XML document.", __FUNCTION__, __LINE__);
                xmlFreeDoc(doc);
                return false;
        }

        if (strcmp(reinterpret_cast<const char *>(root->name), SENSOR_XML_ROOT)) {
                ALOGE("%s line: %d Wrong XML document, cannot find \"%s\" element!", __FUNCTION__, __LINE__, SENSOR_XML_ROOT);
                xmlFreeDoc(doc);
                return false;
        }

        for (xmlNodePtr node = root->xmlChildrenNode; node != NULL; node = node->next) {
                std::vector<Sensor *> sensors;
                Sensor * sensor;
                PlatformService * service;
                int handle;

                if (!strcmp(reinterpret_cast<const char *>(node->name), "ish_sensors")) {
                        service = new ISHService();
                        serviceList.push_back(service);
                        service->enumerateSensors(this, node, sensors);
                } else if (!strcmp(reinterpret_cast<const char *>(node->name), "psh_sensors")) {
                        service = new PSHService();
                        serviceList.push_back(service);
                        service->enumerateSensors(this, node, sensors);
                } else if (!strcmp(reinterpret_cast<const char *>(node->name), "iio_sensors")) {
                        service = new IIOService();
                        serviceList.push_back(service);
                        service->enumerateSensors(this, node, sensors);
                }

                for (size_t i = 0; i < sensors.size(); i++) {
                        sensor = sensors[i];
                        handle = handleAllocator.allocateHandle();
                        if (handle < SENSORS_HANDLE_BASE) {
                                ALOGE("%s line: %d allocate sensor handle error! Only %d sensors are allowed.", __FUNCTION__, __LINE__, SENSORS_HANDLE_COUNT);
                                return false;
                        }
                        sensor->setHandle(handle);
                        if (mStringBuffer.getMemReallocated()) {
                                sensor->setName(mStringBuffer.getNewStrAddr(const_cast<char *>(sensor->getName())));
                                sensor->setVendor(mStringBuffer.getNewStrAddr(const_cast<char *>(sensor->getVendor())));
                                sensor->setStringType(mStringBuffer.getNewStrAddr(const_cast<char *>(sensor->getStringType())));
                                sensor->setRequiredPermission(mStringBuffer.getNewStrAddr(const_cast<char *>(sensor->getRequiredPermission())));
                        }
                        sensorList[handle] = sensor;
                }
                sensors.clear();
        }

        return true;
}

bool SensorEnumerator::initializeSensorDevice(const xmlNodePtr node, struct sensor_device_t &device, const char * typeStr)
{
        xmlNodePtr p = node->xmlChildrenNode;
        xmlChar *str = NULL;
        xmlChar *attr = NULL;
        int defaultType;

        memset(&device, 0, sizeof(device));
        device.type = sensorTypeMap.getSensorType(typeStr);
        device.flags = sensorTypeMap.getSensorDefaultFlags(typeStr);
        defaultType = device.type;

        while (p != NULL) {
                str = xmlNodeGetContent(p);
                ALOGV_IF(VERBOSE, "%s line: %d node name: %s node content: %s", __FUNCTION__, __LINE__, p->name, str);
                if (str == NULL) {
                        p = p->next;
                        continue;
                }

                if (!strcmp(reinterpret_cast<const char *>(str), "0") || !strcmp(reinterpret_cast<const char *>(str), "")) {
                        xmlFree(str);
                        str = NULL;
                        p = p->next;
                        continue;
                }

                if (!strcmp(reinterpret_cast<const char *>(p->name), "name")) {
                        device.name = mStringBuffer.addString(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "vendor")) {
                        device.vendor = mStringBuffer.addString(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "version")) {
                        device.version = atoi(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "type")) {
                        device.type = atoi(reinterpret_cast<char *>(str));
                        if (defaultType < SENSOR_TYPE_DEVICE_PRIVATE_BASE)
                                ALOGW("%s line: %d Android default sensor type changed by config! name: %s default: %d config: %d",
                                      __FUNCTION__, __LINE__, typeStr, defaultType, device.type);
                        else if (defaultType < SENSOR_TYPE_CUSTOMER_BASE)
                                ALOGW("%s line: %d Intel default sensor type changed by config! name: %s default: %d config: %d",
                                      __FUNCTION__, __LINE__, typeStr, defaultType, device.type);
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "stringType")) {
                        if (defaultType < SENSOR_TYPE_DEVICE_PRIVATE_BASE)
                                ALOGW("%s line: %d Android default sensor string type changed by config! name: %s default: %s%s config: %s",
                                      __FUNCTION__, __LINE__, typeStr, sensorTypeMap.getSensorStringTypePrefix(typeStr), typeStr, reinterpret_cast<char *>(str));
                        else if (defaultType < SENSOR_TYPE_CUSTOMER_BASE)
                                ALOGW("%s line: %d Intel default sensor string type changed by config! name: %s default: %s%s config: %s",
                                      __FUNCTION__, __LINE__, typeStr, sensorTypeMap.getSensorStringTypePrefix(typeStr), typeStr, reinterpret_cast<char *>(str));
                        device.stringType = mStringBuffer.addString(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "maxRange")) {
                        attr = xmlGetProp(p, reinterpret_cast<const xmlChar*>("unit"));
                        device.maxRange = atof(reinterpret_cast<char *>(str)) * getUnitValue(reinterpret_cast<const char *>(attr));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "resolution")) {
                        attr = xmlGetProp(p, reinterpret_cast<const xmlChar*>("unit"));
                        device.resolution = atof(reinterpret_cast<char *>(str)) * getUnitValue(reinterpret_cast<const char *>(attr));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "power")) {
                        device.power = atof(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "minDelay")) {
                        device.minDelay = atoi(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "maxDelay")) {
                        device.maxDelay = atoi(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "fifoReservedEventCount")) {
                        device.fifoReservedEventCount = atoi(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "fifoMaxEventCount")) {
                        device.fifoMaxEventCount = atoi(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "requiredPermission")) {
                        device.requiredPermission = mStringBuffer.addString(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "wakeup")) {
                        uint32_t defaultWakeup = device.flags & SENSOR_FLAG_WAKE_UP;
                        uint32_t configWakeup = atoi(reinterpret_cast<char *>(str)) ? SENSOR_FLAG_WAKE_UP : 0;
                        if (defaultType < SENSOR_TYPE_DEVICE_PRIVATE_BASE)
                                ALOGW("%s line: %d Android default wakeup mode changed by config! name: %s default: %s config: %s",
                                      __FUNCTION__, __LINE__, typeStr, defaultWakeup ? "Wakeup" : "Non-wakeup", configWakeup ? "Wakeup" : "Non-wakeup");
                        else if (defaultType < SENSOR_TYPE_CUSTOMER_BASE)
                                ALOGW("%s line: %d Intel default wakeup mode changed by config! name: %s default: %s config: %s",
                                      __FUNCTION__, __LINE__, typeStr, defaultWakeup ? "Wakeup" : "Non-wakeup", configWakeup ? "Wakeup" : "Non-wakeup");
                        device.flags &= ~SENSOR_FLAG_WAKE_UP;
                        device.flags |= configWakeup;
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "report_mode")) {
                        uint32_t defaultReportingMode = device.flags & REPORTING_MODE_MASK;
                        uint32_t configReportingMode = getReportingMode(reinterpret_cast<char *>(str));
                        if (defaultType < SENSOR_TYPE_DEVICE_PRIVATE_BASE)
                                ALOGW("%s line: %d Android default reporting mode changed by config! name: %s default: %s config: %s",
                                      __FUNCTION__, __LINE__, typeStr, getReportingModeName(defaultReportingMode), reinterpret_cast<char *>(str));
                        else if (defaultType < SENSOR_TYPE_CUSTOMER_BASE)
                                ALOGW("%s line: %d Intel default reporting mode changed by config! name: %s default: %s config: %s",
                                      __FUNCTION__, __LINE__, typeStr, getReportingModeName(defaultReportingMode), reinterpret_cast<char *>(str));
                        device.flags &= ~REPORTING_MODE_MASK;
                        device.flags |= configReportingMode;
                }
                p = p->next;
                if (str != NULL) {
                        xmlFree(str);
                        str = NULL;
                }
                if (attr != NULL) {
                        xmlFree(attr);
                        attr = NULL;
                }
        }

        if (device.name == NULL)
                device.name = mStringBuffer.addString("");
        if (device.vendor == NULL)
                device.vendor = mStringBuffer.addString("");
        if (device.requiredPermission == NULL) {
                if (device.type == SENSOR_TYPE_HEART_RATE)
                        device.requiredPermission = mStringBuffer.addString(SENSOR_PERMISSION_BODY_SENSORS);
                else
                        device.requiredPermission = mStringBuffer.addString("");
        }

        if (device.stringType == NULL) {
                const char * prefix = sensorTypeMap.getSensorStringTypePrefix(typeStr);
                char * stringType = new char[strlen(prefix) + strlen(typeStr) + 1];
                strcpy(stringType, prefix);
                strcat(stringType, typeStr);
                device.stringType = mStringBuffer.addString(stringType);
                delete [] stringType;
        }

        if (VERBOSE) {
                ALOGV_IF(VERBOSE, "%s line: %d dump sensor device information:", __FUNCTION__, __LINE__);
                dump(device);
        }

        return true;
}

void SensorEnumerator::dump(struct sensor_device_t &device)
{
        ALOGV_IF(VERBOSE, "************************%s************************\n", mStringBuffer.getNewStrAddr(device.name));
        ALOGV_IF(VERBOSE, "name: %s\n", mStringBuffer.getNewStrAddr(device.name));
        ALOGV_IF(VERBOSE, "vendor: %s\n", mStringBuffer.getNewStrAddr(device.vendor));
        ALOGV_IF(VERBOSE, "version: %d\n", device.version);
        ALOGV_IF(VERBOSE, "handle: %d\n", device.handle);
        ALOGV_IF(VERBOSE, "type: %d\n", device.type);
        ALOGV_IF(VERBOSE, "maxRange: %f\n", device.maxRange);
        ALOGV_IF(VERBOSE, "resolution: %f\n", device.resolution);
        ALOGV_IF(VERBOSE, "power: %f\n", device.power);
        ALOGV_IF(VERBOSE, "minDelay: %d\n", device.minDelay);
        ALOGV_IF(VERBOSE, "fifoReservedEventCount: %u\n", device.fifoReservedEventCount);
        ALOGV_IF(VERBOSE, "fifoMaxEventCount: %u\n", device.fifoMaxEventCount);
        ALOGV_IF(VERBOSE, "stringType: %s\n", mStringBuffer.getNewStrAddr(device.stringType));
        ALOGV_IF(VERBOSE, "requiredPermission: %s\n", mStringBuffer.getNewStrAddr(device.requiredPermission));
        ALOGV_IF(VERBOSE, "maxDelay: %d\n", device.maxDelay);
        ALOGV_IF(VERBOSE, "flags: 0x%x\n", device.flags);
        ALOGV_IF(VERBOSE, "************************%s************************\n", mStringBuffer.getNewStrAddr(device.name));
}
void SensorEnumerator::initializeSensorAdditionalInformation(const xmlNodePtr node, sensor_additional_information_t &information)
{
        xmlNodePtr p = node->xmlChildrenNode;
        xmlChar *str = NULL;
        xmlChar *attr = NULL;

        information.expose_to_android = false;
        information.expose_to_csdk = false;
        information.axisMap[AXIS_X] = AXIS_X;
        information.axisMap[AXIS_Y] = AXIS_Y;
        information.axisMap[AXIS_Z] = AXIS_Z;
        information.scale_android.clear();
        information.scale_csdk.clear();

        while (p != NULL) {
                if (!strcmp(reinterpret_cast<const char *>(p->name), "scale_android")) {
                        initializeScaleInformation(p, information.scale_android);
                        p = p->next;
                        continue;
                }
                if (!strcmp(reinterpret_cast<const char *>(p->name), "scale_csdk")) {
                        initializeScaleInformation(p, information.scale_csdk);
                        p = p->next;
                        continue;
                }
                // Axis map only for physical sensors, so it is 3-axis
                if (!strcmp(reinterpret_cast<const char *>(p->name), "axis_map")) {
                        attr = xmlGetProp(p, reinterpret_cast<const xmlChar*>("axis_x"));
                        if (attr != NULL) {
                                information.axisMap[AXIS_X] = getAxis(reinterpret_cast<const char *>(attr));
                                xmlFree(attr);
                        }
                        attr = xmlGetProp(p, reinterpret_cast<const xmlChar*>("axis_y"));
                        if (attr != NULL) {
                                information.axisMap[AXIS_Y] = getAxis(reinterpret_cast<const char *>(attr));
                                xmlFree(attr);
                        }
                        attr = xmlGetProp(p, reinterpret_cast<const xmlChar*>("axis_z"));
                        if (attr != NULL) {
                                information.axisMap[AXIS_Z] = getAxis(reinterpret_cast<const char *>(attr));
                                xmlFree(attr);
                        }
                        attr = NULL;
                        // Check invalid
                        if (information.axisMap[AXIS_X] == information.axisMap[AXIS_Y] ||
                            information.axisMap[AXIS_Y] == information.axisMap[AXIS_Z] ||
                            information.axisMap[AXIS_Z] == information.axisMap[AXIS_X] ||
                            information.axisMap[AXIS_X] == AXIS_MAX ||
                            information.axisMap[AXIS_Y] == AXIS_MAX ||
                            information.axisMap[AXIS_Z] == AXIS_MAX) {
                                ALOGE("%s line: %d Invalid axis map: AXIS_X: %d AXIS_Y: %d AXIS_Z: %d", __FUNCTION__, __LINE__,
                                      information.axisMap[AXIS_X], information.axisMap[AXIS_Y], information.axisMap[AXIS_Z]);
                                information.axisMap[AXIS_X] = AXIS_X;
                                information.axisMap[AXIS_Y] = AXIS_Y;
                                information.axisMap[AXIS_Z] = AXIS_Z;
                        }
                        p = p->next;
                        continue;
                }

                str = xmlNodeGetContent(p);
                if (str == NULL) {
                        p = p->next;
                        continue;
                }

                if (!strcmp(reinterpret_cast<const char *>(str), "0") || !strcmp(reinterpret_cast<const char *>(str), "")) {
                        xmlFree(str);
                        str = NULL;
                        p = p->next;
                        continue;
                }

                if (!strcmp(reinterpret_cast<const char *>(p->name), "expose_to_android")) {
                        information.expose_to_android = atoi(reinterpret_cast<char *>(str));
                } else if (!strcmp(reinterpret_cast<const char *>(p->name), "expose_to_csdk")) {
                        information.expose_to_csdk = atoi(reinterpret_cast<char *>(str));
                }
                xmlFree(str);
                str = NULL;
                p = p->next;
        }
}

float SensorEnumerator::getUnitValue(const char * name)
{
        if (name == NULL)
                return 1.0;

        if (strcmp(name, "GRAVITY_EARTH") == 0)
                return GRAVITY_EARTH;
        if (strcmp(name, "M_PI") == 0)
                return M_PI;

        return 1.0;
}

uint32_t SensorEnumerator::getReportingMode(const char * reportingModeName)
{
        if (strcmp(reportingModeName, "continuous") == 0)
                return SENSOR_FLAG_CONTINUOUS_MODE;
        if (strcmp(reportingModeName, "on_change") == 0)
                return SENSOR_FLAG_ON_CHANGE_MODE;
        if (strcmp(reportingModeName, "one_shot") == 0)
                return SENSOR_FLAG_ONE_SHOT_MODE;
        if (strcmp(reportingModeName, "special_reporting") == 0)
                return SENSOR_FLAG_SPECIAL_REPORTING_MODE;

        ALOGE("%s line: %d Unknown reporting mode: %s, using continuous mode", __FUNCTION__, __LINE__, reportingModeName);
        return SENSOR_FLAG_CONTINUOUS_MODE;
}

const char * SensorEnumerator::getReportingModeName(uint32_t reportingMode)
{
        switch (reportingMode) {
        case SENSOR_FLAG_CONTINUOUS_MODE:
                return "continuous";
        case SENSOR_FLAG_ON_CHANGE_MODE:
                return "on_change";
        case SENSOR_FLAG_ONE_SHOT_MODE:
                return "one_shot";
        case SENSOR_FLAG_SPECIAL_REPORTING_MODE:
                return "special_reporting";
        }

        return "unknown";
}

void SensorEnumerator::initializeScaleInformation(const xmlNodePtr node, std::vector<float> &scaleInformation)
{
        xmlChar *attr = NULL;
        float scale, unit;

        attr = xmlGetProp(node, reinterpret_cast<const xmlChar*>("unit"));
        unit = getUnitValue(reinterpret_cast<const char *>(attr));
        if (attr != NULL) {
                xmlFree(attr);
        }

        attr = xmlGetProp(node, reinterpret_cast<const xmlChar*>("axis_x"));
        if (attr != NULL) {
                scale = atof(reinterpret_cast<const char *>(attr));
                xmlFree(attr);
                scaleInformation.push_back(scale * unit);
                attr = xmlGetProp(node, reinterpret_cast<const xmlChar*>("axis_y"));
                if (attr != NULL) {
                        scale = atof(reinterpret_cast<const char *>(attr));
                        xmlFree(attr);
                        scaleInformation.push_back(scale * unit);
                        attr = xmlGetProp(node, reinterpret_cast<const xmlChar*>("axis_z"));
                        if (attr != NULL) {
                                scale = atof(reinterpret_cast<const char *>(attr));
                                xmlFree(attr);
                                scaleInformation.push_back(scale * unit);
                        }
                }
        } else {
                char axisNumber[8];
                int count = 0;

                snprintf(axisNumber, 8, "axis_%d", count);
                attr = xmlGetProp(node, reinterpret_cast<const xmlChar*>(axisNumber));
                while (attr != NULL) {
                        scale = atof(reinterpret_cast<const char *>(attr));
                        xmlFree(attr);
                        scaleInformation.push_back(scale * unit);
                        count++;
                        if (count > 99)
                                break;
                        snprintf(axisNumber, 8, "axis_%d", count);
                        attr = xmlGetProp(node, reinterpret_cast<const xmlChar*>(axisNumber));
                }
        }

}

sensor_axis_t SensorEnumerator::getAxis(const char * axisName)
{
        if (strcmp(axisName, "X") == 0 || strcmp(axisName,"x") == 0)
                return AXIS_X;
        if (strcmp(axisName, "Y") == 0 || strcmp(axisName,"y") == 0)
                return AXIS_Y;
        if (strcmp(axisName, "Z") == 0 || strcmp(axisName,"z") == 0)
                return AXIS_Z;

        ALOGE("%s line: %d invalid axis: %s", __FUNCTION__, __LINE__, axisName);
        return AXIS_MAX;
}

bool SensorEnumerator::initializeSensorDevice(struct sensor_device_t &device, const char *name, const char *vendor, const char *required_permission)
{
        memset(&device, 0, sizeof(device));

        if (name == NULL) {
                ALOGE("%s line: %d can't be initialized without a name!", __FUNCTION__, __LINE__);
                return false;
        }

        device.name = mStringBuffer.addString(name);

        if (vendor == NULL)
                device.vendor = mStringBuffer.addString("Unknown");
        else
                device.vendor = mStringBuffer.addString(vendor);
        if (required_permission == NULL)
                device.requiredPermission = mStringBuffer.addString("");
        else
                device.requiredPermission = mStringBuffer.addString(required_permission);

        const char * prefix = sensorTypeMap.getSensorStringTypePrefix(name);
        char * stringType = new char[strlen(prefix) + strlen(name) + 1];
        strcpy(stringType, prefix);
        strcat(stringType, name);
        for (size_t i = strlen(prefix); i < strlen(prefix) + strlen(name); i++) {
                if (!isalnum(stringType[i])) {
                        stringType[i] = '_';
                } else if (isupper(stringType[i])) {
                        stringType[i] = tolower(stringType[i]);
                }
        }
        device.stringType = mStringBuffer.addString(stringType);
        delete [] stringType;

        device.version = 1;
        device.type = sensorTypeMap.getSensorType(device.name);
        device.maxRange = 1;
        device.resolution = 1.0;
        device.power = 0.006;
        device.flags = sensorTypeMap.getSensorDefaultFlags(device.name);

        return true;
}

char * SensorEnumerator::getStringBufferBase()
{
        return mStringBuffer.base();
}

int SensorEnumerator::getStringBufferSize() {
        return mStringBuffer.usedSize();
}
