/*
 *
 *  Quick Controller
 *
 *  Copyright (C) 2008-2009  Intel Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/****************************************************************************
File Name:          QCPSDK.C

Description:
    This file implements to call the Device Control Service interfaces.
         This class has following functions.
         1.InitDCS               Initialize the Device Control Service
         2.InitBrightness        Get brightness initalizal information
         3.LCDBrightnessUp       Brightness adjust up
         4.LCDBrightnessDown     Brightness adjust down
         5.EnableAutoRotation    Enable the auto rotation
         6.DisableAutoRotation   Disable the auto rotation
Environment (opt):
    OS: Windows XP table and Ubuntu with latest service pack,
    SE: CodeBlocks
Notes (opt):
  =====================================================================
  Revision   Revision History               Author     Date
  =====================================================================

  0.1        Create                         Chen Dequan   2008-7-26

  =====================================================================

****************************************************************************/
#include "../CMPC_QCPBasicIF.h"
#include "../CMPC_QCPDef.h"
#include "QCPSDK.h"

#if defined(__UNIX__)
    #include <X11/Xlib.h>
    #include <X11/extensions/Xrandr.h>
#endif

/**************************************************************************************************
* Function Name : CDCSBrightRotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
CDCSBrightRotation::CDCSBrightRotation()
{
    m_iDCSRet = DCS_FAIL_OPERATION;
    m_iRet = DCS_FAIL_OPERATION;
    m_iBrightMax = 0;
    m_iCurrentBright = 0;
    m_bEnabled= 0;           //Auto rotation disabled(0)
}

/**************************************************************************************************
* Function Name : ~CDCSBrightRotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
CDCSBrightRotation::~CDCSBrightRotation()
{
}

/**************************************************************************************************
* Function Name : InitDCS                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::InitDCS()
{
    m_iDCSRet = DCS_Initialize();         //Initialize DCS
    if (DCS_SUCCESS != m_iDCSRet)
    {
        return false;
    }

    return true;
}

/**************************************************************************************************
* Function Name : InitBrightness                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::InitBrightness(int *piBrightLevel)
{
    if ((NULL == piBrightLevel) || (DCS_SUCCESS != m_iDCSRet))
    {
        return false; //The DCS hasn't been launched
    }

    m_iBrightMax = BRIGHTLEVEL;//Getting the brightness max value:7

    m_iRet = DCS_GetLCDBrightness(&m_iCurrentBright);//Getting current brightness
    if ( DCS_SUCCESS != m_iRet )
    {
        *piBrightLevel = m_iCurrentBright; //Return 0
        return false;
    }

    if( m_iCurrentBright >= 0 )
    {
        *piBrightLevel = m_iCurrentBright; //Return current brightness level for update UI
        return true;
    }

    *piBrightLevel = m_iCurrentBright;     //Return 0
    return false;
}

/**************************************************************************************************
* Function Name : LCDBrightnessUp                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::LCDBrightnessUp(int iBrightLevel, int *piBrightLevel)
{
    if ( (NULL == piBrightLevel) || (DCS_SUCCESS != m_iDCSRet) )
    {
        return false; // The DCS hasn't been launched
    }
    m_iCurrentBright = iBrightLevel;
    if ( m_iCurrentBright < m_iBrightMax )
    {
        iBrightLevel = m_iCurrentBright + BRIGHTNESSCHANGELEVEL;//Up brightness

        m_iRet = DCS_SetLCDBrightness(iBrightLevel);//Setting brightness
        if (DCS_SUCCESS != m_iRet)
        {
            *piBrightLevel = m_iCurrentBright;
            return false;
        }

        m_iCurrentBright = iBrightLevel; //Update current brightness level
        *piBrightLevel   = iBrightLevel; //Update UI
        return true;
    }

    *piBrightLevel = m_iCurrentBright;
	return false;
}

/**************************************************************************************************
* Function Name : LCDBrightnessDown                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::LCDBrightnessDown(int iBrightLevel, int *piBrightLevel)
{
    if ( (NULL == piBrightLevel) || (DCS_SUCCESS != m_iDCSRet) )
    {
        return false;// The DCS hasn't been launched
    }

    m_iCurrentBright = iBrightLevel;
    if ( m_iCurrentBright >= BRIGHTNESSCHANGELEVEL )
    {
        iBrightLevel = m_iCurrentBright - BRIGHTNESSCHANGELEVEL; //Down brightnss

        m_iRet = DCS_SetLCDBrightness(iBrightLevel);
        if (DCS_SUCCESS != m_iRet)
        {
            *piBrightLevel = m_iCurrentBright;
            return false;
        }

        m_iCurrentBright = iBrightLevel; //Update current brightness level
        *piBrightLevel   = iBrightLevel; //Update UI
        return true;
    }

    *piBrightLevel = m_iCurrentBright;
	return false;
}

/**************************************************************************************************
* Function Name : SetLCDBrightness                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::SetLCDBrightness(int iBrightLevel)
{
    if (DCS_SUCCESS != m_iDCSRet)
    {
        return false; // The DCS hasn't been launched
    }

    //Setting brightness level when user click the progress bar
    m_iRet = DCS_SetLCDBrightness(iBrightLevel);
    if (DCS_SUCCESS != m_iRet)
    {
        return false;
    }
    m_iCurrentBright = iBrightLevel;

    return true;
}

/**************************************************************************************************
* Function Name : EnableAutoRotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::EnableAutoRotation(BOOL* pbStatus)
{
    if (( NULL == pbStatus ) || ( DCS_SUCCESS != m_iDCSRet ))
    {
        return false;
    }

    m_iRet = DCS_EnableDesktopAutoRotation();   //Calling the DCS API to enable auto rotation
    if ( DCS_SUCCESS != m_iRet )
    {
        *pbStatus = AUTOROTATION_DISABLED;//Return enabled(0): Manual rotation
        return false;
    }

    *pbStatus = AUTOROTATION_ENABLED;     //Return enabled(1): Auto rotation
    return true;
}

/**************************************************************************************************
* Function Name : DisableAutoRotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::DisableAutoRotation(BOOL* pbStatus)
{
    if (( NULL == pbStatus ) || (DCS_SUCCESS != m_iDCSRet))
    {
        return false;
    }

    m_iRet = DCS_DisableDesktopAutoRotation();//Disable auto rotation
    if ( DCS_SUCCESS != m_iRet )
    {
        *pbStatus = AUTOROTATION_ENABLED;
        return false;
    }

    *pbStatus = AUTOROTATION_DISABLED;    //Return disabled(0):Manual rotation
    return true;
}

/**************************************************************************************************
* Function Name : GetAutoRotationStatus                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CDCSBrightRotation::GetAutoRotationStatus(BOOL* pEnabled)
{
    if (( NULL == pEnabled ) || ( DCS_SUCCESS != m_iDCSRet ))
    {
        return false;
    }

    //Getting current status of auto rotation
    m_iRet = DCS_GetDesktopAutoRotationStatus(pEnabled);
    if ( DCS_SUCCESS != m_iRet )
    {
        *pEnabled = AUTOROTATION_DISABLED;//Return disabled(0):Manual rotation
        return false;
    }

    return true;
}
