#
# This file is part of Python Download Manager
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Author: Kenneth Christiansen <kenneth.christiansen@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import dbus
import logging
import cPickle
import urllib2
import tempfile
from string import digits, letters

from constants import Session, DownloadState, ExceptionType

log = logging.getLogger("downloadmanager.client")

def add_ranstr(basename):
    if "." in basename:
        basename, ext = basename.rsplit(".", 1)
        return tempfile.mktemp('.' + ext,  basename + '-', '')

    return tempfile.mktemp('', basename + '-', '')


class Info(dict):
    def __init__(self, uri, filename):
        self['uri'] = uri
        self['filename'] = filename
        self['state'] = None

        # update data if info file exists on disk
        self.load()

    def __getattr__(self, attr):
        try:
            return self[attr]
        except KeyError:
            raise AttributeError(attr)

    def __setattr__(self, attr, value):
        self[attr] = value
        self.save()

    def load(self):
        path = self.filename + ".info"
        if os.path.exists(path):
            info = cPickle.load(open(path, "rb"))
            self.update(info)

    def save(self):
        info = {}
        info.update(self)
        path = self.filename + ".info"
        try:
            cPickle.dump(info, open(path, "wb"),
                         cPickle.HIGHEST_PROTOCOL)
        except IOError, e:
            log.error("Couldn't save infofile: %s" % e)

    def delete(self):
        try:
            os.unlink(self.filename + ".info")
        except Exception, e:
            log.error("Couldn't unlink infofile: %s" % e)


class AsyncDownloader(object):
    def __init__(self, mger, obj_path, source, target, data=None):
        self.manager = mger
        self.obj_path = obj_path

        self._cancelled_callbacks = []
        self._download_callbacks = []
        self._filtered_callbacks = []
        self._queued_callbacks = []
        self._paused_callbacks = []
        self._end_callbacks = []

        self.state_dispatcher = {
            DownloadState.NOT_STARTED: [],
            DownloadState.CONNECTING: [],
            DownloadState.IN_PROGRESS: self._download_callbacks,
            DownloadState.PAUSED: self._paused_callbacks,
            DownloadState.CANCELLED: self._cancelled_callbacks,
            DownloadState.QUEUED: self._queued_callbacks,
            DownloadState.FILTERED: self._filtered_callbacks,
            DownloadState.COMPLETED: self._end_callbacks,
            DownloadState.EXCEPTION: self._end_callbacks
        }

        self.source = source
        self.target = target
        self.active = False
        self.total_size = -1

        self.data = data

        self.force = False

        self._init_session()
        self._state = self._session.GetState()

    def __get_state(self):
        return self._state

    def __set_state(self, value):
        self._state = value
        self._session.SetState(value)

    state = property(__get_state, __set_state)

    def _call_if_state_match(self, cb, match_state):
        if self.state == match_state:
            args = self._handle_state(match_state)
            cb(*args)

    def on_finished_add(self, cb):
        self._call_if_state_match(cb, DownloadState.COMPLETED)
        self._end_callbacks.insert(0, cb)

    def on_finished_remove(self, cb):
        self._end_callbacks.remove(cb)

    def on_download_started_add(self, cb):
        self._call_if_state_match(cb, DownloadState.IN_PROGRESS)
        self._download_callbacks.insert(0, cb)

    def on_download_started_remove(self, cb):
        self._download_callbacks.remove(cb)

    def on_cancelled_add(self, cb):
        self._call_if_state_match(cb, DownloadState.CANCELLED)
        self._cancelled_callbacks.insert(0, cb)

    def on_cancelled_remove(self, cb):
        self._cancelled_callbacks.remove(cb)

    def on_paused_add(self, cb):
        self._call_if_state_match(cb, DownloadState.PAUSED)
        self._paused_callbacks.insert(0, cb)

    def on_paused_remove(self, cb):
        self._paused_callbacks.remove(cb)

    def on_filtered_add(self, cb):
        self._call_if_state_match(cb, DownloadState.FILTERED)
        self._filtered_callbacks.insert(0, cb)

    def on_filtered_remove(self, cb):
        self._filtered_callbacks.remove(cb)

    def on_queued_add(self, cb):
        self._call_if_state_match(cb, DownloadState.QUEUED)
        self._queued_callbacks.insert(0, cb)

    def on_queued_remove(self, cb):
        self._queued_callbacks.remove(cb)

    def _init_session(self):
        self.session_bus_name = self.manager._bus_object.GetNameOwner(
                                    Session.DBUS_SERVICE_NAME,
                                    dbus_interface='org.freedesktop.DBus')
        obj = self.manager._bus.get_object(self.session_bus_name,
                                   self.obj_path, introspect=False)
        self._session = dbus.Interface(obj, Session.DBUS_IFACE)

        self._state_changed_callbacks = []
        self._sig_match = self._session.connect_to_signal("StateChanged",
                                                          self.__state_changed)

    def _disconnect_dbus_callbacks(self):
        if self._sig_match:
            self._sig_match.remove()
        else:
            # for backwards compatible dbus-python version 0.71
            try:
                self.manager._bus.remove_signal_receiver(None,
                                                 "StateChanged",
                                                 Session.DBUS_IFACE,
                                                 self.session_bus_name,
                                                 self.obj_path)
            except dbus.DBusException, e:
                # dbus-python version 0.71 has a bug,
                # always raising exception in signal removing
                pass
        self._state_changed_callbacks = []

    def _wrap_exception(self):
        try:
            etype, errno, emsg = self.get_exception_info()
        except TypeError:
            return None # No exception was thrown

        if etype == ExceptionType.HTTP:
            e = urllib2.HTTPError(self.source, errno, emsg, None, None)
        elif etype == ExceptionType.URL:
            if errno != -1:
                arg = (errno, emsg)
            else:
                arg = emsg
            e = urllib2.URLError(arg)
        elif etype == ExceptionType.IO:
            msg = "[Errorno %s] %s" % (errno, emsg)
            e = IOError(msg)
            e.errno = errno
        elif etype == ExceptionType.UNKNOWN:
            msg = "Unknown error: %s" % emsg
            e = Exception(msg)
        else:
            log.error("Unknown exception type thrown %d" % etype)
            return None
        return e

    def _handle_state(self, state):
        args = ()
        if state in [DownloadState.COMPLETED, DownloadState.EXCEPTION]:
            args = (self._wrap_exception(), self.get_mime_type())
            self._disconnect_dbus_callbacks()
        elif state in [DownloadState.FILTERED]:
            args = (self.get_mime_type(), )
        elif state in [DownloadState.CANCELLED]:
            self._disconnect_dbus_callbacks()

        return args

    def __state_changed(self, state):
        self._state = state
        # make copy as it might get cleaned when we remove
        # the dbus callbacks with {disconnect_dbus_callbacks},
        # but we still want to call a last time, and only
        # after our handling.
        callbacks = self._state_changed_callbacks

        args = self._handle_state(state)
        cbs = self.state_dispatcher[state]
        for cb in cbs:
            cb(*args)

        for cb in callbacks:
            cb(self, state)

    def _on_state_changed_add(self, cb):
        self._state_changed_callbacks.insert(0, cb)
        state = self._session.GetState()
        cb(self, state)

    def get_progress(self):
        if self.total_size is -1:
            self.total_size = int(self._session.GetTotalSize())

        try:
            size = os.path.getsize(self.target)
        except:
            size = 0

        return size, self.total_size

    def start(self, resume=False, compress=False):
        if self.state not in [DownloadState.NOT_STARTED,
                              DownloadState.PAUSED,
                              DownloadState.QUEUED]:
            log.debug("Trying to start a process already started (%s)" % self.source)
            return

        if self.force:
            self._session.Start(resume, compress)
            return

        if not self.manager.limit_reached():
            self.manager.set_active(self)
            self._session.Start(resume, compress)
        else:
            self.manager.set_inactive(self)
            self.manager._queue.append(self)
            self._resume = resume
            self._session.MarkAsQueued()

    def pause(self):
        self._session.Pause()

    def cancel(self):
        self.manager._remove_from_manager(self)
        self._session.Cancel()

    def get_mime_type(self):
        res = self._session.GetMimeType()
        if res is not None:
            return str(res)
        else:
            return res

    def get_exception_info(self):
        return self._session.GetExceptionInfo()

    def set_accepted_mimetypes(self, accept_list):
        if accept_list is not None:
            self._session.SetAcceptedMimeTypes(accept_list)


class DownloadManager(object):
    DBUS_SERVICE_NAME = 'br.org.indt.DownloadManager'
    DBUS_OBJ_PATH = '/br/org/indt/DownloadManager'
    DBUS_INTERFACE_NAME = 'br.org.indt.DownloadManager'

    def __init__(self, limit=3):
        self._limit = limit
        self.items = {}        # maps URI to AsyncDownloader
        self._queue = []       # based on URI
        self.__active_items = 0
        self._item_added_callbacks = []
        self._item_removed_callbacks = []
        self._init_dbus()
        self.default_dir = "/tmp"

    def __get_active_items(self):
        return self.__active_items

    def __set_active_items(self, value):
        self.__active_items = value

        if value < self._limit and len(self._queue) > 0:
            item = self._queue.pop()
            item.start(item._resume)

    _active_items = property(__get_active_items, __set_active_items)

    def _init_dbus(self):
        self._bus = dbus.SessionBus()
        # to resolve unique bus name
        self._bus_object = self._bus.get_object('org.freedesktop.DBus',
                                                '/org/freedesktop/DBus',
                                                introspect=False)
        obj = self._bus.get_object(self.DBUS_SERVICE_NAME, self.DBUS_OBJ_PATH,
                                  introspect=False)
        self._download_mger = dbus.Interface(obj, self.DBUS_INTERFACE_NAME)

    def on_item_added_add(self, cb):
        self._item_added_callbacks.insert(0, cb)

    def on_item_added_remove(self, cb):
        self._item_added_callbacks.remove(cb)

    def on_item_removed_add(self, cb):
        self._item_removed_callbacks.insert(0, cb)

    def on_item_removed_remove(self, cb):
        self._item_removed_callbacks.remove(cb)

    def limit_reached(self):
        if self._active_items >= self._limit:
            return True
        else:
            return False

    def set_active(self, item):
        if item.active or item.force:
            return
        item.active = True
        self._active_items += 1

    def set_inactive(self, item):
        if not item.active or item.force:
            return
        item.active = False
        self._active_items -= 1

    def _state_changed(self, item, state):
        # is set on all starts, resumes and auto. starts from queue
        if state == DownloadState.CONNECTING:
            self.set_active(item)

        # pause actually terminates the connection and we connect
        # again on resume
        if state == DownloadState.PAUSED:
            self.set_inactive(item)

        # all below states terminates the connection
        if state in [DownloadState.COMPLETED, DownloadState.CANCELLED,
                     DownloadState.FILTERED, DownloadState.EXCEPTION]:
            self._remove_from_manager(item)
            self._destroy(item)
            return

        item.info.state = state
        item.info.save()

    def _remove_from_manager(self, item):
        # if not in items, could have been removed earlier (cancel)
        # no need to inform that item was removed
        if not item.target in self.items:
            log.info("target not in items, maybe it was cancelled")
            return

        # if in queue, remove so it won't be waked up after it dies
        if item in self._queue:
            self._queue.remove(item)

        # this below line might start queued items
        self.set_inactive(item)

        group = item.group

        del self.items[item.target]

        for cb in self._item_removed_callbacks:
            cb(group)

    def _add_to_manager(self, item):
        if item.target in self.items:
            return

        self.items[item.target] = item
        item._on_state_changed_add(self._state_changed)

        group = item.group

        for cb in self._item_added_callbacks:
            cb(group)

    def _create(self, source, target, data):
        obj_path = self._download_mger.CreateSession(source, target)
        item = AsyncDownloader(self, obj_path, source, target, data)

        item.info = Info(source, target)

        return item

    def _destroy(self, item):
        # delete info file
        if item.info is not None:
            item.info.delete()
            item.info = None

        self._download_mger.DeleteSession(item.obj_path)

    def add(self, source, target=None, data=None, group=None, ignore_limit=False):
        if target is None:
            basename = os.path.basename(source)
            target = os.path.join(self.default_dir, add_ranstr(basename))

        assert target not in self.items

        item = self._create(source, target, data)

        item.group = group
        item.force = ignore_limit

        self._add_to_manager(item)

        self._apply_previous_state(item)

        return item

    def _apply_previous_state(state, item):
        state = item.info.state

        if state == DownloadState.IN_PROGRESS:
            item.start(True)
        else:
            item.state = state

    def get_info_for_target(self, target):
        item = self.items.get(target) or None

        if item is None:
            info = Info(None, target)
            if info.state is None:
                info = None
        else:
            info = item.info

        return info, item
