import sys
import threading
import time
import traceback

import gobject
import ecore

from dispatcher import *


def send_to_main_thread_async(comm_attr):
    return send_to_thread_async("main_thread", comm_attr)


def send_to_e_thread_async(comm_attr):
    return send_to_thread_async("e_thread", comm_attr)


def send_to_e_thread_sync(comm_attr):
    return send_to_thread_sync("e_thread", comm_attr)


def print_thread(msg):
    print "[%s]:" % threading.currentThread().getName(), msg
    sys.stdout.flush()


class EThread(threading.Thread):
    def __init__(self, comm, name):
        threading.Thread.__init__(self, name=name)
        self._wid = gobject.io_add_watch(comm.in_fd,
                                         gobject.IO_IN,
                                         self._run_main_thr_cbs)
        self.e_thread = 'EThread'
        self.comm = comm

    def _run_main_thr_cbs(self, source, condition):
        # not dealing with errors
        self.comm.run_cbs()
        return True

    def _run_e_thr_cbs(self, fd_handler):
        # not dealing with errors
        self.comm.run_cbs()
        return True

    def run(self):
        ecore.fd_handler_add(self.comm.in_fd,
                             ecore.ECORE_FD_READ,
                             self._run_e_thr_cbs)
        ecore.main_loop_begin()

    def _quit_e_thr(self):
        ecore.main_loop_quit()
        return False

    @send_to_e_thread_async("comm")
    def _stop_e_thr(self):
        ecore.idler_add(self._quit_e_thr)

    def stop(self):
        gobject.source_remove(self._wid)
        self._stop_e_thr()


class Test(object):
    name = "Test"

    def __init__(self, ml, comm):
        self.ml = ml
        self.main_thread = "MainThread"
        self.e_thread = "EThread"
        self.comm = comm

    def run(self):
        pass

    def start(self):
        self.run()

    @send_to_main_thread_async("comm")
    def stop(self):
        ml.quit()


class TestRR(Test):
    name = "TestRR"

    def __init__(self, ml, comm):
        Test.__init__(self, ml=ml, comm=comm)
        self.round = 20

    @send_to_main_thread_async("comm")
    def _in_main_thr(self):
        self.round -= 1
        print_thread("_in_main_thr(): round=%d" % self.round)

        if self.round > 0:
            self._in_e_thr("round=%d" % self.round)
        else:
            self.stop()

    @send_to_e_thread_async("comm")
    def _in_e_thr(self, msg):
        print_thread("_in_e_thr(): msg=[%s]" % msg)
        self._in_main_thr()

    def run(self):
        self._in_e_thr("round=%d" % self.round)


class TestSync(Test):
    name = "TestSync"

    @send_to_e_thread_sync("comm")
    def _in_e_thr_ok(self):
        print_thread("_in_e_thr_ok(): returning 'Ok'")
        return "Ok"

    @send_to_e_thread_sync("comm")
    def _in_e_thr_notok(self):
        print_thread("_in_e_thr_notok(): raising 'Not ok'")
        raise RuntimeError("Not ok")

    def run(self):
        ret = self._in_e_thr_ok()
        print_thread("_in_e_thr_ok() returned [%s]" % ret)

        try:
            ret = self._in_e_thr_notok()
        except Exception, e:
            print_thread("_in_e_thr_notok() raised [%s]" % e)
            traceback.print_exc()

        self.stop()


class TestAsync(Test):
    name = "TestAsync"

    @send_to_e_thread_async("comm")
    def _in_e_thr_ok(self):
        print_thread("_in_e_thr_ok(): returning 'Ok' after 3 s")
        time.sleep(3)
        return "Ok"

    @send_to_e_thread_async("comm")
    def _in_e_thr_notok(self):
        print_thread("_in_e_thr_notok(): raising 'Not ok' after 2 s")
        time.sleep(2)
        raise RuntimeError("Not ok")

    def _show_exception(self, exc_info):
        gobject.source_remove(self._timer_src)
        print_thread("got exception [%s] after %f s" %
                     (exc_info[1], time.time() - self._time_before))
        traceback.print_exception(*exc_info)
        self.stop()

    def _after_e_thr_notok(self, ret):
        print_thread("_in_e_thr_notok returned [%s] after %f s" %
                     (ret, time.time() - self._time_before))

    def _after_e_thr_ok(self, ret):
        print_thread("_in_e_thr_ok() returned [%s] after %f s" %
                     (ret, time.time() - self._time_before))

        self._time_before = time.time()
        self._in_e_thr_notok(err_handler=self._show_exception,
                             ret_handler=self._after_e_thr_notok)

    def _keep_alive(self):
        print_thread("more 500 msecs")
        return True

    def run(self):
        self._time_before = time.time()
        self._timer_src = gobject.timeout_add(500, self._keep_alive)
        ret = self._in_e_thr_ok(err_handler=self._show_exception,
                                ret_handler=self._after_e_thr_ok)


if __name__ == "__main__":
    gobject.threads_init()

    comm = CallbackPipe("MainThread", "EThread")
    ethr = EThread(comm=comm, name="EThread")
    ethr.start()

    tests = (TestRR, TestSync, TestAsync,)

    ml = gobject.MainLoop()
    for i, klass in enumerate(tests):
        print " ===== Starting test %d: %s =====" % (i, klass.name)

        test = klass(ml, comm)

        gobject.idle_add(test.start)
        ml.run()

        print " ===== Finished test %d: %s =====" % (i, klass.name)

    ethr.stop()
