/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <fcntl.h>

#include "pgmgleseglbackend.h"
#include "pgmglesviewport.h"

/*  */
typedef struct {
  EGLint       error;
  const gchar *string;
} EglError;

/*  */
static EglError egl_error_map[] = {
  { EGL_NOT_INITIALIZED,     "not initialized"     },
  { EGL_BAD_ACCESS,          "bad access"          },
  { EGL_BAD_ALLOC,           "bad allocation"      },
  { EGL_BAD_ATTRIBUTE,       "bad attribute"       },
  { EGL_BAD_CONFIG,          "bad configuration"   },
  { EGL_BAD_CONTEXT,         "bad context"         },
  { EGL_BAD_CURRENT_SURFACE, "bad current surface" },
  { EGL_BAD_DISPLAY,         "bad display"         },
  { EGL_BAD_MATCH,           "bad match"           },
  { EGL_BAD_NATIVE_PIXMAP,   "bad native pixmap"   },
  { EGL_BAD_NATIVE_WINDOW,   "bad native window"   },
  { EGL_BAD_PARAMETER,       "bad parameter"       },
  { EGL_BAD_SURFACE,         "bad surface"         },
  { 0,                       "unknown error"       }
};

#ifdef HAVE_TSLIB

/* Prototypes */
static gboolean event_prepare  (GSource *source,
                                gint *timeout);
static gboolean event_check    (GSource *source);
static gboolean event_dispatch (GSource *source,
                                GSourceFunc callback,
                                gpointer data);

/* Touch screen event handling functions */
static GSourceFuncs event_funcs = {
  event_prepare, event_check, event_dispatch, NULL
};

#endif /* HAVE_TSLIB */

GST_DEBUG_CATEGORY_STATIC (pgm_gles_eglbackend_debug);
#define GST_CAT_DEFAULT pgm_gles_eglbackend_debug

static PgmGlesBackendClass *parent_class = NULL;

/* Private functions */

/* */
static gboolean
get_egl_error (const gchar *function)
{
  guint i = 0;
  EGLint error;

  error = eglGetError ();

  if (error == EGL_SUCCESS)
    return TRUE;

  /* Search for the error */
  while (egl_error_map[i].error && error != egl_error_map[i].error)
    i++;

  /* Then log it */
  GST_ERROR ("%s failed: %s\n", function, egl_error_map[i].string);

  return FALSE;
}

#ifdef HAVE_TSLIB

/* Events preparing */
static gboolean
event_prepare (GSource *source,
               gint *timeout)
{
  PgmGlesEglBackendSource *eglbackend_source =
    (PgmGlesEglBackendSource *) source;
  PgmGlesBackend *backend = PGM_GLES_BACKEND (eglbackend_source->eglbackend);
  gint requested_fps = backend->context->requested_fps;

  /* If there are rendering updates */
  if (backend->context->update_tag)
    {
      /* If the rendering is blocked on the vblank, we give a timeout of 0 */
      if (requested_fps == 0)
        *timeout = 0;
      /* If the rendering is done in a timeout, we use the same timeout */
      else
        *timeout = requested_fps;
    }
  /* If there are no rendering updates, we poll with no timeout */
  else
    *timeout = -1;

  return FALSE;
}

/* Events checking */
static gboolean
event_check (GSource *source)
{
  PgmGlesEglBackendSource *eglbackend_source =
    (PgmGlesEglBackendSource *) source;

  return (eglbackend_source->poll_fd.revents & G_IO_IN) != 0;
}

/* Events dispatching */
static gboolean
event_dispatch (GSource *source,
                GSourceFunc callback,
                gpointer data)
{
  PgmGlesEglBackendSource *eglbackend_source =
    (PgmGlesEglBackendSource*) source;
  PgmGlesBackend *backend = PGM_GLES_BACKEND (eglbackend_source->eglbackend);
  PgmGlesViewport *glesviewport = backend->context->glesviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glesviewport);
  static gboolean clicked = FALSE;
  static gint last_x = -1;
  static gint last_y = -1;
  struct ts_sample sample;

  /* Get the event */
  while (ts_read (eglbackend_source->ts_device, &sample, 1) > 0)
    {
      GST_DEBUG_OBJECT (eglbackend_source->eglbackend, "touchscreen event: "
                        "(time=%ld.%06ld, x=%d y=%d pressure=%d)",
                        sample.tv.tv_sec, sample.tv.tv_usec, sample.x,
                        sample.y, sample.pressure);

      /* Build the event */
      if (sample.pressure)
        {
          /* Press */
          if (!clicked)
            {
              PgmEventButton *pgmevent;
              pgmevent = (PgmEventButton*) pgm_event_new (PGM_BUTTON_PRESS);
              pgmevent->time = sample.tv.tv_sec * 1000
                + sample.tv.tv_usec * 0.001f;
              pgmevent->pressure = sample.pressure;
              pgmevent->button = PGM_BUTTON_LEFT;
              pgmevent->x = last_x = sample.x;
              pgmevent->y = last_y = sample.y;
              clicked = TRUE;
              pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
            }
          /* Pressure */
          else if (last_x == sample.x && last_y == sample.y)
            {
              PgmEventButton *pgmevent;
              pgmevent = (PgmEventButton*) pgm_event_new (PGM_BUTTON_PRESSURE);
              pgmevent->time = sample.tv.tv_sec * 1000
                + sample.tv.tv_usec * 0.001f;
              pgmevent->pressure = sample.pressure;
              pgmevent->button = PGM_BUTTON_LEFT;
              pgmevent->x = last_x = sample.x;
              pgmevent->y = last_y = sample.y;
              pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
            }
          /* Motion */
          else
            {
              PgmEventMotion *pgmevent;
              pgmevent = (PgmEventMotion*) pgm_event_new (PGM_MOTION_NOTIFY);
              pgmevent->time = sample.tv.tv_sec * 1000
                + sample.tv.tv_usec * 0.001f;
              pgmevent->pressure = sample.pressure;
              pgmevent->x = last_x = sample.x;
              pgmevent->y = last_y = sample.y;
              pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
            }
        }
      else
        {
          PgmEventButton *pgmevent;
          pgmevent = (PgmEventButton*) pgm_event_new (PGM_BUTTON_RELEASE);
          pgmevent->time = sample.tv.tv_sec * 1000
            + sample.tv.tv_usec * 0.001f;
          pgmevent->pressure = sample.pressure;
          pgmevent->button = PGM_BUTTON_LEFT;
          pgmevent->x = sample.x;
          pgmevent->y = sample.y;
          clicked = FALSE;
          last_x = -1;
          last_y = -1;
          pgm_viewport_push_event (viewport, (PgmEvent*) pgmevent);
        }
    }

  return TRUE;
}

/* Add touch screen event handling source to the rendering main context */
static gboolean
add_event_source (PgmGlesEglBackend *eglbackend)
{
  PgmGlesBackend *glesbackend = PGM_GLES_BACKEND (eglbackend);
  GMainContext *render_context = glesbackend->context->render_context;
  PgmGlesEglBackendSource *source;
  const gchar *device_string;
  struct tsdev *ts_device;
  gint flags;
  gint fd;

  /* Get the device name in the environment, open and configure it */
  device_string = g_getenv ("TSLIB_TSDEVICE");
  ts_device = ts_open (device_string, 0);
  if (!ts_device)
    {
      GST_WARNING_OBJECT (eglbackend, "could not open touch screen device: %s",
                          device_string);
      return FALSE;
    }
  if (ts_config (ts_device))
    {
      GST_WARNING_OBJECT (eglbackend, "could not configure touch screen device:"
                          "%s", device_string);
      ts_close (ts_device);
      return FALSE;
    }
  GST_DEBUG_OBJECT (eglbackend, "configured touch screen device: %s",
                    device_string);

  /* The touch screen device file descriptor needs to be non-blocking so that
   * we can flush completely the received events. If we are not doing that, the
   * release events are not sent when the stylus releases the touchscreen but
   * at the next pressure. */
  fd = ts_fd (ts_device);
  flags = fcntl (fd, F_GETFL, 0);
  fcntl (fd, F_SETFL, flags | O_NONBLOCK);

  /* Create and initialize the touch screen event handling dedicated source */
  source = (PgmGlesEglBackendSource *)
    g_source_new (&event_funcs, sizeof (PgmGlesEglBackendSource));
  source->poll_fd.fd = fd;
  source->poll_fd.events = G_IO_IN;
  source->ts_device = ts_device;
  source->eglbackend = eglbackend;
  g_source_add_poll ((GSource *) source, &source->poll_fd);

  /* Attach it */
  g_source_set_can_recurse ((GSource *) source, TRUE);
  g_source_set_priority ((GSource *) source, G_PRIORITY_DEFAULT-20);
  eglbackend->event_tag = g_source_attach ((GSource *) source, render_context);

  return TRUE;
}

/* Remove touch screen event handling source from the rendering main context */
static void
remove_event_source (PgmGlesEglBackend *eglbackend)
{
  PgmGlesBackend *glesbackend = PGM_GLES_BACKEND (eglbackend);
  GMainContext *render_context = glesbackend->context->render_context;
  GSource *source;

  /* Get the source from the rendering GMainContext */
  source = g_main_context_find_source_by_id (render_context,
                                             eglbackend->event_tag);

  /* And remove it */
  if (source)
    {
      ts_close (((PgmGlesEglBackendSource*) source)->ts_device);
      g_source_destroy (source);
      g_source_unref ((GSource *) source);
    }
}

#endif /* HAVE_TSLIB */

/* PgmGlesBackend methods */

static gboolean
pgm_gles_egl_backend_create_window (PgmGlesBackend *glesbackend)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);
  gint swap_interval = 1;
  const gchar *env_var;
  EGLConfig config;
  gint num_configs;

  gint attrib[] = {
    EGL_RED_SIZE, 5,
    EGL_GREEN_SIZE, 6,
    EGL_BLUE_SIZE, 5,
    EGL_DEPTH_SIZE, 0,
    EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
    EGL_NONE };

  g_return_val_if_fail (eglbackend != NULL, FALSE);

  GST_DEBUG_OBJECT (eglbackend, "create_window");

  /* Initialize EGL for the default display */
  eglbackend->display = eglGetDisplay ((NativeDisplayType) EGL_DEFAULT_DISPLAY);
  if (eglbackend->display == EGL_NO_DISPLAY)
    {
      GST_ERROR_OBJECT (eglbackend, "cannot get a display");
      return FALSE;
    }
  if (!eglInitialize (eglbackend->display, NULL, NULL))
    {
      get_egl_error ("eglInitialize");
      return FALSE;
    }

  /* Get and log EGL informations */
  eglbackend->vendor = eglQueryString (eglbackend->display, EGL_VENDOR);
  eglbackend->version = eglQueryString (eglbackend->display, EGL_VERSION);
  eglbackend->extensions = eglQueryString (eglbackend->display, EGL_EXTENSIONS);

  GST_DEBUG_OBJECT (eglbackend, "EGL vendor: %s", eglbackend->vendor);
  GST_DEBUG_OBJECT (eglbackend, "EGL version: %s", eglbackend->version);
  GST_DEBUG_OBJECT (eglbackend, "EGL extensions: %s", eglbackend->extensions);

  if (!eglChooseConfig (eglbackend->display, attrib, &config, 1, &num_configs))
    {
      get_egl_error ("eglCreateWindowSurface");
      return FALSE;
    }

  /* Create the context and make it current */
  eglbackend->surface = eglCreateWindowSurface (eglbackend->display, config,
                                                (NativeWindowType) 0, NULL);
  if (!get_egl_error ("eglCreateWindowSurface"))
    return FALSE;
  eglbackend->context = eglCreateContext (eglbackend->display, config,
                                          NULL, NULL);
  if (!get_egl_error ("eglCreateContext"))
    return FALSE;
  eglMakeCurrent (eglbackend->display, eglbackend->surface, eglbackend->surface,
                  eglbackend->context);
  if (!get_egl_error ("eglMakeCurrent"))
    return FALSE;

#ifdef HAVE_TSLIB
  /* Touch screen event source */
  if (!add_event_source (eglbackend))
    {
      GST_ERROR_OBJECT (eglbackend, "couldn't add event handling source\n");
      eglMakeCurrent (eglbackend->display, EGL_NO_SURFACE, EGL_NO_SURFACE,
                      EGL_NO_CONTEXT);
      eglTerminate (eglbackend->display);
      return FALSE;
    }
#endif /* HAVE_TSLIB */

  /* Get and store the window size, the resolution in this case is the same */
  eglQuerySurface (eglbackend->display, eglbackend->surface, EGL_WIDTH,
                   &eglbackend->size_mm_width);
  eglbackend->resolution_width = eglbackend->size_mm_width;
  eglQuerySurface (eglbackend->display, eglbackend->surface, EGL_HEIGHT,
                   &eglbackend->size_mm_height);
  eglbackend->resolution_height = eglbackend->size_mm_height;

  /* Specify the minimum number of video frame periods per buffer swap, this
   * is actually how vertical synchronization is handled */
  env_var = g_getenv ("PGM_GLES_SWAP_INTERVAL");
  if (env_var)
    swap_interval = atoi (env_var);
  eglSwapInterval (eglbackend->display, swap_interval);
  GST_DEBUG_OBJECT (eglbackend, "using a swap interval of %d", swap_interval);

  eglbackend->created = TRUE;

  return TRUE;
}

static gboolean
pgm_gles_egl_backend_destroy_window (PgmGlesBackend *glesbackend)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_DEBUG_OBJECT (eglbackend, "destroy_window");

  if (eglbackend->created)
    {
#ifdef HAVE_TSLIB
      remove_event_source (eglbackend);
#endif /* HAVE_TSLIB */

      eglMakeCurrent (eglbackend->display, EGL_NO_SURFACE, EGL_NO_SURFACE,
                      EGL_NO_CONTEXT);
      eglTerminate (eglbackend->display);
      eglbackend->created = FALSE;
    }

  return TRUE;
}

static gboolean
pgm_gles_egl_backend_set_visibility (PgmGlesBackend *glesbackend,
                                     gboolean visible)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_DEBUG_OBJECT (eglbackend, "set_visibility");

  /* FIXME */

  return TRUE;
}

static void
pgm_gles_egl_backend_swap_buffers (PgmGlesBackend *glesbackend)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_LOG_OBJECT (eglbackend, "swap_buffers");

  eglSwapBuffers (eglbackend->display, eglbackend->surface);
}

static gpointer
pgm_gles_egl_backend_get_proc_address (PgmGlesBackend *glesbackend,
                                       const gchar *proc_name)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_LOG_OBJECT (eglbackend, "get_proc_address");

  return eglGetProcAddress (proc_name);
}

static void
pgm_gles_egl_backend_get_screen_size_mm (PgmGlesBackend *glesbackend,
                                         gint *width,
                                         gint *height)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_LOG_OBJECT (eglbackend, "get_screen_size_mm");

  *width = eglbackend->size_mm_width;
  *height = eglbackend->size_mm_height;
}

static void
pgm_gles_egl_backend_get_screen_resolution (PgmGlesBackend *glesbackend,
                                            gint *width,
                                            gint *height)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (glesbackend);

  GST_LOG_OBJECT (eglbackend, "get_screen_resolution");

  *width = eglbackend->resolution_width;
  *height = eglbackend->resolution_height;
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlesEglBackend, pgm_gles_egl_backend,
                         PGM_TYPE_GLES_BACKEND);

void
pgm_gles_egl_backend_register (GTypeModule *module)
{
  pgm_gles_egl_backend_register_type (module);
}

static void
pgm_gles_egl_backend_dispose (GObject *object)
{
  PgmGlesEglBackend *eglbackend = PGM_GLES_EGL_BACKEND (object);

  GST_DEBUG_OBJECT (eglbackend, "dispose");

  if (eglbackend->created)
    pgm_gles_egl_backend_destroy_window (PGM_GLES_BACKEND (eglbackend));

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gles_egl_backend_class_init (PgmGlesEglBackendClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlesBackendClass *glesbackend_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gles_eglbackend_debug, "pgm_gles_eglbackend", 0,
                           "OpenGL ES plugin: PgmGlesEglBackend");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  glesbackend_class = PGM_GLES_BACKEND_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_dispose);

  /* PgmGlesBackend virtual table */
  glesbackend_class->create_window =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_create_window);
  glesbackend_class->destroy_window =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_destroy_window);
  glesbackend_class->swap_buffers =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_swap_buffers);
  glesbackend_class->get_proc_address =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_get_proc_address);
  glesbackend_class->set_visibility =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_set_visibility);
  glesbackend_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_get_screen_size_mm);
  glesbackend_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_gles_egl_backend_get_screen_resolution);
}

static void
pgm_gles_egl_backend_class_finalize (PgmGlesEglBackendClass *klass)
{
  return;
}

static void
pgm_gles_egl_backend_init (PgmGlesEglBackend *eglbackend)
{
  GST_DEBUG_OBJECT (eglbackend, "init");

  eglbackend->display = 0;
  eglbackend->surface = 0;
  eglbackend->context = 0;
  eglbackend->created = FALSE;
}

/* Public methods */

PgmGlesBackend*
pgm_gles_egl_backend_new (PgmGlesContext *glescontext)
{
  PgmGlesBackend *glesbackend;

  glesbackend = g_object_new (PGM_TYPE_GLES_EGL_BACKEND, NULL);
  GST_DEBUG_OBJECT (PGM_GLES_EGL_BACKEND (glesbackend),
                    "created new eglbackend");

  glesbackend->context = glescontext;

  return glesbackend;
}
