
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist_menu.c 1541 2006-11-13 16:25:16Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "disc.h"
#include "download.h"
#include "environment.h"
#include "filelist.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "oxine.h"
#include "playlist.h"
#include "playback_menu.h"
#include "playlist_menu.h"
#include "playlist_m3u.h"

extern oxine_t *oxine;

static l_list_t *prev_filelist = NULL;
static l_list_t *next_filelist = NULL;

static filelist_t *current_filelist = NULL;
static filelist_t *toplevel_filelist = NULL;

static otk_widget_t *button_up = NULL;
static otk_widget_t *button_prev = NULL;
static otk_widget_t *button_next = NULL;
static otk_widget_t *button_home = NULL;

#define FILELIST_BUTTONS_NUM 1
static otk_widget_t *filelist_menu_buttons[FILELIST_BUTTONS_NUM];
static otk_widget_t *filelist_menu_list = NULL;
static otk_widget_t *filelist_menu_window = NULL;
static otk_widget_t *filelist_menu_title = NULL;

static char *current_thumbnail = NULL;
static odk_osd_image_t *filelist_menu_thumbnail = NULL;

/*
 * ***************************************************************************
 * Some prototypes of methods declared in this file
 * ***************************************************************************
 */
static void filelist_menu_update_list (void);
static void filelist_menu_update_buttons (void);
static void filelist_menu_update_thumbnail (const char *thumbnail_mrl);

static void filelist_menu_set_current_filelist (filelist_t * newlist);

static void filelist_focus_enter_cb (void *entry_cb_data);
static void filelist_focus_leave_cb (void *entry_cb_data);

static void filelist_select_cb (void *entry_cb_data);
static void filelist_activate_cb (void *entry_cb_data);
static void filelist_remove_cb (void *entry_cb_data);

static void playlist_save_cb (void *oxine_p);
static playitem_t *playlist_get_selected (void);
static fileitem_t *subtitle_get_selected (void);

static void show_subtitle_menu_cb (void *oxine_p);

/*
 * ***************************************************************************
 * Methods that are part of the filelist GUI.
 * ***************************************************************************
 */
static void
filelist_menu_update_buttons (void)
{
    bool enabled;
    bool has_prev = (l_list_length (prev_filelist) > 0);
    bool has_next = (l_list_length (next_filelist) > 0);
    bool is_toplevel = (current_filelist != toplevel_filelist);

    otk_widget_set_enabled (button_up, is_toplevel);
    otk_widget_set_enabled (button_home, is_toplevel);
    otk_widget_set_enabled (button_prev, has_prev);
    otk_widget_set_enabled (button_next, has_next);

    fileitem_t *selected = subtitle_get_selected ();
    enabled = (selected && is_file_allowed (selected->mrl,
                                            ALLOW_FILES_SUBTITLE));
    otk_widget_set_enabled (filelist_menu_buttons[0], enabled);
}


static void
filelist_menu_addto_list (fileitem_t * item)
{
    otk_widget_t *w;
    w = otk_listentry_new (filelist_menu_list, item->title,
                           filelist_activate_cb, item,
                           filelist_select_cb, item,
                           filelist_remove_cb, item);
    otk_widget_set_focus_callbacks (w, filelist_focus_enter_cb, item,
                                    filelist_focus_leave_cb, item);
}


static void
filelist_menu_update_list (void)
{
    if (!filelist_menu_list)
        return;

    /* If we're currently not in the toplevel filelist and the filelist does
     * NOT have a parent we're probably in a removable disc, that has been
     * removed (see disc.c). To rescue ourself of segmentation faults we jump
     * to the toplevel filelist and clear the prev and next lists. */
    if ((current_filelist != toplevel_filelist)
        && (current_filelist->parent == NULL)) {
        filelist_menu_set_current_filelist (toplevel_filelist);
        l_list_clear (prev_filelist, NULL);
        l_list_clear (next_filelist, NULL);
    }

    otk_clear_list (filelist_menu_list);

    mutex_lock (&current_filelist->mutex);
#ifdef HAVE_DISC_POLLING
    mutex_lock (&oxine->removable_discs->mutex);
#endif

    /* next we add all the entries from the current filelist */
    {
        fileitem_t *item = filelist_first (current_filelist);
        while (item) {
            filelist_menu_addto_list (item);
            item = filelist_next (current_filelist, item);
        }
    }

#ifdef HAVE_DISC_POLLING
    /* last we add all removable drives currently available */
    if (current_filelist == toplevel_filelist) {
        fileitem_t *item = filelist_first (oxine->removable_discs);
        while (item) {
            filelist_menu_addto_list (item);
            item = filelist_next (oxine->removable_discs, item);
        }
    }
#endif

#ifdef HAVE_DISC_POLLING
    mutex_unlock (&oxine->removable_discs->mutex);
#endif
    mutex_unlock (&current_filelist->mutex);

    otk_list_set_pos (filelist_menu_list, current_filelist->top_position);
    otk_list_set_focus (filelist_menu_list, current_filelist->cur_position);
}


static void
filelist_menu_update_thumbnail (const char *thumbnail_mrl)
{
    if (filelist_menu_thumbnail)
        odk_osd_hide_image (filelist_menu_thumbnail);
    filelist_menu_thumbnail = NULL;

    if (thumbnail_mrl) {
        char *mrl = NULL;

        if (is_downloadable (thumbnail_mrl))
            mrl = download_to_cache (thumbnail_mrl, NULL, 0);
        else
            mrl = ho_strdup (thumbnail_mrl);

        if (mrl && (access (mrl, R_OK) == 0)) {
            filelist_menu_thumbnail = odk_osd_show_image (oxine->odk, mrl,
                                                          110, 380, 180, 260,
                                                          ODK_ALIGN_CENTER |
                                                          ODK_ALIGN_VCENTER,
                                                          true, 0x000000);
        }

        ho_free (mrl);
    }
}


static void
filelist_menu_set_current_filelist (filelist_t * newlist)
{
    filelist_t *prev = (filelist_t *) l_list_last (prev_filelist);
    filelist_t *next = (filelist_t *) l_list_last (next_filelist);

    /* We are returning to the directory on top of the prev stack. We 
     * remove it from the prev stack and add the current directory to the 
     * next stack. */
    if (newlist == prev) {
        l_list_remove (prev_filelist, newlist);
        l_list_append (next_filelist, current_filelist);
    }

    /* We are returning to the directory on top of the next stack. We 
     * remove it from the next stack and add the current directory to the 
     * prev stack. */
    else if (newlist == next) {
        l_list_remove (next_filelist, newlist);
        l_list_append (prev_filelist, current_filelist);
    }

    /* We are entering a directory that is neither on top of the prev stack
     * nor on top of the next stack. We clear the next stack and add the
     * current directory to the prev stack. */
    else {
        l_list_clear (next_filelist, NULL);
        l_list_append (prev_filelist, current_filelist);
    }

    current_filelist->top_position = otk_list_get_pos (filelist_menu_list);
    current_filelist->cur_position = otk_list_get_focus (filelist_menu_list);
    filelist_ref_set (&current_filelist, newlist);

    /* If we enter a directory that contains a thumbnail image we display
     * this. */
    if (current_thumbnail)
        ho_free (current_thumbnail);
    current_thumbnail = get_thumbnail (newlist->mrl);

    if (filelist_menu_title) {
        if (current_filelist == toplevel_filelist)
            otk_label_set_text (filelist_menu_title, _("Choose an entry..."));
        else {
            char *tmp = ho_strdup (current_filelist->title);
            char *title = tmp;

            if (title[0] == '[')
                title++;
            if (title[strlen (title) - 1] == ']')
                title[strlen (title) - 1] = '\0';
            otk_label_set_text (filelist_menu_title, title);

            ho_free (tmp);
        }
    }

    filelist_menu_update_list ();
    filelist_menu_update_buttons ();
    show_user_interface (oxine);
}


static void
filelist_remove_cb (void *entry_cb_data)
{
}


/*
 * This is the callback for a click on an item in the list.
 */
static void
filelist_select_cb (void *entry_cb_data)
{
    filelist_menu_update_buttons ();
    show_user_interface (oxine);
}


/*
 * This is the callback for a doubleclick on an item in the list.
 */
static void
filelist_activate_cb (void *entry_cb_data)
{
    playitem_t *playitem = playlist_get_selected ();
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    if ((fileitem->type == FILE_TYPE_MOUNTPOINT)
        && !disc_mount (fileitem->mrl)) {
        show_message_dialog (backto_menu_cb, oxine, NULL, NULL, DIALOG_OK,
                             NULL, _("Could not open disc!"));
        return;
    }

    filelist_expand (fileitem);

    if (fileitem->sublist) {
        filelist_menu_set_current_filelist (fileitem->sublist);
    }

    if (fileitem->type == FILE_TYPE_REGULAR) {
        if (is_file_allowed (fileitem->mrl, ALLOW_FILES_SUBTITLE)) {
            ho_free (playitem->subtitle_mrl);
            playitem->subtitle_mrl = ho_strdup (fileitem->mrl);
            show_playlist_menu_cb (oxine);
        }
    }

    filelist_menu_update_buttons ();
}


static void
filelist_focus_enter_cb (void *entry_cb_data)
{
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    if (!fileitem)
        return;

    /* If the fileitem is a directory this would point to the standard thumbnail file. */
    char *thumbnail_mrl = get_thumbnail (fileitem->mrl);

    /* If the fileitem has its own thumbnail we show it. */
    if (fileitem->thumbnail_mrl) {
        filelist_menu_update_thumbnail (fileitem->thumbnail_mrl);
    }

    /* Else we show the thumbnail of the directory this fileitem points to. */
    else if (thumbnail_mrl) {
        filelist_menu_update_thumbnail (thumbnail_mrl);
    }

    /* Last we try to show the thumbnail of the current directory. */
    else {
        filelist_menu_update_thumbnail (current_thumbnail);
    }

    if (thumbnail_mrl)
        ho_free (thumbnail_mrl);
}


static void
filelist_focus_leave_cb (void *entry_cb_data)
{
    /* If available we show the thumbnail of the current directory. */
    filelist_menu_update_thumbnail (current_thumbnail);
}


static void
filelist_up_cb (void *oxine_p)
{
    if (current_filelist) {
#ifdef HAVE_DISC_POLLING
        if (current_filelist->parent == oxine->removable_discs) {
            filelist_menu_set_current_filelist (toplevel_filelist);
        } else
#endif
        {
            filelist_menu_set_current_filelist (current_filelist->parent);
        }
    }
}


static void
filelist_home_cb (void *oxine_p)
{
    if (current_filelist != toplevel_filelist) {
        filelist_menu_set_current_filelist (toplevel_filelist);
    }
}


static void
filelist_prev_cb (void *oxine_p)
{
    filelist_t *last = (filelist_t *) l_list_last (prev_filelist);
    if (last) {
        filelist_menu_set_current_filelist (last);
    }
}


static void
filelist_next_cb (void *oxine_p)
{
    filelist_t *last = (filelist_t *) l_list_last (next_filelist);
    if (last) {
        filelist_menu_set_current_filelist (last);
    }
}


static void
filelist_menu_show_list (void)
{
    int x = 780 - 4 * 46 + 6;
    int y = 100;

    filelist_menu_title =
        otk_label_new (oxine->otk, 220, 117, x - 240,
                       OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER,
                       _("Choose an entry..."));
    otk_widget_set_font (filelist_menu_title, "sans", 32);

    filelist_menu_list =
        otk_list_new (oxine->otk, 220, 140, 560, 440,
                      30, 30, true, true,
                      OTK_LIST_SINGLE_SELECTION, playlist_get_selected ());

    button_prev =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_ARROW_LEFT),
                               filelist_prev_cb, oxine);
    x += 46;
    button_next =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_ARROW_RIGHT),
                               filelist_next_cb, oxine);
    x += 46;
    button_up =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_ARROW_UP),
                               filelist_up_cb, oxine);
    x += 46;
    button_home =
        otk_bitmap_button_new (oxine->otk, x, y, 40, 35,
                               odk_get_bitmap (BITMAP_HOME),
                               filelist_home_cb, oxine);
}


static void
filelist_menu_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (oxine->current_menu != show_subtitle_menu_cb)
        return;

    if (!(event->type == OXINE_EVENT_KEY))
        return;

    switch (event->source.key) {
    case OXINE_KEY_HOME:
        filelist_home_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_BACK:
        if (current_filelist != toplevel_filelist)
            filelist_up_cb (oxine);
        else
            show_playlist_menu_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_PREV:
        filelist_prev_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_NEXT:
        filelist_next_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


static fileitem_t **
filelist_get_selected_fileitems (int *num_selected)
{
    return (fileitem_t **) otk_list_get_selected (filelist_menu_list,
                                                  num_selected);
}


static fileitem_t *
subtitle_get_selected (void)
{
    int num_selected;
    fileitem_t **fileitems = filelist_get_selected_fileitems (&num_selected);
    if (!fileitems)
        return NULL;

    fileitem_t *res = fileitems[0];
    ho_free (fileitems);

    return res;
}


static void
playitem_add_subtitle (void *playitem_p)
{
    playitem_t *playitem = (playitem_t *) playitem_p;
    fileitem_t *selected = subtitle_get_selected ();

    if (selected && is_file_allowed (selected->mrl, ALLOW_FILES_SUBTITLE)) {
        ho_free (playitem->subtitle_mrl);
        playitem->subtitle_mrl = ho_strdup (selected->mrl);
        show_playlist_menu_cb (oxine);
    }
}


static void
filelist_menu_show_gui (void)
{
    show_menu_background (OXINE_BACKGROUNDS "/subtitlemenu.png");

    if (filelist_menu_window) {
        otk_window_set_current (oxine->otk, filelist_menu_window);

        current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);

        filelist_menu_update_list ();
        filelist_menu_update_buttons ();

        return;
    }
    odk_add_event_handler (oxine->odk, filelist_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    filelist_menu_window = create_new_window (true, true);

    int x = 20;
    int y = 100;
    filelist_menu_buttons[0] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Add"),
                             playitem_add_subtitle, playlist_get_selected ());
    y += 40;
    otk_text_button_new (oxine->otk, x, y, 180, 35, _("Back"),
                         show_playlist_menu_cb, oxine);

    filelist_menu_show_list ();
    filelist_menu_update_list ();
    filelist_menu_update_buttons ();
}


static void
show_subtitle_menu_cb (void *oxine_p)
{
    if (!playlist_get_selected ())
        return;

    if (!toplevel_filelist) {
        filelist_ref_set (&toplevel_filelist,
                          filelist_new (NULL, NULL, NULL,
                                        ALLOW_FILES_SUBTITLE));
        filelist_ref_set (&current_filelist, toplevel_filelist);
        {
            const char *mrl = get_dir_home ();
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        {
            const char *mrl = "/";
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        filelist_sort (toplevel_filelist, NULL);
    }

    if (!prev_filelist)
        prev_filelist = l_list_new ();
    if (!next_filelist)
        next_filelist = l_list_new ();

    hide_user_interface (oxine);

    filelist_menu_show_gui ();

    oxine->current_menu = show_subtitle_menu_cb;
    oxine->backto_menu = show_subtitle_menu_cb;

    show_user_interface (oxine);
}


/* 
 * *************************************************************** 
 * Playlist Menu Stuff
 * *************************************************************** 
 */
#define PLAYLIST_BUTTONS_NUM 12
static otk_widget_t *playlist_menu_buttons[PLAYLIST_BUTTONS_NUM];
static otk_widget_t *playlist_menu_list = NULL;
static otk_widget_t *playlist_menu_window = NULL;

static otk_widget_t *playlist_menu_editbox = NULL;

static void playlist_menu_update_list (void);
static void playlist_menu_update_buttons (void);

static void playlist_select_cb (void *entry_cb_data);
static void playlist_activate_cb (void *entry_cb_data);
static void playlist_remove_cb (void *entry_cb_data);

static void
playlist_menu_update_buttons (void)
{
    bool enabled;

    enabled = (playlist_length (oxine->rw_playlist) > 0);
    otk_widget_set_enabled (playlist_menu_buttons[0], enabled);
    otk_widget_set_enabled (playlist_menu_buttons[2], enabled);

    if (!enabled)
        otk_widget_set_focus (playlist_menu_buttons[3]);

    enabled &= !(odk_current_is_playback_mode (oxine->odk)
                 && (oxine->current_playlist == oxine->rw_playlist));
    otk_widget_set_enabled (playlist_menu_buttons[1], enabled);

    enabled = (otk_list_get_selected_count (playlist_menu_list) > 0);
    otk_widget_set_enabled (playlist_menu_buttons[4], enabled);
    otk_widget_set_enabled (playlist_menu_buttons[5], enabled);
    otk_widget_set_enabled (playlist_menu_buttons[6], enabled);

    playitem_t *selected = playlist_get_selected ();

    enabled = selected && is_file_allowed (selected->mrl, ALLOW_FILES_VIDEO);
    otk_widget_set_enabled (playlist_menu_buttons[7], enabled);

    if (selected && (selected->subtitle_mrl != NULL)) {
        otk_button_set_text (playlist_menu_buttons[7], _("Delete subtitle"));
    } else {
        otk_button_set_text (playlist_menu_buttons[7], _("Add subtitle"));
    }

    enabled = odk_current_is_playback_mode (oxine->odk);
    otk_widget_set_visible (playlist_menu_buttons[11], enabled);
}


static void
playlist_menu_update_list (void)
{
    assert (playlist_menu_list);

    int enabled = 0;
    otk_clear_list (playlist_menu_list);

    playitem_t *cur = playlist_first (oxine->rw_playlist);
    while (cur) {
        enabled = 1;

        char title[1024];
        if (cur->subtitle_mrl)
            snprintf (title, 1024, "%s (+%s)", cur->title, _("Subtitle"));
        else
            snprintf (title, 1024, "%s", cur->title);

        otk_widget_t *w = otk_listentry_new (playlist_menu_list, title,
                                             playlist_activate_cb, cur,
                                             playlist_select_cb, cur,
                                             playlist_remove_cb, cur);

        if (odk_current_is_playback_mode (oxine->odk)
            && (oxine->current_playlist == oxine->rw_playlist)
            && (cur == playlist_get_current (oxine->rw_playlist)))
            otk_widget_set_enabled (w, false);

        cur = playlist_next (oxine->rw_playlist, cur);
    }

    otk_list_set_pos (playlist_menu_list, oxine->rw_playlist->top_position);
    otk_list_set_focus (playlist_menu_list, oxine->rw_playlist->cur_position);
}


static void
playlist_activate_cb (void *playitem_p)
{
    playitem_t *playitem = (playitem_t *) playitem_p;

    playlist_play_item (oxine, show_playback_menu_cb,
                        oxine->rw_playlist, playitem);
}


static void
playlist_select_cb (void *playitem_p)
{
    playlist_menu_update_buttons ();
    show_user_interface (oxine);
}


static void
playlist_remove_cb (void *playitem_p)
{
    playitem_t *playitem = (playitem_t *) playitem_p;

    if (!(odk_current_is_playback_mode (oxine->odk)
          && (playitem == playlist_get_current (oxine->rw_playlist)))) {

        playlist_remove (oxine->rw_playlist, playitem);

        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);
        oxine->rw_playlist->cur_position =
            otk_list_get_focus (playlist_menu_list);

        playlist_menu_update_buttons ();
        playlist_menu_update_list ();
        show_user_interface (oxine);
    }
}


static void
playlist_clear_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (!odk_current_is_playback_mode (oxine->odk)
        || (oxine->current_playlist != oxine->rw_playlist)) {

        playlist_clear (oxine->rw_playlist);

        playlist_menu_update_list ();
        playlist_menu_update_buttons ();
        show_user_interface (oxine);
    }
}


static playitem_t *
playlist_get_selected (void)
{
    int num_selected;
    playitem_t **playitems =
        (playitem_t **) otk_list_get_selected (playlist_menu_list,
                                               &num_selected);

    if (num_selected != 1)
        return NULL;

    playitem_t *res = playitems[0];
    ho_free (playitems);

    return res;
}


static int
playlist_get_selected_pos (void)
{
    int num_selected;

    int *positions = otk_list_get_selected_pos (playlist_menu_list,
                                                &num_selected);

    if (num_selected != 1)
        return -1;

    int pos = positions[0];
    ho_free (positions);
    return pos;
}


static void
playlist_remove_subtitle_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    playitem_t *selected = playlist_get_selected ();
    if (selected && selected->subtitle_mrl) {
        int position = playlist_get_selected_pos ();
        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);

        ho_free (selected->subtitle_mrl);
        selected->subtitle_mrl = NULL;

        playlist_menu_update_list ();
        otk_list_set_selected (playlist_menu_list, position, true);

        playlist_menu_update_buttons ();
        otk_widget_set_focus (playlist_menu_buttons[7]);

        show_user_interface (oxine);
    }
}


static void
playlist_subtitle_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    playitem_t *selected = playlist_get_selected ();
    if (selected && selected->subtitle_mrl) {
        playlist_remove_subtitle_cb (oxine);
    } else {
        show_subtitle_menu_cb (oxine);
    }
}


static void
playlist_remove_selected_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    playitem_t *selected = playlist_get_selected ();

    if (!selected)
        return;
    if (odk_current_is_playback_mode (oxine->odk)
        && (selected == playlist_get_current (oxine->rw_playlist)))
        return;

    int position = playlist_get_selected_pos ();
    oxine->rw_playlist->top_position = otk_list_get_pos (playlist_menu_list);

    playlist_remove (oxine->rw_playlist, selected);

    playlist_menu_update_list ();
    if (position >= playlist_length (oxine->rw_playlist))
        otk_list_set_selected (playlist_menu_list, position - 1, true);
    else
        otk_list_set_selected (playlist_menu_list, position, true);

    playlist_menu_update_buttons ();
    otk_widget_set_focus (playlist_menu_buttons[4]);

    show_user_interface (oxine);
}


static void
playlist_move_up_selected_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    playitem_t *selected = playlist_get_selected ();
    if (selected) {
        int position = playlist_get_selected_pos () - 1;
        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);

        playlist_move_up (oxine->rw_playlist, selected);

        playlist_menu_update_list ();
        while (position < 0)
            position++;
        otk_list_set_selected (playlist_menu_list, position, true);

        playlist_menu_update_buttons ();
        otk_widget_set_focus (playlist_menu_buttons[5]);

        show_user_interface (oxine);
    }
}


static void
playlist_move_down_selected_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    playitem_t *selected = playlist_get_selected ();
    if (selected) {
        int position = playlist_get_selected_pos () + 1;
        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);

        playlist_move_down (oxine->rw_playlist, selected);

        playlist_menu_update_list ();
        while (position >= playlist_length (oxine->rw_playlist))
            position--;
        otk_list_set_selected (playlist_menu_list, position, true);

        playlist_menu_update_buttons ();
        otk_widget_set_focus (playlist_menu_buttons[6]);

        show_user_interface (oxine);
    }
}


static void
playlist_play_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    playlist_play_first (oxine, show_playback_menu_cb, oxine->rw_playlist);
}


static void
playlist_save_now_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    assert (playlist_menu_editbox);

    char *title = otk_editbox_get_text (playlist_menu_editbox);
    if (!title || (strlen (title) == 0)) {
        ho_free (title);
        playlist_save_cb (oxine);
        return;
    }

    char *mrl = ho_strdup_printf ("%s/%s.m3u",
                                  get_dir_oxine_playlists (),
                                  title);
    playlist_m3u_save (oxine->rw_playlist, mrl);
    ho_free (mrl);
    ho_free (title);

    playlist_menu_editbox = NULL;
    show_playlist_menu_cb (oxine_p);
}


static void
playlist_save_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (playlist_length (oxine->rw_playlist) > 0) {
        hide_user_interface (oxine);

        create_new_window (false, true);
        otk_border_new (oxine->otk, 100, 200, 600, 200);
        otk_label_new (oxine->otk, 120, 250, 560,
                       OTK_ALIGN_LEFT | OTK_ALIGN_BOTTOM,
                       _("Please enter a name for the playlist:"));

        playlist_menu_editbox =
            otk_editbox_new (oxine->otk, 120, 270, 560, 40, 100, NULL, NULL);
        otk_widget_set_focus (playlist_menu_editbox);

        otk_widget_t *b;
        b = otk_text_button_new (oxine->otk, 230, 330, 160, 40, _("OK"),
                                 playlist_save_now_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_text_button_new (oxine->otk, 410, 330, 160, 40, _("Cancel"),
                                 show_playlist_menu_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);

        show_user_interface (oxine);
    }
}


static void
playlist_menu_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if ((oxine->current_menu == show_playlist_menu_cb)
        && (event->type == OXINE_EVENT_KEY)
        && (event->source.key == OXINE_KEY_BACK)) {
        show_filelist_menu_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
    }
}

void
show_playlist_menu_cb (void *oxine_p)
{
    hide_user_interface (oxine);

    oxine->current_menu = show_playlist_menu_cb;
    oxine->backto_menu = show_playlist_menu_cb;
    if (odk_current_is_logo_mode (oxine->odk))
        oxine->playback_ended_menu = show_playlist_menu_cb;

    show_menu_background (OXINE_BACKGROUNDS "/playlistmenu.png");

    if (playlist_menu_window) {
        otk_window_set_current (oxine->otk, playlist_menu_window);

        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);
        oxine->rw_playlist->cur_position =
            otk_list_get_focus (playlist_menu_list);

        playlist_menu_update_list ();
        playlist_menu_update_buttons ();

        show_user_interface (oxine);

        return;
    }
    odk_add_event_handler (oxine->odk, playlist_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    playlist_menu_window = create_new_window (true, true);

    int x = 20;
    int y = 100;
    playlist_menu_buttons[0] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Play"),
                                                    playlist_play_cb, oxine);
    y += 40;
    playlist_menu_buttons[1] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Clear list"),
                                                    playlist_clear_cb, oxine);
    y += 40;
    playlist_menu_buttons[2] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Save list"),
                                                    playlist_save_cb, oxine);

    y += 50;
    playlist_menu_buttons[3] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Add"),
                                                    show_filelist_menu_cb,
                                                    oxine);
    y += 40;
    playlist_menu_buttons[4] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Remove"),
                                                    playlist_remove_selected_cb,
                                                    oxine);
    y += 40;
    playlist_menu_buttons[5] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Move up"),
                                                    playlist_move_up_selected_cb,
                                                    oxine);
    y += 40;
    playlist_menu_buttons[6] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Move down"),
                                                    playlist_move_down_selected_cb,
                                                    oxine);
    y += 40;
    playlist_menu_buttons[7] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Add subtitle"),
                                                    playlist_subtitle_cb,
                                                    oxine);
    y += 50;
    playlist_menu_buttons[9] = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                                    _("Media-Browser"),
                                                    show_filelist_menu_cb,
                                                    oxine);
    y += 40;
    playlist_menu_buttons[10] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Mainmenu"),
                             show_main_menu_cb, oxine);
    y += 40;
    playlist_menu_buttons[11] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Current title"),
                             show_playback_menu_cb, oxine);

    playlist_menu_list = otk_list_new (oxine->otk, 220, 100, 560, 480,
                                       30, 35, true, true,
                                       OTK_LIST_SINGLE_SELECTION, oxine);

    playlist_menu_update_list ();
    playlist_menu_update_buttons ();

    show_user_interface (oxine);
}


void
playlist_menu_free (void)
{
    if (current_thumbnail)
        ho_free (current_thumbnail);

    if (next_filelist)
        l_list_free (next_filelist, NULL);
    next_filelist = NULL;
    if (prev_filelist)
        l_list_free (prev_filelist, NULL);
    prev_filelist = NULL;

    filelist_ref_set (&current_filelist, NULL);
    filelist_ref_set (&toplevel_filelist, NULL);
}
