
/*
 * Copyright (C) 2002-2003 Stefan Holst
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_lirc.c 1434 2006-10-30 14:11:17Z mschwerin $
 *
 */
#include "config.h"

#ifdef HAVE_LIRC
#include <lirc/lirc_client.h>
#endif

#include <errno.h>
#include <regex.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"

#include "odk.h"
#include "odk_keymap.h"
#include "odk_lirc.h"

#ifdef DEBUG
#define LIRC_VERBOSE 1
#else
#define LIRC_VERBOSE 0
#endif

#ifdef HAVE_LIRC
typedef struct {
    odk_input_t class;

    odk_t *odk;
    void (*event_handler) (void *data, oxine_event_t * ev);

    int lirc_fd;
    struct lirc_config *lircconfig;
} lirc_input_t;


static void
extract_how (oxine_event_t * ev, char *config)
{
    char *plus = index (config, '+');
    char *minus = index (config, '-');

    if (plus) {
        if (plus[1] == '\0')
            ev->data.how = 1;
        else
            ev->data.how = atoi (plus);
    }

    if (minus) {
        if (minus[1] == '\0')
            ev->data.how = -1;
        else
            ev->data.how = atoi (minus);
    }
}


static void *
lirc_thread (void *this)
{
    lirc_input_t *li = (lirc_input_t *) this;

    while (1) {
        pthread_testcancel ();

        char *code;
        while (lirc_nextcode (&code) == 0) {
            pthread_testcancel ();

            if (code == NULL)
                break;

            int ret;
            char *config;
            while ((ret = lirc_code2char (li->lircconfig, code, &config) == 0)
                   && (config != NULL)) {
                pthread_testcancel ();

                oxine_event_t ev;
                ev.type = OXINE_EVENT_KEY;
                if (odk_keymap_lirc2action (&ev, config)) {
                    extract_how (&ev, config);
                    li->event_handler (li->odk, &ev);
                }

                else {
                    warn (_("Got unknown LIRC command: '%s'"), config);
                }
            }
            free (code);

            if (ret == -1)
                break;
        }
    }

    pthread_exit (NULL);
    return NULL;
}


odk_input_t *
start_lirc (odk_t * odk, event_handler_t event_handler)
{
    lirc_input_t *li = ho_new (lirc_input_t);
    li->odk = odk;
    li->event_handler = event_handler;

    /* Initialize LIRC client. */
    if ((li->lirc_fd = lirc_init ("oxine", LIRC_VERBOSE)) == -1) {
        error (_("Failed to initialize LIRC."));

        info (_("Make sure you have lircd running."));
        info (_("Make sure you have the permissions "
                "to connect to the LIRC socket."));

        ho_free (li);
        return NULL;
    }

    /* Try to read configuration. First we try the configuration in the users
     * oxine config directory and then we try the standard configuration. */
    const char *filename = get_file_lirc_config ();
    if (lirc_readconfig ((char *) filename, &li->lircconfig, NULL) != 0) {
        warn (_("Could not find %s, trying to use default "
                "configuration file %s/lircrc."), filename, OXINE_DATADIR);

        if (lirc_readconfig (OXINE_DATADIR "/lircrc",
                             &li->lircconfig, NULL) != 0) {
            error (_("Could not find a LIRC configuration file."));

            lirc_deinit ();
            ho_free (li);
            return NULL;
        } else {
            debug ("Successfully opened %s/lircrc." OXINE_DATADIR);
        }
    } else {
        debug ("Successfully opened %s.", filename);
    }

    /* Create the thread to handle LIRC commands. */
    if (pthread_create (&li->class.thread, NULL, lirc_thread, li) != 0) {
        error (_("Could not create LIRC thread: %s!"), strerror (errno));

        lirc_deinit ();
        ho_free (li);
        return NULL;
    }

    info (_("Successfully started the LIRC input plugin."));

    return (odk_input_t *) li;
}


void
stop_lirc (odk_input_t * oi)
{
    lirc_input_t *li = (lirc_input_t *) oi;
    pthread_cancel (oi->thread);

    lirc_freeconfig (li->lircconfig);
    lirc_deinit ();
    ho_free (li);

    info (_("Successfully stopped the LIRC input plugin."));
}
#endif /* HAVE_LIRC */
