
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menulist.c 1522 2006-11-11 13:57:10Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "codeset.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "menulist.h"
#include "otk.h"
#include "oxine.h"
#include "xmlparser.h"

#include "help_menu.h"
#include "main_menu.h"
#include "filelist_menu.h"
#include "playlist_menu.h"
#include "settings_menu.h"
#include "weather_menu.h"

extern oxine_t *oxine;

static menuitem_t *
menuitem_load (xml_node_t * node, menulist_t * menulist)
{
    menuitem_t *item = ho_new (menuitem_t);

    item->title = NULL;
    item->submenu = NULL;
    item->is_uplink = 0;
    item->cb = NULL;
    item->cb_data = oxine;

    node = node->child;
    while (node) {
        if (!strcasecmp (node->name, "title")) {
            char *codeset = get_system_encoding ();
            recode_t *xr = recode_init ("utf8", codeset);
            item->title = recode (xr, node->data);
            recode_done (xr);
            if (codeset)
                ho_free (codeset);
        }

        else if (!strcasecmp (node->name, "action")) {
            char *type = xml_parser_get_property (node, "type");
            char *param = xml_parser_get_property (node, "parameter");
            if (type) {
                if ((strcasecmp (type, "mrl_play") == 0) && param) {
                    item->cb = play_mrl_cb;
                    item->cb_data = ho_strdup (param);
                } else if ((strcasecmp (type, "auto_play") == 0) && param) {
                    if (strcasecmp (param, "dvd") == 0) {
                        item->cb = play_dvd_cb;
                        item->cb_data = NULL;
                    } else if (strcasecmp (param, "vcd") == 0) {
                        item->cb = play_vcd_cb;
                        item->cb_data = NULL;
                    } else if (strcasecmp (param, "cddb") == 0) {
                        item->cb = play_audiocd_cb;
                        item->cb_data = NULL;
                    } else if (strcasecmp (param, "dvb") == 0) {
                        item->cb = play_dvb_cb;
                        item->cb_data = NULL;
                    } else if (strcasecmp (param, "v4l") == 0) {
                        item->cb = play_v4l_cb;
                        item->cb_data = NULL;
                    }
#ifdef HAVE_VDR
                    else if (strcasecmp (param, "vdr") == 0) {
                        item->cb = play_vdr_cb;
                        item->cb_data = NULL;
                    }
#endif
                    else {
                        error (_("Unknown parameter '%s' for "
                                 "type 'autoplay'!"), param);
                    }
                } else if (strcasecmp (type, "main_menu") == 0) {
                    item->cb = show_main_menu_cb;
                } else if (strcasecmp (type, "music_menu") == 0) {
                    item->cb = enter_music_menu_cb;
                } else if (strcasecmp (type, "video_menu") == 0) {
                    item->cb = enter_video_menu_cb;
                } else if (strcasecmp (type, "playlist_menu") == 0) {
                    item->cb = show_playlist_menu_cb;
                } else if (strcasecmp (type, "help_menu") == 0) {
                    item->cb = show_help_menu_cb;
                } else if (strcasecmp (type, "settings_menu") == 0) {
                    item->cb = show_settings_menu_cb;
#ifdef HAVE_WEATHER
                } else if (strcasecmp (type, "weather_menu") == 0) {
                    item->cb = weather_menu_show_cb;
#endif
                } else if (strcasecmp (type, "uplink_menu") == 0) {
                    item->cb = show_sub_menu_cb;
                    item->submenu = menulist->parent;
                    item->cb_data = menulist->parent;
                    item->is_uplink = 1;
                } else if (strcasecmp (type, "sub_menu") == 0) {
                    item->cb = show_sub_menu_cb;
                    item->submenu = menulist_new (param, menulist);
                    item->cb_data = item->submenu;
                } else if (strcasecmp (type, "shutdown") == 0) {
                    item->cb = shutdown_cb;
                } else if (strcasecmp (type, "shellcommand") == 0) {
                    item->cb = execute_cb;
                    item->cb_data = ho_strdup (param);
                }
            }
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }

        node = node->next;
    }

    if (item && !item->cb) {
        ho_free (item->title);
        ho_free (item);
        item = NULL;
    }

    return item;
}

static void
menuitem_free_cb (void *data)
{
    menuitem_t *menuitem = (menuitem_t *) data;
    assert (menuitem);

    if (menuitem->title)
        ho_free (menuitem->title);
    if (menuitem->cb_data == oxine)
        menuitem->cb_data = NULL;
    if (menuitem->cb_data == menuitem->submenu)
        menuitem->cb_data = NULL;
    if (menuitem->cb_data)
        ho_free (menuitem->cb_data);
    if (menuitem->submenu && !menuitem->is_uplink)
        menulist_free (menuitem->submenu);
    ho_free (menuitem);
}


static void
read_nodes (menulist_t * menulist, xml_node_t * node)
{
    while (node) {
        if (!strcasecmp (node->name, "button")) {
            menuitem_t *item = menuitem_load (node, menulist);
            if (item && item->cb)
                l_list_append (menulist->list, item);
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }

        node = node->next;
    }
}


static int
main_menu_parse (menulist_t * menulist, const char *mrl)
{
    int size;
    char *file = read_entire_file (mrl, &size);
    xml_node_t *node;

    if (!file)
        return 0;

    xml_parser_init (file, strlen (file), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node) < 0) {
        error (_("Parsing '%s' failed!"), mrl);
        return 0;
    }

    if (strcasecmp (node->name, "oxine_menu")) {
        error (_("Root node of '%s' must be '%s'!"), mrl, "oxine_menu");
        return 0;
    }

    read_nodes (menulist, node->child);
    xml_parser_free_tree (node);
    ho_free (file);

    return 1;
}


menulist_t *
menulist_new (const char *mrl, menulist_t * parent)
{
    menulist_t *menulist = ho_new (menulist_t);

    menulist->list = l_list_new ();
    menulist->parent = parent;

    info (_("Loading menu file '%s'..."), mrl);
    if (!main_menu_parse (menulist, mrl)) {
        menulist_free (menulist);
        return NULL;
    }

    return menulist;
}


void
menulist_free (menulist_t * menulist)
{
    assert (menulist);

    if (menulist->list)
        l_list_free (menulist->list, menuitem_free_cb);
    ho_free (menulist);
}
