/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: ToolbarController.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: hr $ $Date: 2007/06/27 15:40:57 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "precompiled_sd.hxx"

#include "framework/ToolBarController.hxx"

#include "framework/FrameworkHelper.hxx"
#include "framework/FactoryContainer.hxx"
#include "ViewShellBase.hxx"
#include "DrawController.hxx"
#include "ToolBarManager.hxx"

#include <comphelper/stl_types.hxx>
#include <boost/bind.hpp>

using namespace ::com::sun::star;
using namespace ::com::sun::star::uno;
using namespace ::com::sun::star::drawing::framework;

using ::rtl::OUString;

namespace sd { namespace framework {

Reference<XInterface> SAL_CALL ToolbarController_createInstance (
    const Reference<XComponentContext>& rxContext)
{
    return Reference<XInterface>((XWeak*)new ToolBarController(rxContext), UNO_QUERY);
}




::rtl::OUString ToolbarController_getImplementationName (void) throw(RuntimeException)
{
    return ::rtl::OUString(
        RTL_CONSTASCII_USTRINGPARAM("com.sun.star.comp.Draw.framework.ToolbarController"));
}




Sequence<rtl::OUString> SAL_CALL ToolbarController_getSupportedServiceNames (void)
    throw (RuntimeException)
{
	static const ::rtl::OUString sServiceName(
        ::rtl::OUString::createFromAscii("com.sun.star.drawing.framework.ToolbarController"));
	return Sequence<rtl::OUString>(&sServiceName, 1);
}




class ToolBarController::ToolBarDescriptor
{
public:
    Reference<XResourceId> mxToolBarId;
    Reference<XToolBar> mxToolBar;
    Reference<XToolBarFactory> mxFactory;
    static bool CompareResourceId(
        const ToolBarDescriptor& rD, const Reference<XResourceId>& rxResourceId)
    { return rD.mxToolBarId->compareTo(rxResourceId)==0; }
};


// ToolBarContainer

class ToolBarController::ToolBarContainer
    : public ::std::vector<ToolBarDescriptor>
{
public:
    ToolBarContainer (void) {}
};

//===== ToolbarController =====================================================

ToolBarController::ToolBarController (
    const Reference<XComponentContext>&rxContext)
    throw()
    : ToolBarControllerInterfaceBase(MutexOwner::maMutex),
      mpBase(NULL),
      mpToolBarManagerLock(),
      mpFactoryContainer(new FactoryContainer()),
      mpToolBarContainer(new ToolBarContainer()),
      mbMainViewSwitchUpdatePending(false)
{
    (void)rxContext;
}




ToolBarController::~ToolBarController (void)
    throw()
{
}




void SAL_CALL ToolBarController::disposing (void)
{
    ::osl::MutexGuard aGuard (maMutex);

    Reference<XComponent> xComponent (mxConfigurationController, UNO_QUERY);
    if (xComponent.is())
        xComponent->removeEventListener(this);
    xComponent = Reference<XComponent>(mxController, UNO_QUERY);
    if (xComponent.is())
        xComponent->removeEventListener(this);

    while ( ! mpToolBarContainer->empty())
    {
        ReleaseToolBar(mpToolBarContainer->front().mxToolBarId);
    }
    mpToolBarContainer.reset();
    mpBase = NULL;
}




//----- XToolBarController ----------------------------------------------------

void SAL_CALL ToolBarController::addToolBarFactory(
    const ::rtl::OUString& rsToolBarURL,
    const Reference<drawing::framework::XToolBarFactory >& rxToolBarFactory)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->AddFactory(rsToolBarURL, rxToolBarFactory);
}
    



void SAL_CALL ToolBarController::removeToolBarFactoryForURL (
    const ::rtl::OUString& rsToolBarURL)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->RemoveFactoryForURL(rsToolBarURL);
}




void SAL_CALL ToolBarController::removeToolBarFactoryForReference(
    const Reference<XToolBarFactory>& rxToolBarFactory)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->RemoveFactoryForReference(rxToolBarFactory);
}




void SAL_CALL ToolBarController::updateStart (
    const Reference<XConfiguration>& rxRequestedConfiguration,
    const Reference<XConfiguration>& rxCurrentConfiguration,
    const Sequence<Reference<XResourceId> >& rResourcesToDeactivate)
    throw (RuntimeException)
{
    (void)rxRequestedConfiguration;
    
    // Lock the ToolBarManager and tell it to lock the ViewShellManager as
    // well.  This way the ToolBarManager can optimize the releasing of
    // locks and arranging of updates of both tool bars and the view shell
    // stack.
    if (mpBase != NULL)
    {
        mpToolBarManagerLock.reset(new ToolBarManager::UpdateLock(mpBase->GetToolBarManager()));
        mpBase->GetToolBarManager()->LockViewShellManager();
    }

    // Deactivate the tool bars in rResourcesToDeactivate.
    sal_Int32 nCount (rResourcesToDeactivate.getLength());
    for (sal_Int32 nIndex=0; nIndex<nCount; ++nIndex)
    {
        Reference<XResourceId> xResourceId (rResourcesToDeactivate[nIndex]);
        if (xResourceId->getResourceURL().match(FrameworkHelper::msToolBarURLPrefix))
            if (ReleaseToolBar(xResourceId))
                rxCurrentConfiguration->removeResource(xResourceId);
    }

    // Look at the resources to deactivate for the view in the center pane.
    // If that is present then remember to update the visible tool bars in
    // updateEnd().  This will be replaced when the ToolBarManager is moved
    // to the drawing framework.
    for (sal_Int32 nIndex=0; nIndex<nCount; ++nIndex)
    {
        Reference<XResourceId> xResourceId (rResourcesToDeactivate[nIndex]);
        if (xResourceId->getResourceURL().match(FrameworkHelper::msViewURLPrefix))
            if (xResourceId->isBoundToURL(
                FrameworkHelper::msCenterPaneURL, AnchorBindingMode_DIRECT))
            {
                mbMainViewSwitchUpdatePending = true;
            }
    }
}




void SAL_CALL ToolBarController::updateEnd (
    const Reference<XConfiguration>& rxRequestedConfiguration,
    const Reference<XConfiguration>& rxCurrentConfiguration,
    const Sequence<Reference<XResourceId> >& rResourcesToActivate)
    throw (RuntimeException)
{
    (void)rxRequestedConfiguration;

    // Activate the tool bars in rResourcesToActivate.
    sal_Int32 nCount (rResourcesToActivate.getLength());
    for (sal_Int32 nIndex=0; nIndex<nCount; ++nIndex)
    {
        Reference<XResourceId> xResourceId (rResourcesToActivate[nIndex]);
        if (xResourceId->getResourceURL().match(FrameworkHelper::msToolBarURLPrefix))
            if (CreateToolBar(xResourceId))
                rxCurrentConfiguration->addResource(xResourceId);
    }

    // Look at the resources to activate for the view in the center pane.
    // If that is present then remember to update the visible tool bars in
    // updateEnd().  This will be replaced when the ToolBarManager is moved
    // to the drawing framework.
    for (sal_Int32 nIndex=0; nIndex<nCount; ++nIndex)
    {
        Reference<XResourceId> xResourceId (rResourcesToActivate[nIndex]);
        if (xResourceId->getResourceURL().match(FrameworkHelper::msViewURLPrefix))
            if (xResourceId->isBoundToURL(
                FrameworkHelper::msCenterPaneURL, AnchorBindingMode_DIRECT))
            {
                mbMainViewSwitchUpdatePending = true;
            }
    }
    if (mbMainViewSwitchUpdatePending)
    {
        mbMainViewSwitchUpdatePending = false;
        // Update the set of visible tool bars and deactivate those that are
        // no longer visible.  This is done before the old view shell is
        // destroyed in order to avoid unnecessary updates of those tool
        // bars.
        ::boost::shared_ptr<ToolBarManager> pToolBarManager (mpBase->GetToolBarManager());
        ::boost::shared_ptr<FrameworkHelper> pFrameworkHelper (
            FrameworkHelper::Instance(*mpBase));
        ViewShell* pViewShell
            = pFrameworkHelper->GetViewShell(FrameworkHelper::msCenterPaneURL).get();
        if (pViewShell != NULL)
        {
            pToolBarManager->MainViewShellChanged(*pViewShell);
            pToolBarManager->SelectionHasChanged(
                *pViewShell,
                *pViewShell->GetView());
            pToolBarManager->PreUpdate();
        }
    }

    // Releasing the update lock of the ToolBarManager  will let the
    // ToolBarManager with the help of the ViewShellManager take care of
    // updating tool bars and view shell with the minimal amount of
    // shell stack modifications and tool bar updates.
    mpToolBarManagerLock.reset();
}




Reference<XToolBar> SAL_CALL ToolBarController::getToolBar (
    const Reference<XResourceId>& rxToolBarId)
    throw (RuntimeException)
{
    ToolBarDescriptor aDescriptor = GetActiveToolBar(rxToolBarId);
    return aDescriptor.mxToolBar;
}




//----- XInitialization -------------------------------------------------------

void SAL_CALL ToolBarController::initialize (const Sequence<Any>& aArguments)
    throw (Exception, RuntimeException)
{
    if (aArguments.getLength() > 0)
    {
        try
        {
            // Get the XController from the first argument.
            mxController = Reference<frame::XController>(aArguments[0], UNO_QUERY_THROW);

            Reference<XComponent> xComponent (mxController, UNO_QUERY);
            if (xComponent.is())
                xComponent->addEventListener(this);

            // Tunnel through the controller to obtain a ViewShellBase.
            Reference<lang::XUnoTunnel> xTunnel (mxController, UNO_QUERY);
            if (xTunnel.is())
            {
                ::sd::DrawController* pController = reinterpret_cast<sd::DrawController*>(
                    xTunnel->getSomething(sd::DrawController::getUnoTunnelId()));
                if (pController != NULL)
                    mpBase = pController->GetViewShellBase();
            }

            // Get the configuration controller.
            Reference<XControllerManager> xControllerManager (mxController, UNO_QUERY_THROW);
            mxConfigurationController = xControllerManager->getConfigurationController();
            if ( ! mxConfigurationController.is())
                throw RuntimeException();
            else
            {
                xComponent = Reference<XComponent>(mxConfigurationController, UNO_QUERY);
                if (xComponent.is())
                    xComponent->addEventListener(this);
            }

            mpFactoryContainer->SetModuleController(xControllerManager->getModuleController());
        }
        catch (RuntimeException&)
        {
            DBG_ASSERT(false, "ToolBarController::initialize(): caught exception");
        }
    }
}




//----- lang::XEventListener --------------------------------------------------

void SAL_CALL ToolBarController::disposing (const lang::EventObject& rEventObject)
    throw (RuntimeException)
{
    if (rEventObject.Source == mxController)
        mxController = NULL;
    else if (rEventObject.Source == mxConfigurationController)
        mxConfigurationController = NULL;
}




//-----------------------------------------------------------------------------

bool ToolBarController::CreateToolBar (const Reference<XResourceId>& rxToolBarId)
{
    bool bAdded (false);
    
    {
        ::osl::MutexGuard aGuard (maMutex);
        
        ToolBarDescriptor aDescriptor (GetActiveToolBar(rxToolBarId));
        if ( ! aDescriptor.mxToolBar.is())
        {
            aDescriptor.mxToolBarId = rxToolBarId;
            aDescriptor.mxFactory = Reference<XToolBarFactory>(
                mpFactoryContainer->GetFactory(rxToolBarId->getResourceURL()), UNO_QUERY);
            if (aDescriptor.mxFactory.is())
            {
                aDescriptor.mxToolBar = aDescriptor.mxFactory->createToolBar(
                    rxToolBarId,
                    mxController);
                mpToolBarContainer->push_back(aDescriptor);

                bAdded = true;
            }
        }
    }

    if (bAdded)
    {
        ConfigurationChangeEvent aEvent;
        aEvent.Type = FrameworkHelper::msResourceActivationEvent;
        aEvent.ResourceId = rxToolBarId;
        if (mxConfigurationController.is())
        {
            try
            {
                mxConfigurationController->notifyEvent(aEvent);
            }
            catch (lang::DisposedException)
            {
                mxConfigurationController = NULL;
            }
        }
    }

    return bAdded;
}




bool ToolBarController::ReleaseToolBar (const Reference<XResourceId>& rxToolBarId)
{
    bool bRemoved (false);
    ToolBarDescriptor aDescriptor;
    
    // Release the tool bar and remove it from the configuration.
    {
        ::osl::MutexGuard aGuard (maMutex);
        ToolBarContainer::iterator iDescriptor (
            ::std::find_if(
                mpToolBarContainer->begin(),
                mpToolBarContainer->end(),
                ::boost::bind(&ToolBarDescriptor::CompareResourceId, _1, rxToolBarId)));
        aDescriptor = *iDescriptor;
        
        if (iDescriptor!=mpToolBarContainer->end())
        {
            mpToolBarContainer->erase(iDescriptor);
            bRemoved = true;
        }
    }

    if (bRemoved)
    {
        // Notifiy the tool bar being deactivated to listeners of the ConfigurationController.
        ConfigurationChangeEvent aEvent;
        aEvent.Type = FrameworkHelper::msResourceDeactivationEvent;
        aEvent.ResourceId = rxToolBarId;
        if (mxConfigurationController.is())
        {
            try
            {
                mxConfigurationController->notifyEvent(aEvent);
            }
            catch (lang::DisposedException)
            {
                mxConfigurationController = NULL;
            }
        }

        aDescriptor.mxFactory->releaseToolBar(aDescriptor.mxToolBar);
    }

    return bRemoved;
}




ToolBarController::ToolBarDescriptor ToolBarController::GetActiveToolBar (
    const Reference<XResourceId>& rxToolBarId) const
{
    ToolBarContainer::const_iterator iDescriptor (
        ::std::find_if(
            mpToolBarContainer->begin(),
            mpToolBarContainer->end(),
            ::boost::bind(&ToolBarDescriptor::CompareResourceId, _1, rxToolBarId)));
    if (iDescriptor != mpToolBarContainer->end())
        return *iDescriptor;
    else
        return ToolBarDescriptor();
}




void ToolBarController::ThrowIfDisposed (void) const
    throw (::com::sun::star::lang::DisposedException)
{
	if (rBHelper.bDisposed || rBHelper.bInDispose)
	{
        throw lang::DisposedException (
            OUString(RTL_CONSTASCII_USTRINGPARAM(
                "ToolBarController object has already been disposed")),
            const_cast<uno::XWeak*>(static_cast<const uno::XWeak*>(this)));
    }
}




} } // end of namespace sd::framework
