/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#ifndef __IWL_AMFH_H__
#define __IWL_AMFH_H__

struct iwl_amfh_t;
struct iwl_al_amfh_config;

extern struct iwl_amfh_t *iwl_amfh_em;

/**
 * struct iwl_amfh_config - handlers to AMFH from AL.
 * @push: callback which pushes data from the AMFH to the IDI stream.
 * @available: callback to check how much space in available for pushing.
 *
 * In general, before each push, the available callback should be called,
 * and only if there is enough space in the IDI stream we push the data.
 */
struct iwl_amfh_config {
	int (*push)(const void*, u32);
	bool (*available)(u32);
};

/**
 * iwl_amfh_init - initializes the amfh emulation block.
 * @al_config: init configuration for AMFH (not iwl_amfh_config!)
 * @trans: pointer to the transport layer of the driver
 *
 * Most AMFH initialization is setting up the RX rings, and most configuration
 * is exactly these parameters. The call is sync, and should be run in kthread
 * context.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_amfh_init(struct iwl_al_amfh_config *al_config,
				struct iwl_trans *trans);

/**
 * iwl_amfh_free - frees the amfh emulation block.
 *
 * These call currently does nothing. The call is sync.
 */
void iwl_amfh_free(void);

/**
 * iwl_amfh_start - starts the operation of the amfh emulation block.
 * @amfh_config: start configuration of AMFH, callbacks to AL functions.
 *
 * After AMFH was initialized, it can be started (in a kthread context).
 * The start flows allocates the RX rings and configures the HW to start
 * transmitting RX on them. Note that this call can come either after init,
 * or after iwl_amfh_stop, and no other condition is valid.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_amfh_start(struct iwl_amfh_config *amfh_config);

/**
 * iwl_amfh_stop - stops the operation of the amfh emulation block.
 *
 * Stopping the AMFH stops both the processing thread and stops the HW
 * RX rings. It also frees the resources of the rings which were allocated
 * on iwl_amfh_start. This call is sync and should be called from a kthread
 * context.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_amfh_stop(void);

/**
 * iwl_amfh_resume - signals AMFH to resume its operation.
 *
 * AMFH can pause its operation when there is not enough space
 * in the IDI stream for the next chunk of data. After each read
 * of data from the stream this function should be called in order
 * to signal AMFH to check the available space again.
 *
 * Always succeeds.
 */
int iwl_amfh_resume(void);

/**
 * iwl_amfh_rx_handler - notifies AMFH of new RX packets.
 *
 * This call notifies the AMFH of new RX, which triggers its
 * handling thread to check and process whatever there is on
 * the RX ring.
 *
 * Always succeeds.
 */
u32 iwl_amfh_rx_handler(void);

#endif /* __IWL_AMFH_H__ */
