/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** visual-queue.c - manages the bubbles (synchronous, asynchronous and
**                  snap-decisions) on screen
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include "sd-bubble.h"
#include "queue.h"
#include "visual-queue.h"
#include "settings.h"

G_DEFINE_TYPE (VisualQueue, visual_queue, G_TYPE_OBJECT);

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), VISUAL_QUEUE_TYPE, VisualQueuePrivate))

struct _VisualQueuePrivate {
	GList*    list;
	Queue*    queue;
	gint      right_edge;
	gint      current_y;
	gint      top_edge;
	gint      bottom_edge;
	guint     default_id;
	Settings* settings;
};

enum
{
	EMPTY,
	LAST_SIGNAL
};

static guint g_visual_queue_signals[LAST_SIGNAL] = { 0 };

/*-- internal API ------------------------------------------------------------*/

static void
visual_queue_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (visual_queue_parent_class)->dispose (gobject);
}

static void
_remove_sd (gpointer data)
{
	g_assert (data != NULL);
	SdBubble* sd = SD_BUBBLE (data);
	g_object_unref (sd);
}

static void
visual_queue_finalize (GObject* gobject)
{
	VisualQueue* visual_queue = NULL;
	VisualQueuePrivate* priv  = NULL;

	// sanity checks
	g_assert (gobject);
	visual_queue = VISUAL_QUEUE (gobject);
	g_assert (visual_queue);
	g_assert (IS_VISUAL_QUEUE (visual_queue));
	priv = GET_PRIVATE (visual_queue);
	g_assert (priv);

	if (priv->list != NULL)
		g_list_free_full (priv->list, _remove_sd);

	/* chain up to the parent class */
	G_OBJECT_CLASS (visual_queue_parent_class)->finalize (gobject);
}

static void
visual_queue_init (VisualQueue* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */
}

static void
visual_queue_get_property (GObject*    gobject,
                           guint       prop,
                           GValue*     value,
                           GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
visual_queue_class_init (VisualQueueClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (VisualQueuePrivate));

	gobject_class->dispose      = visual_queue_dispose;
	gobject_class->finalize     = visual_queue_finalize;
	gobject_class->get_property = visual_queue_get_property;

	g_visual_queue_signals[EMPTY] = g_signal_new (
		"empty",
		G_OBJECT_CLASS_TYPE (gobject_class),
		G_SIGNAL_RUN_LAST,
		G_STRUCT_OFFSET (VisualQueueClass, empty),
		NULL,
		NULL,
		g_cclosure_marshal_VOID__VOID,
		G_TYPE_NONE,
		0);
}

gboolean
_visual_queue_state_changed_handler (SdBubble* sd_bubble,
                                     gpointer  data)
{
	g_return_val_if_fail (data, TRUE);

	VisualQueue* self = VISUAL_QUEUE (data);
	visual_queue_relayout (self);

	return FALSE;
}

gboolean
_visual_queue_focus_changed_handler (SdBubble* sd_bubble,
                                     gpointer  data)
{
	g_return_val_if_fail (data, TRUE);

	VisualQueue*        self = VISUAL_QUEUE (data);
	VisualQueuePrivate* priv = GET_PRIVATE (self);
	SdBubble*           sd   = NULL;

	guint    i       = 0;
	guint    id      = sd_bubble_get_notification_id (sd_bubble);
	gboolean focused = sd_bubble_is_focused (sd_bubble);

	visual_queue_relayout (self);

	if (visual_queue_get_size (self) == 1)
		return FALSE;
	else
	{
		if (focused)
		{
			/* change state */
			for (i = 0; i < visual_queue_get_size (self); i++)
			{
				sd = SD_BUBBLE (g_list_nth_data (priv->list, i));
				if (sd_bubble_get_notification_id (sd) == id)
					sd_bubble_set_state (sd, SD_BUBBLE_STATE_DEFAULT);
				else
					sd_bubble_set_state (sd, SD_BUBBLE_STATE_CONTRACTED);
			}
			priv->default_id = id;
		}
		else
			return FALSE;
	}
	
	visual_queue_relayout (self);

	return FALSE;
}

gboolean
_visual_queue_closed_handler (SdBubble* sd_bubble,
                              gpointer  data)
{
	g_return_val_if_fail (data, TRUE);

	VisualQueue*        self = VISUAL_QUEUE (data);
	VisualQueuePrivate* priv = GET_PRIVATE (self);

	gint id = sd_bubble_get_notification_id (sd_bubble);
	visual_queue_remove_by_id (self, id);
	queue_remove_by_id (priv->queue, id);
	priv->current_y -= sd_bubble_get_height (sd_bubble);
	if (priv->current_y < priv->top_edge)
		priv->current_y = priv->top_edge;
	g_object_unref (sd_bubble);

	if (priv->default_id == id)
	{
		if (visual_queue_get_size (self) >= 1)
		{
			SdBubble* sd     = SD_BUBBLE (g_list_first (priv->list)->data);
			gint      new_id = sd_bubble_get_notification_id (sd);
			sd_bubble_set_state (sd, SD_BUBBLE_STATE_DEFAULT);
			priv->default_id = new_id;
		}
		else
			priv->default_id = 0;
	}

	visual_queue_relayout (self);

	return FALSE;
}

static void
_visual_queue_remove_sd (gpointer data)
{
	g_assert (data != NULL);
	SdBubble* sd = SD_BUBBLE (data);
	sd_bubble_fade_out (sd);
	//g_object_unref (sd);
}

gint
_visual_queue_compare_id (gconstpointer a,
                          gconstpointer b)
{
	gint result;
	gint id_1;
	gint id_2;

	if (!a || !b)
		return -1;

	if (!IS_SD_BUBBLE (a))
		return -1;

	id_1 = sd_bubble_get_notification_id ((SdBubble*) a);
	id_2 = *((gint*) b);

	if (id_1 < id_2)
		result = -1;

	if (id_1 == id_2)
		result = 0;

	if (id_1 > id_2)
		result = 1;

	return result;
}

/*-- public API --------------------------------------------------------------*/

VisualQueue*
visual_queue_new (gint   right_edge,
                  gint   top_edge,
                  gint   bottom_edge,
                  Queue* queue)
{
	VisualQueue*        self = NULL;
	VisualQueuePrivate* priv = NULL;

	g_return_val_if_fail (queue && IS_QUEUE (queue), NULL);

	self = g_object_new (VISUAL_QUEUE_TYPE, NULL);
	if (!self)
		return NULL;

	priv = GET_PRIVATE (self);

	priv->list        = NULL;
	priv->settings    = settings_new ();
	priv->right_edge  = right_edge + 2.0 * priv->settings->drop_shadow->size;
	priv->top_edge    = top_edge;
	priv->current_y   = priv->top_edge;
	priv->bottom_edge = bottom_edge;
	priv->queue       = queue;
	priv->default_id  = 0;

	return self;
}

void
visual_queue_del (VisualQueue* self)
{
	VisualQueuePrivate* priv = NULL;

	g_return_if_fail (self && IS_VISUAL_QUEUE (self));

	priv = GET_PRIVATE (self);

	g_list_free_full (priv->list, _remove_sd);
	priv->list = NULL;

	settings_del (priv->settings);

	g_object_unref (self);
}

void
visual_queue_pull_notification (VisualQueue* self)
{
	VisualQueuePrivate* priv = NULL;
	Notification*       n    = NULL;
	SdBubble*           sd   = NULL;

	g_return_if_fail (self && IS_VISUAL_QUEUE (self));

	priv = GET_PRIVATE (self);

	n = queue_pop (priv->queue);
	if (!n)
		return;

	sd = sd_bubble_new (notification_get_actions (n),
	                    notification_get_num_actions (n),
	                    priv->settings);
	sd_bubble_set_notification_id (sd, notification_get_id (n));
	sd_bubble_set_summary (sd, notification_get_summary (n));
	sd_bubble_set_body (sd, notification_get_body (n));
	sd_bubble_set_dbus_sender (sd, notification_get_dbus_sender (n));
	sd_bubble_set_button_tint (sd, notification_get_button_tint (n));
	sd_bubble_set_icon_pixbuf (sd, notification_get_icon_pixbuf (n));
	notification_set_visible (n);
	if (g_list_length (priv->list) >= 1)
		sd_bubble_set_state (sd, SD_BUBBLE_STATE_CONTRACTED);
	else
	{
		sd_bubble_set_state (sd, SD_BUBBLE_STATE_DEFAULT);
		priv->default_id = sd_bubble_get_notification_id (sd);
	}

	priv->list = g_list_append (priv->list, (gpointer) sd);
	sd_bubble_move (sd,
	                priv->right_edge,
	                priv->current_y);
	//sd_bubble_fade_in (sd);
	sd_bubble_show (sd);

	g_signal_connect (sd,
	                  "closed",
	                  G_CALLBACK (_visual_queue_closed_handler),
	                  self);
	g_signal_connect (sd,
	                  "focus-changed",
	                  G_CALLBACK (_visual_queue_focus_changed_handler),
	                  self);
	g_signal_connect (sd,
	                  "state-changed",
	                  G_CALLBACK (_visual_queue_state_changed_handler),
	                  self);

	priv->current_y += sd_bubble_get_height (sd);

	visual_queue_relayout (self);
}

guint
visual_queue_get_size (VisualQueue* self)
{
	g_return_val_if_fail (self && IS_VISUAL_QUEUE (self), 0);

	return (g_list_length (GET_PRIVATE (self)->list));
}

gboolean
visual_queue_is_full (VisualQueue* self)
{
	g_return_val_if_fail (self, TRUE);

	return (GET_PRIVATE (self)->bottom_edge < GET_PRIVATE (self)->current_y);
}

void
visual_queue_remove_by_id (VisualQueue* self,
                           gint         id)
{
	GList*              element = NULL;
	VisualQueuePrivate* priv    = NULL;

	g_return_if_fail (self && IS_VISUAL_QUEUE (self));

	priv = GET_PRIVATE (self);
	element = g_list_find_custom (priv->list,
	                              (gconstpointer) &id,
	                              _visual_queue_compare_id);
	if (element)
	{
		priv->list = g_list_remove_link (priv->list, element);
		g_list_free_full (element, _visual_queue_remove_sd);
		visual_queue_relayout (self);
	}
	else if (id != 0) /* notifications with ID 0 are not used */
		g_print ("%s() - Could not find notification with ID = %d\n",
		         G_STRFUNC,
		         id);
	else
		g_print ("%s() - Could not find notification with ID = %d\n",
		         G_STRFUNC,
		         id);

	if (visual_queue_get_size (self) == 0)
		g_signal_emit (self, g_visual_queue_signals[EMPTY], 0);
}

void
visual_queue_relayout (VisualQueue* self)
{
	VisualQueuePrivate* priv = NULL;

	priv = GET_PRIVATE (self);

	gint  y = priv->top_edge;
	guint i = 0;
	for (i = 0; i < visual_queue_get_size (self); i++)
	{
		SdBubble* sd = NULL;
		sd = (SdBubble*) g_list_nth_data (priv->list, i);
		sd_bubble_move (sd, priv->right_edge, y);
		y += sd_bubble_get_height (sd) - 2.0 * priv->settings->drop_shadow->size;
	}
}
