/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** sd-bubble.c - implements the bubbles for snap-decisions
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <gtk/gtk.h>

#include <egg/egg-hack.h>
#include <egg/egg-alpha.h>

#include "dbus.h"
#include "bubble.h"
#include "button.h"
#include "split-button.h"
#include "close-button.h"
#include "sd-bubble.h"
#include "metric-converter.h"
#include "raico-blur.h"

G_DEFINE_TYPE (SdBubble, sd_bubble, G_TYPE_OBJECT);

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), SD_BUBBLE_TYPE, SdBubblePrivate))

struct _SdBubblePrivate {
	MetricConverter* mc;
	GtkWidget*       widget;
	gboolean         visible;
	gboolean         focused;
	gboolean         hovered;
	gboolean         composited;
	SdBubbleState    state;
	GdkPixbuf*       icon_pixbuf;
	guint            num_actions;
	gchar**          actions;
	gboolean         uses_split_button;
	GString*         dbus_sender;
	GString*         summary;
	GString*         body;
	guint            notification_id;
	guint            redraw_handler_id;
	cairo_surface_t* contracted_bg_surface;
	cairo_surface_t* default_bg_surface;
	cairo_surface_t* expanded_bg_surface;
	cairo_surface_t* icon_small_surface;
	cairo_surface_t* icon_big_surface;
	cairo_surface_t* summary_surface;
	cairo_surface_t* body_surface;
	Button*          accept_button;
	Button*          reject_button;
	SplitButton*     split_button;
	CloseButton*     close_button;
	guint            timer_id;
	gboolean         uses_button_tint;
	Settings*        settings;
	EggAlpha*        alpha;
	EggTimeline*     timeline;
};

enum
{
	TIMED_OUT = 0,
	CLOSED,
	STATE_CHANGED,
	FOCUS_CHANGED,
	LAST_SIGNAL
};

#define FPS                60
#define SD_BUBBLE_TIMEOUT  120
#define WINDOW_MAX_OPACITY 1.0f

static guint g_sd_bubble_signals[LAST_SIGNAL] = { 0 };

/*-- internal API ------------------------------------------------------------*/

static void
sd_bubble_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (sd_bubble_parent_class)->dispose (gobject);
}

static void
sd_bubble_finalize (GObject* gobject)
{
	SdBubblePrivate* priv = GET_PRIVATE (SD_BUBBLE (gobject));

	if (priv->icon_pixbuf != NULL)
		g_object_unref (priv->icon_pixbuf);

	if (priv->contracted_bg_surface != NULL)
		cairo_surface_destroy (priv->contracted_bg_surface);

	if (priv->default_bg_surface != NULL)
		cairo_surface_destroy (priv->default_bg_surface);

	if (priv->expanded_bg_surface != NULL)
		cairo_surface_destroy (priv->expanded_bg_surface);

	if (priv->icon_small_surface != NULL)
		cairo_surface_destroy (priv->icon_small_surface);

	if (priv->icon_big_surface != NULL)
		cairo_surface_destroy (priv->icon_big_surface);

	if (priv->summary_surface != NULL)
		cairo_surface_destroy (priv->summary_surface);

	if (priv->body_surface != NULL)
		cairo_surface_destroy (priv->body_surface);

	if (priv->accept_button != NULL)
		button_del (priv->accept_button);

	if (priv->reject_button != NULL)
		button_del (priv->reject_button);

	if (priv->split_button != NULL)
		split_button_del (priv->split_button);

	if (priv->mc != NULL)
		g_object_unref (priv->mc);

	if (priv->timer_id != 0)
		g_source_remove (priv->timer_id);

	if (priv->actions != NULL)
	{
		guint i = 0;

		for (; i < priv->num_actions; i++)
			g_free (priv->actions[i]);

		g_free (priv->actions);
	}

	if (priv->dbus_sender)
		g_string_free (priv->dbus_sender, TRUE);

	if (priv->summary)
		g_string_free (priv->summary, TRUE);

	if (priv->body)
		g_string_free (priv->body, TRUE);

	if (priv->widget)
		gtk_widget_destroy (priv->widget);

	if (priv->redraw_handler_id != 0)
	{
		g_source_remove (priv->redraw_handler_id);
		priv->redraw_handler_id = 0;
	}
		
	if (priv->timer_id != 0)
	{
		g_source_remove (priv->timer_id);
		priv->timer_id = 0;
	}

	if (priv->alpha != NULL)
	{
		g_object_unref (priv->alpha);
		priv->alpha = NULL;
	}

	if (priv->timeline != NULL)
	{
		g_object_unref (priv->timeline);
		priv->timeline = NULL;
	}

	/* chain up to the parent class */
	G_OBJECT_CLASS (sd_bubble_parent_class)->finalize (gobject);
}

static void
sd_bubble_init (SdBubble* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */
}

static void
sd_bubble_get_property (GObject*    gobject,
                        guint       prop,
                        GValue*     value,
                        GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
sd_bubble_class_init (SdBubbleClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (SdBubblePrivate));

	gobject_class->dispose      = sd_bubble_dispose;
	gobject_class->finalize     = sd_bubble_finalize;
	gobject_class->get_property = sd_bubble_get_property;

	g_sd_bubble_signals[TIMED_OUT] = g_signal_new (
		"timed-out",
		G_OBJECT_CLASS_TYPE (gobject_class),
		G_SIGNAL_RUN_LAST,
		G_STRUCT_OFFSET (SdBubbleClass, timed_out),
		NULL,
		NULL,
		g_cclosure_marshal_VOID__VOID,
		G_TYPE_NONE,
		0);

	g_sd_bubble_signals[CLOSED] = g_signal_new (
		"closed",
		G_OBJECT_CLASS_TYPE (gobject_class),
		G_SIGNAL_RUN_LAST,
		G_STRUCT_OFFSET (SdBubbleClass, closed),
		NULL,
		NULL,
		g_cclosure_marshal_VOID__VOID,
		G_TYPE_NONE,
		0);

    g_sd_bubble_signals[STATE_CHANGED] = g_signal_new (
		"state-changed",
		G_OBJECT_CLASS_TYPE (gobject_class),
		G_SIGNAL_RUN_LAST,
		G_STRUCT_OFFSET (SdBubbleClass, state_changed),
		NULL,
		NULL,
		g_cclosure_marshal_VOID__VOID,
		G_TYPE_NONE,
		0);

    g_sd_bubble_signals[FOCUS_CHANGED] = g_signal_new (
		"focus-changed",
		G_OBJECT_CLASS_TYPE (gobject_class),
		G_SIGNAL_RUN_LAST,
		G_STRUCT_OFFSET (SdBubbleClass, focus_changed),
		NULL,
		NULL,
		g_cclosure_marshal_VOID__VOID,
		G_TYPE_NONE,
		0);
}

void
_render_sd_background (SdBubble* self,
                       cairo_t*  cr)
{
	SdBubblePrivate* priv = GET_PRIVATE (self);

	switch (GET_PRIVATE (self)->state)
	{
		case SD_BUBBLE_STATE_CONTRACTED:
			cairo_set_source_surface (cr, priv->contracted_bg_surface, 0.0, 0.0);
			cairo_paint (cr);
		break;

		case SD_BUBBLE_STATE_DEFAULT:
			cairo_set_source_surface (cr, priv->default_bg_surface, 0.0, 0.0);
			cairo_paint (cr);
		break;

		case SD_BUBBLE_STATE_EXPANDED:
			cairo_set_source_surface (cr, priv->expanded_bg_surface, 0.0, 0.0);
			cairo_paint (cr);
		break;

		default :
			g_print ("Outch, no state set yet :(\n");
		break;
	}
}

void
_render_sd_state (SdBubble* self,
                  cairo_t*  cr)
{
	SdBubblePrivate* priv     = GET_PRIVATE (self);
	gdouble          button_y = 0;

	switch (priv->state)
	{
		case SD_BUBBLE_STATE_CONTRACTED:
			cairo_set_source_surface (cr,
			                          priv->icon_small_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->contracted_horiz_padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->contracted_vert_padding);
			cairo_paint (cr);
			cairo_set_source_surface (cr,
			                          priv->summary_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->contracted_icon_size +
			                          2 * priv->settings->background->contracted_horiz_padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding);
			cairo_paint (cr);
			if (priv->hovered)
			{
				close_button_set_position (priv->close_button,
					                       priv->settings->drop_shadow->size +
				                           priv->settings->close_button->position_x,
					                       priv->settings->drop_shadow->size +
				                           priv->settings->close_button->position_y);
				close_button_paint (priv->close_button, cr);
			}
		break;

		case SD_BUBBLE_STATE_DEFAULT:
			cairo_set_source_surface (cr,
			                          priv->icon_big_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding);
			cairo_paint (cr);
			cairo_set_source_surface (cr,
			                          priv->summary_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->icon_size +
			                          2 * priv->settings->background->padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding);
			cairo_paint (cr);
			cairo_set_source_surface (cr,
			                          priv->body_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->icon_size +
			                          2 * priv->settings->background->padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding +
			                          cairo_image_surface_get_height (priv->summary_surface));
			cairo_paint (cr);

			button_y = priv->settings->drop_shadow->size +
				       2 * priv->settings->background->padding;
			button_y += MAX (priv->settings->background->icon_size,
			                 cairo_image_surface_get_height (priv->summary_surface) +
			                 cairo_image_surface_get_height (priv->body_surface));

			if (priv->uses_split_button)
			{
				split_button_set_position (priv->split_button,
				                           priv->settings->drop_shadow->size +
				                           priv->settings->background->icon_size +
				                           2 * priv->settings->background->padding,
					                       button_y);
				split_button_paint (priv->split_button, cr);
			}
			else
			{
				button_set_position (priv->reject_button,
				                     priv->settings->drop_shadow->size +
				                     priv->settings->background->icon_size +
				                     2 * priv->settings->background->padding,
					                 button_y);
				button_paint (priv->reject_button, cr);
			}

			button_set_position (priv->accept_button,
			                     priv->settings->drop_shadow->size +
			                     priv->settings->background->icon_size +
			                     2 * priv->settings->background->padding +
			                     priv->settings->background->button_padding +
			                     priv->settings->button->width,
			                     button_y);
			button_paint (priv->accept_button, cr);
		break;

		case SD_BUBBLE_STATE_EXPANDED:
			cairo_set_source_surface (cr,
			                          priv->icon_big_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding);
			cairo_paint (cr);
			cairo_set_source_surface (cr,
			                          priv->summary_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->icon_size +
			                          2 * priv->settings->background->padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding);
			cairo_paint (cr);
			cairo_set_source_surface (cr,
			                          priv->body_surface,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->icon_size +
			                          2 * priv->settings->background->padding,
			                          priv->settings->drop_shadow->size +
			                          priv->settings->background->padding +
			                          cairo_image_surface_get_height (priv->summary_surface));
			cairo_paint (cr);
			button_y = priv->settings->drop_shadow->size +
				       2 * priv->settings->background->padding;
			button_y += MAX (priv->settings->background->icon_size,
			                 cairo_image_surface_get_height (priv->summary_surface) +
			                 cairo_image_surface_get_height (priv->body_surface));
			split_button_set_position (priv->split_button,
			                           priv->settings->drop_shadow->size +
			                           priv->settings->background->icon_size +
			                           2 * priv->settings->background->padding,
			                           button_y);
			split_button_paint (priv->split_button, cr);
		break;

		default :
			g_print ("Outch, no state set yet :(\n");
		break;
	}
}

static void
_screen_changed_handler (GtkWindow* window,
                         GdkScreen* old_screen,
                         gpointer   data)
{
	GdkScreen* screen = gtk_widget_get_screen (GTK_WIDGET (window));
	GdkVisual* visual = gdk_screen_get_rgba_visual (screen);

	if (!visual)
		visual = gdk_screen_get_system_visual (screen);

	gtk_widget_set_visual (GTK_WIDGET (window), visual);
}

static void
_composited_changed_handler (GtkWidget* window,
                             gpointer   data)
{
	SdBubble* sd_bubble = NULL;

	sd_bubble = (SdBubble*) G_OBJECT (data);

	GET_PRIVATE (sd_bubble)->composited = gdk_screen_is_composited (
						gtk_widget_get_screen (window));
}

void
_recalc_sd_bubble_size (SdBubble* self)
{
	SdBubblePrivate* priv = GET_PRIVATE (self);

	gint new_width  = 0;
	gint new_height = 0;

	switch (priv->state)
	{
		case SD_BUBBLE_STATE_CONTRACTED:
			new_width  = 2.0 * priv->settings->drop_shadow->size +
				         priv->settings->background->width;
			new_height = 2.0 * priv->settings->drop_shadow->size +
				         priv->settings->background->contracted_height;
		break;

		case SD_BUBBLE_STATE_DEFAULT:
			new_width  = 2.0 * priv->settings->drop_shadow->size +
				         priv->settings->background->width;
			new_height = 2.0 * priv->settings->drop_shadow->size +
				         priv->settings->background->padding;
			new_height += MAX (priv->settings->background->icon_size,
			               cairo_image_surface_get_height (priv->summary_surface) +
			               cairo_image_surface_get_height (priv->body_surface));
			new_height += 2 * priv->settings->background->padding;
			new_height += priv->settings->button->height;
		break;

		case SD_BUBBLE_STATE_EXPANDED:
			new_width  = 2.0 * priv->settings->drop_shadow->size +
				         priv->settings->background->width;
			new_height = 2.0 * priv->settings->drop_shadow->size +
				         priv->settings->background->padding;
			new_height += MAX (priv->settings->background->icon_size,
			               cairo_image_surface_get_height (priv->summary_surface) +
			               cairo_image_surface_get_height (priv->body_surface));
			new_height += split_button_get_expanded_height (priv->split_button);
			new_height += 2 * priv->settings->background->padding;
		break;

		default :
			g_print ("Outch, no state set yet :(\n");
		break;
	}

	/* these two are really needed... gtk_window_resize() to make the drawing
	** work and gtk_widget_set_size_request() to have
	** sd_bubble_get_width/height() report the correct values right away */
	gtk_window_resize (GTK_WINDOW (priv->widget), new_width, new_height);
	gtk_widget_set_size_request (priv->widget, new_width, new_height);
}

static gboolean
_draw_handler (GtkWidget* window,
               cairo_t*   cr,
               gpointer   data)
{
	SdBubble* self = NULL;

	self = (SdBubble*) G_OBJECT (data);

	/* clear bubble-background */
	cairo_scale (cr, 1.0f, 1.0f);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

	/* render drop-shadow and sd-bubble background */
	_render_sd_background (self, cr);
    
	/* render content of sd-bubble depending on layout */
	_render_sd_state (self, cr);

	return FALSE;
}

static gboolean
_redraw_handler (SdBubble* self)
{
	GtkWindow*       window = NULL;
	SdBubblePrivate* priv   = NULL;

	if (!self)
		return FALSE;

	if (!sd_bubble_is_visible (self))
		return FALSE;

	priv = GET_PRIVATE (self);

	if (!priv->composited)
		return FALSE;

	window = GTK_WINDOW (priv->widget);

	if (!GTK_IS_WINDOW (window))
		return FALSE;

	gtk_widget_queue_draw (priv->widget);

	return FALSE;
}

static gboolean
_focus_in_handler (GtkWidget*     widget,
                   GdkEventFocus* event,
                   gpointer       data)
{
	SdBubble*        self = NULL;
	SdBubblePrivate* priv = NULL;

	self = (SdBubble*) G_OBJECT (data);
	priv = GET_PRIVATE (self);

	if (priv->state == SD_BUBBLE_STATE_CONTRACTED)
	{
		gint     x      = -1;
		gint     y      = -1;
		gboolean is_hit = FALSE;

		gtk_widget_get_pointer (widget, &x, &y);
		is_hit = close_button_is_hit (priv->close_button, x, y);
		if (is_hit)
		{
			g_source_remove (priv->redraw_handler_id);
			priv->redraw_handler_id = 0;
			g_source_remove (priv->timer_id);
			priv->timer_id = 0;
			dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
					                sd_bubble_get_notification_id (self),
					                3);
			sd_bubble_fade_out (self);
		}
	}

	priv->focused = TRUE;
	g_signal_emit (self, g_sd_bubble_signals[FOCUS_CHANGED], 0);

	return FALSE;
}

static gboolean
_focus_out_handler (GtkWidget*      widget,
                    GdkEventFocus*  event,
                    gpointer        data)
{
	SdBubble*        self = NULL;
	SdBubblePrivate* priv = NULL;

	self = (SdBubble*) G_OBJECT (data);
	priv = GET_PRIVATE (self);

	priv->focused = FALSE;
	g_signal_emit (self, g_sd_bubble_signals[FOCUS_CHANGED], 0);

	return FALSE;
}

static gboolean
_enter_notify_handler (GtkWidget*        widget,
                       GdkEventCrossing* event,
                       gpointer          data)
{
	SdBubble*        self = NULL;
	SdBubblePrivate* priv = NULL;

	self = (SdBubble*) G_OBJECT (data);
	priv = GET_PRIVATE (self);

	priv->hovered = TRUE;
	gtk_widget_queue_draw (widget);

	return FALSE;
}

static gboolean
_leave_notify_handler (GtkWidget*        widget,
                       GdkEventCrossing* event,
                       gpointer          data)
{
	SdBubble*        self = NULL;
	SdBubblePrivate* priv = NULL;

	self = (SdBubble*) G_OBJECT (data);
	priv = GET_PRIVATE (self);

	priv->hovered = FALSE;
	gtk_widget_queue_draw (widget);

	return FALSE;
}

static gboolean
_button_press_handler (GtkWidget*      widget,
                       GdkEventButton* event,
                       gpointer        data)
{
	SdBubble*        self   = NULL;
	SdBubblePrivate* priv   = NULL;
    gboolean         is_hit = FALSE;

	self = (SdBubble*) G_OBJECT (data);
	priv = GET_PRIVATE (self);

	/* don't react on anything but the left mouse-button */
	if (event->button != 1)
		return FALSE;

	/* check for button-hits */
	switch (priv->state)
	{
		case SD_BUBBLE_STATE_CONTRACTED:
		{
			is_hit = close_button_is_hit (priv->close_button,
			                              (gint) event->x,
			                              (gint) event->y);
			if (is_hit)
				close_button_set_state (priv->close_button,
				                        CLOSE_BUTTON_STATE_PRESSED);
		}
		break;

		case SD_BUBBLE_STATE_DEFAULT:
		{
			if (priv->uses_split_button)
			{
				is_hit = split_button_is_hit (priv->split_button,
					                          (gint) event->x,
					                          (gint) event->y);
				if (is_hit)
					split_button_set_state (priv->split_button,
						                    BUTTON_STATE_PRESSED);
			}
			else
			{
				is_hit = button_is_hit (priv->reject_button,
					                    (gint) event->x,
					                    (gint) event->y);
				if (is_hit)
					button_set_state (priv->reject_button,
					                  BUTTON_STATE_PRESSED);
			}

			is_hit = button_is_hit (priv->accept_button,
			                        (gint) event->x,
			                        (gint) event->y);
			if (is_hit)
				button_set_state (priv->accept_button,
				                  BUTTON_STATE_PRESSED);
		}
		break;

		case SD_BUBBLE_STATE_EXPANDED:
		{
			is_hit = split_button_is_hit (priv->split_button,
			                              (gint) event->x,
			                              (gint) event->y);

			if (is_hit)
				split_button_set_state (priv->split_button,
				                        SPLIT_BUTTON_STATE_PRESSED);
		}
		break;

		default :
			// nothing to do
		break;
	}

	gtk_widget_queue_draw (priv->widget);

	return FALSE;
}

gboolean
_button_release_handler (GtkWidget*      widget,
                         GdkEventButton* event,
                         gpointer        data)
{
	SdBubble*        self              = NULL;
	SdBubblePrivate* priv              = NULL;
    gboolean         is_hit            = FALSE;
	gboolean         is_reject_pressed = FALSE;
	gboolean         is_accept_pressed = FALSE;
	gboolean         is_split_pressed  = FALSE;
	gboolean         is_close_pressed  = FALSE;
	ButtonState      state             = BUTTON_STATE_NONE;
	SplitButtonState split_state       = SPLIT_BUTTON_STATE_NONE;
	CloseButtonState close_state       = CLOSE_BUTTON_STATE_NONE;

	/* don't react on anything but the left mouse-button */
	if (event->button != 1)
		return FALSE;

	self = (SdBubble*) G_OBJECT (data);
	priv = GET_PRIVATE (self);

	switch (priv->state)
	{
		case SD_BUBBLE_STATE_CONTRACTED:
			is_close_pressed = close_button_get_state (priv->close_button) == CLOSE_BUTTON_STATE_PRESSED;
			is_hit = close_button_is_hit (priv->close_button,
			                              (gint) event->x,
			                              (gint) event->y);

			if (is_hit && is_close_pressed)
			{
				close_state = CLOSE_BUTTON_STATE_HOVER;
				close_button_set_state (priv->close_button, close_state);

				g_source_remove (priv->redraw_handler_id);
				priv->redraw_handler_id = 0;
				g_source_remove (priv->timer_id);
				priv->timer_id = 0;
				dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
						                sd_bubble_get_notification_id (self),
						                3);
				sd_bubble_fade_out (self);
				gtk_widget_queue_draw (priv->widget);
				return FALSE;
			}
			else if (is_hit && !is_close_pressed)
				close_state = CLOSE_BUTTON_STATE_HOVER;
			else if (!is_hit && is_close_pressed)
				close_state = CLOSE_BUTTON_STATE_NORMAL;
			else if (!is_hit && !is_close_pressed)
				close_state = CLOSE_BUTTON_STATE_NORMAL;

			close_button_set_state (priv->close_button, close_state);

			if (!is_hit)
			{
				priv->focused = TRUE;
				g_signal_emit (self, g_sd_bubble_signals[FOCUS_CHANGED], 0);
			}
		break;


		case SD_BUBBLE_STATE_DEFAULT:
			is_reject_pressed = button_get_state (priv->reject_button) == BUTTON_STATE_PRESSED;
			is_accept_pressed = button_get_state (priv->accept_button) == BUTTON_STATE_PRESSED;

			is_hit = button_is_hit (priv->accept_button,
			                        (gint) event->x,
			                        (gint) event->y);
			if (is_hit && is_accept_pressed)
			{
				state = BUTTON_STATE_HOVER;
				button_set_state (priv->accept_button, state);

				dbus_send_action_signal (sd_bubble_get_dbus_sender (self),
				                         sd_bubble_get_notification_id (self),
				                         sd_bubble_get_action_by_index (self, 0));
				dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
				                        sd_bubble_get_notification_id (self),
				                        3);
				sd_bubble_fade_out (self);
				gtk_widget_queue_draw (priv->widget);
				return FALSE;
			}
			else if (is_hit && !is_accept_pressed)
				state = BUTTON_STATE_HOVER;
			else if (!is_hit && is_accept_pressed)
				state = BUTTON_STATE_NORMAL;
			else if (!is_hit && !is_accept_pressed)
				state = BUTTON_STATE_NORMAL;

			button_set_state (priv->accept_button, state);
			
			if (priv->uses_split_button)
			{
				is_hit = split_button_is_arrow_hit (priv->split_button,
					                                (gint) event->x,
					                                (gint) event->y);
				if (!is_hit)
				{
					is_split_pressed = split_button_get_state (priv->split_button) == SPLIT_BUTTON_STATE_PRESSED;

					is_hit = split_button_is_hit (priv->split_button,
						                          (gint) event->x,
						                          (gint) event->y);
					if (is_hit && is_split_pressed)
					{
						split_state = SPLIT_BUTTON_STATE_HOVER;
						split_button_set_state (priv->split_button, split_state);

						dbus_send_action_signal (sd_bubble_get_dbus_sender (self),
								                 sd_bubble_get_notification_id (self),
								                 sd_bubble_get_action_by_index (self, 2));
						dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
								                sd_bubble_get_notification_id (self),
								                3);
						sd_bubble_fade_out (self);
						gtk_widget_queue_draw (priv->widget);
						return FALSE;
					}
					else if (is_hit && !is_split_pressed)
						split_state = SPLIT_BUTTON_STATE_HOVER;
					else if (!is_hit && is_split_pressed)
						split_state = SPLIT_BUTTON_STATE_NORMAL;
					else if (!is_hit && !is_split_pressed)
						split_state = SPLIT_BUTTON_STATE_NORMAL;

					split_button_set_state (priv->split_button, split_state);
				}
				else
				{
					split_button_set_state (priv->split_button,
					                        SPLIT_BUTTON_STATE_HOVER);

					split_button_set_form (priv->split_button,
						                   SPLIT_BUTTON_FORM_EXPANDED);
					sd_bubble_set_state (self, SD_BUBBLE_STATE_EXPANDED);
				}
			}
			else
			{
				is_hit = button_is_hit (priv->reject_button,
					                    (gint) event->x,
					                    (gint) event->y);
				if (is_hit && is_reject_pressed)
				{
					state = BUTTON_STATE_HOVER;
					button_set_state (priv->reject_button, state);

					dbus_send_action_signal (sd_bubble_get_dbus_sender (self),
						                     sd_bubble_get_notification_id (self),
						                     sd_bubble_get_action_by_index (self, 2));
					dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
						                    sd_bubble_get_notification_id (self),
						                    3);
					sd_bubble_fade_out (self);
					gtk_widget_queue_draw (priv->widget);
					return FALSE;
				}
				else if (is_hit && !is_reject_pressed)
					state = BUTTON_STATE_HOVER;
				else if (!is_hit && is_reject_pressed)
					state = BUTTON_STATE_NORMAL;
				else if (!is_hit && !is_reject_pressed)
					state = BUTTON_STATE_NORMAL;

				button_set_state (priv->reject_button, state);
			}
		break;

		case SD_BUBBLE_STATE_EXPANDED:
		{
			is_split_pressed = split_button_get_state (priv->split_button) == SPLIT_BUTTON_STATE_PRESSED;

			is_hit = split_button_is_arrow_hit (priv->split_button,
			                                    (gint) event->x,
			                                    (gint) event->y);
			if (!is_hit)
			{
				is_hit = split_button_is_hit (priv->split_button,
					                          (gint) event->x,
					                          (gint) event->y);
				if (is_hit && is_split_pressed)
				{
					split_state = SPLIT_BUTTON_STATE_HOVER;

					gint index = 0;
					index = 2 * (split_button_get_hit_label (priv->split_button) + 1);
					dbus_send_action_signal (sd_bubble_get_dbus_sender (self),
						                     sd_bubble_get_notification_id (self),
						                     sd_bubble_get_action_by_index (self, index));
					dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
						                    sd_bubble_get_notification_id (self),
						                    3);
					sd_bubble_fade_out (self);
					gtk_widget_queue_draw (priv->widget);
					return FALSE;
				}
				else if (is_hit && !is_split_pressed)
					split_state = SPLIT_BUTTON_STATE_HOVER;
				else if (!is_hit && is_split_pressed)
					split_state = SPLIT_BUTTON_STATE_NORMAL;
				else if (!is_hit && !is_split_pressed)
					split_state = SPLIT_BUTTON_STATE_NORMAL;

				split_button_set_state (priv->split_button, split_state);
			}
			else
			{
				split_button_set_state (priv->split_button,
				                        SPLIT_BUTTON_STATE_HOVER);

				split_button_set_form (priv->split_button,
				                       SPLIT_BUTTON_FORM_COLLAPSED);
				sd_bubble_set_state (self, SD_BUBBLE_STATE_DEFAULT);
			}
		}
		break;

		default :
			// nothing to do			
		break;
	}

	gtk_widget_queue_draw (priv->widget);

	return FALSE;
}

gboolean
_motion_notify_handler (GtkWidget*      widget,
                        GdkEventMotion* event,
                        gpointer        data)
{
	SdBubble*        self              = NULL;
	SdBubblePrivate* priv              = NULL;
	gboolean         is_hit            = FALSE;
	gboolean         is_reject_pressed = FALSE;
	gboolean         is_accept_pressed = FALSE;
	gboolean         is_split_pressed  = FALSE;
	gboolean         is_close_pressed  = FALSE;
	ButtonState      state             = BUTTON_STATE_NONE;
	SplitButtonState split_state       = SPLIT_BUTTON_STATE_NONE;
	CloseButtonState close_state       = CLOSE_BUTTON_STATE_NONE;

	self = (SdBubble*) G_OBJECT (data);
	priv = GET_PRIVATE (self);

	switch (priv->state)
	{
		case SD_BUBBLE_STATE_CONTRACTED:
		{
			is_close_pressed = close_button_get_state (priv->close_button) == CLOSE_BUTTON_STATE_PRESSED;
			is_hit = close_button_is_hit (priv->close_button,
			                              (gint) event->x,
			                              (gint) event->y);
			if (is_close_pressed && is_hit)
				close_state = CLOSE_BUTTON_STATE_PRESSED;
			else if (is_close_pressed && !is_hit)
				close_state = CLOSE_BUTTON_STATE_PRESSED;
			else if (!is_close_pressed && is_hit)
				close_state = CLOSE_BUTTON_STATE_HOVER;
			else if (!is_close_pressed && !is_hit)
				close_state = CLOSE_BUTTON_STATE_NORMAL;

			close_button_set_state (priv->close_button, close_state);
		}
		break;

		case SD_BUBBLE_STATE_DEFAULT:
		{
			if (priv->uses_split_button)
			{
				is_accept_pressed = button_get_state (priv->accept_button) == BUTTON_STATE_PRESSED;
				is_split_pressed = split_button_get_state (priv->split_button) == SPLIT_BUTTON_STATE_PRESSED;

				is_hit = split_button_is_hit (priv->split_button,
				                              (gint) event->x,
				                              (gint) event->y);
				if (is_hit && is_split_pressed)
					split_state = SPLIT_BUTTON_STATE_PRESSED;
				else if (is_hit && !is_split_pressed)
					split_state = is_accept_pressed ? SPLIT_BUTTON_STATE_NORMAL : SPLIT_BUTTON_STATE_HOVER;
				else if (!is_hit && is_split_pressed)
					split_state = SPLIT_BUTTON_STATE_PRESSED;
				else if (!is_hit && !is_split_pressed)
					split_state = SPLIT_BUTTON_STATE_NORMAL;

				split_button_set_state (priv->split_button, split_state);

				is_hit = button_is_hit (priv->accept_button,
					                    (gint) event->x,
					                    (gint) event->y);
				if (is_accept_pressed && is_hit)
					state = BUTTON_STATE_PRESSED;
				else if (is_accept_pressed && !is_hit)
					state = BUTTON_STATE_PRESSED;
				else if (!is_accept_pressed && is_hit)
						state = is_split_pressed ? BUTTON_STATE_NORMAL : BUTTON_STATE_HOVER;
				else if (!is_accept_pressed && !is_hit)
					state = BUTTON_STATE_NORMAL;

				button_set_state (priv->accept_button, state);
			}
			else
			{
				is_reject_pressed = button_get_state (priv->reject_button) == BUTTON_STATE_PRESSED;
				is_accept_pressed = button_get_state (priv->accept_button) == BUTTON_STATE_PRESSED;
				is_hit = button_is_hit (priv->reject_button,
					                    (gint) event->x,
					                    (gint) event->y);
				if (is_reject_pressed && is_hit)
					state = BUTTON_STATE_PRESSED;
				else if (is_reject_pressed && !is_hit)
					state = BUTTON_STATE_PRESSED;
				else if (!is_reject_pressed && is_hit)
					state = is_accept_pressed ? BUTTON_STATE_NORMAL : BUTTON_STATE_HOVER;
				else if (!is_reject_pressed && !is_hit)
					state = BUTTON_STATE_NORMAL;

				button_set_state (priv->reject_button, state);

				is_hit = button_is_hit (priv->accept_button,
					                    (gint) event->x,
					                    (gint) event->y);
				if (is_accept_pressed && is_hit)
					state = BUTTON_STATE_PRESSED;
				else if (is_accept_pressed && !is_hit)
					state = BUTTON_STATE_PRESSED;
				else if (!is_accept_pressed && is_hit)
						state = is_reject_pressed ? BUTTON_STATE_NORMAL : BUTTON_STATE_HOVER;
				else if (!is_accept_pressed && !is_hit)
					state = BUTTON_STATE_NORMAL;

				button_set_state (priv->accept_button, state);
			}
		}
		break;

		case SD_BUBBLE_STATE_EXPANDED:
		{
			is_accept_pressed = button_get_state (priv->accept_button) == BUTTON_STATE_PRESSED;
			is_split_pressed = split_button_get_state (priv->split_button) == SPLIT_BUTTON_STATE_PRESSED;
			is_hit = split_button_is_hit (priv->split_button,
			                              (gint) event->x,
			                              (gint) event->y);
			if (is_split_pressed && is_hit)
				split_state = SPLIT_BUTTON_STATE_PRESSED;
			else if (is_split_pressed && !is_hit)
				split_state = SPLIT_BUTTON_STATE_PRESSED;
			else if (!is_split_pressed && is_hit)
				split_state = is_accept_pressed ? SPLIT_BUTTON_STATE_NORMAL : SPLIT_BUTTON_STATE_HOVER;
			else if (!is_split_pressed && !is_hit)
				split_state = SPLIT_BUTTON_STATE_NORMAL;

			split_button_set_state (priv->split_button, split_state);
		}
		break;

		default :
			// nothing to do
		break;
	}

	gtk_widget_queue_draw (priv->widget);

	return FALSE;
}

void
_pattern_fill (cairo_t*  cr,
               GdkColor* color,
               gdouble   opacity)
{
	cairo_pattern_t* pattern = NULL;
	cairo_surface_t* surface = NULL;
	cairo_t*         cr_dots = NULL;
	surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 4, 4);
	pattern = cairo_pattern_create_for_surface (surface);
	cr_dots = cairo_create (surface);
	cairo_set_operator (cr_dots, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr_dots);
	cairo_set_operator (cr_dots, CAIRO_OPERATOR_OVER);
	cairo_scale (cr_dots, 1.0, 1.0);
	cairo_set_source_rgba (cr_dots,
	                       (gdouble) color->red / (gdouble) 0xFFFF,
	                       (gdouble) color->green / (gdouble) 0xFFFF,
	                       (gdouble) color->blue / (gdouble) 0xFFFF,
	                       opacity);
	cairo_rectangle (cr_dots, 1.0, 1.0, 1.0, 1.0);
	cairo_rectangle (cr_dots, 3.0, 3.0, 1.0, 1.0);
	cairo_fill (cr_dots);
	cairo_set_source (cr, pattern);
	cairo_pattern_set_extend (pattern, CAIRO_EXTEND_REPEAT);
	cairo_fill_preserve (cr);
	cairo_surface_destroy (surface);
	cairo_pattern_destroy (pattern);
}

void
_fill_spotlight (cairo_t*  cr,
                 gdouble   width,
                 gdouble   height,
                 gdouble   offset_x,
                 gdouble   offset_y,
                 gdouble   x,
                 gdouble   y,
                 gdouble   radius,
                 GdkColor* center_color,
                 gdouble   center_opacity,
                 GdkColor* end_color,
                 gdouble   end_opacity,
                 gdouble   paint_opacity)
{
	cairo_pattern_t* pattern = cairo_pattern_create_radial (x,
	                                                        y,
	                                                        0.0,
	                                                        x,
	                                                        y,
	                                                        radius);
	cairo_surface_t* surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                       width,
	                                                       height);

	cairo_t* cr_spotlight = cairo_create (surface);
	cairo_set_operator (cr_spotlight, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr_spotlight);
	cairo_set_operator (cr_spotlight, CAIRO_OPERATOR_OVER);
	cairo_scale (cr_spotlight, 1.0, 1.0);
	
	cairo_pattern_add_color_stop_rgba (pattern,
	                                   0.0,
	                                   (gdouble) center_color->red / (gdouble) 0xFFFF,
	                                   (gdouble) center_color->green / (gdouble) 0xFFFF,
	                                   (gdouble) center_color->blue / (gdouble) 0xFFFF,
	                                   center_opacity);
	cairo_pattern_add_color_stop_rgba (pattern,
	                                   1.0,
	                                   (gdouble) end_color->red / (gdouble) 0xFFFF,
	                                   (gdouble) end_color->green / (gdouble) 0xFFFF,
	                                   (gdouble) end_color->blue / (gdouble) 0xFFFF,
	                                   end_opacity);
	cairo_set_source (cr_spotlight, pattern);
	cairo_paint_with_alpha (cr_spotlight, paint_opacity);
	cairo_set_source_surface (cr, surface, offset_x, offset_y);
	cairo_fill_preserve (cr);
	cairo_surface_destroy (surface);
	cairo_pattern_destroy (pattern);
	cairo_destroy (cr_spotlight);
}

void
_refresh_sd_background (SdBubble* self)
{
	SdBubblePrivate* priv = GET_PRIVATE (self);

	/* contracted background - shadow */
	cairo_surface_destroy (priv->contracted_bg_surface);
	priv->contracted_bg_surface = NULL;
	gdouble width  = priv->settings->background->width;
	gdouble height = priv->settings->background->contracted_height;
	priv->contracted_bg_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                          width + 2.0 * priv->settings->drop_shadow->size,
	                                                          height + 2.0 * priv->settings->drop_shadow->size);
	cairo_t* cr = cairo_create (priv->contracted_bg_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings->drop_shadow->size +
	                        priv->settings->drop_shadow->offset_x, FALSE),
	                 align (priv->settings->drop_shadow->size +
	                        priv->settings->drop_shadow->offset_y, FALSE),
	                 priv->settings->background->outline_corner_radius,
	                 width,
	                 height);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->drop_shadow->color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->drop_shadow->color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->drop_shadow->color.blue / (gdouble) 0xFFFF,
	                       priv->settings->drop_shadow->opacity);
	cairo_fill_preserve (cr);
	raico_blur_t* blur = NULL;
	blur = raico_blur_create (RAICO_BLUR_QUALITY_LOW);
	raico_blur_set_radius (blur, (guint) priv->settings->drop_shadow->blur);
	raico_blur_apply (blur, priv->contracted_bg_surface);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings->drop_shadow->size, FALSE),
	                 align (priv->settings->drop_shadow->size, FALSE),
	                 priv->settings->background->outline_corner_radius,
	                 width,
	                 height);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_fill_preserve (cr);

	/* contracted background - pattern */
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	_pattern_fill (cr,
	               &priv->settings->background->pattern_color,
	               priv->settings->background->pattern_opacity);

	/* contracted background - fill */
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->fill_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->fill_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->fill_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->fill_opacity);
	cairo_fill (cr);

	/* contracted background - outline */
	draw_round_rect (cr,
                 1.0,
                 align (priv->settings->drop_shadow->size, TRUE),
                 align (priv->settings->drop_shadow->size, TRUE),
                 priv->settings->background->outline_corner_radius,
                 width - priv->settings->background->outline_thickness,
                 height - priv->settings->background->outline_thickness);
	cairo_set_line_width (cr, priv->settings->background->outline_thickness);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->outline_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->outline_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->outline_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->outline_opacity);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_stroke_preserve (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	cairo_stroke (cr);

	/* contracted background - inner fill/outline */
	draw_round_rect (cr,
                 1.0,
                 align (priv->settings->drop_shadow->size +
                        priv->settings->background->outline_thickness +
                        priv->settings->background->inner_outline_margin, TRUE),
                 align (priv->settings->drop_shadow->size +
                        priv->settings->background->outline_thickness +
                        priv->settings->background->inner_outline_margin, TRUE),
                 priv->settings->background->outline_corner_radius,
                 width -
	             3.0 * priv->settings->background->outline_thickness -
                 2.0 * priv->settings->background->inner_outline_margin,
                 height -
	             3.0 * priv->settings->background->outline_thickness -
	             2.0 * priv->settings->background->inner_outline_margin);
	cairo_set_line_width (cr, priv->settings->background->inner_outline_thickness);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->inner_outline_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->inner_outline_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->inner_outline_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->inner_outline_opacity);
	cairo_stroke_preserve (cr);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->inner_fill_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->inner_fill_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->inner_fill_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->inner_fill_opacity);	
	cairo_fill (cr);

	cairo_destroy (cr);
	cr = NULL;

	/* default background - shadow */
	cairo_surface_destroy (priv->default_bg_surface);
	priv->default_bg_surface = NULL;
	height = priv->settings->background->padding;
	height += MAX (priv->settings->background->icon_size,
	               cairo_image_surface_get_height (priv->summary_surface) +
	               cairo_image_surface_get_height (priv->body_surface));
	height += 2.0 * priv->settings->background->padding + priv->settings->button->height;
	priv->default_bg_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                       width + 2.0 * priv->settings->drop_shadow->size,
	                                                       height + 2.0 * priv->settings->drop_shadow->size);
	cr = cairo_create (priv->default_bg_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings->drop_shadow->size +
	                        priv->settings->drop_shadow->offset_x, FALSE),
	                 align (priv->settings->drop_shadow->size +
	                        priv->settings->drop_shadow->offset_y, FALSE),
	                 priv->settings->background->outline_corner_radius,
	                 width,
	                 height);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->drop_shadow->color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->drop_shadow->color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->drop_shadow->color.blue / (gdouble) 0xFFFF,
	                       priv->settings->drop_shadow->opacity);
	cairo_fill_preserve (cr);
	raico_blur_apply (blur, priv->default_bg_surface);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings->drop_shadow->size, FALSE),
	                 align (priv->settings->drop_shadow->size, FALSE),
	                 priv->settings->background->outline_corner_radius,
	                 width,
	                 height);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_fill_preserve (cr);

	/* default background - pattern */
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	_pattern_fill (cr,
	               &priv->settings->background->pattern_color,
	               priv->settings->background->pattern_opacity);

	/* default background - fill */
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->fill_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->fill_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->fill_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->fill_opacity);
	cairo_fill_preserve (cr);

	/* default background - spotlight */
	_fill_spotlight (cr,
	                 width,
	                 height,
	                 priv->settings->drop_shadow->size,
	                 priv->settings->drop_shadow->size,
	                 priv->settings->background->spotlight_position_x,
	                 priv->settings->background->spotlight_position_y,
	                 priv->settings->background->spotlight_radius,
	                 &priv->settings->background->spotlight_center_color,
	                 priv->settings->background->spotlight_center_opacity,
	                 &priv->settings->background->spotlight_end_color,
	                 priv->settings->background->spotlight_end_opacity,
	                 priv->settings->background->spotlight_opacity);

	/* default background - outline */
	draw_round_rect (cr,
                 1.0,
                 align (priv->settings->drop_shadow->size, TRUE),
                 align (priv->settings->drop_shadow->size, TRUE),
                 priv->settings->background->outline_corner_radius,
                 width - priv->settings->background->outline_thickness,
                 height - priv->settings->background->outline_thickness);
	cairo_set_line_width (cr, priv->settings->background->outline_thickness);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->outline_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->outline_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->outline_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->outline_opacity);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_stroke_preserve (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	cairo_stroke (cr);
	cairo_destroy (cr);
	cr = NULL;

	/* exit early if the bubble doesn't use a split-button at all*/
	if (!priv->uses_split_button)
		return;

	/* expanded background - shadow */
	cairo_surface_destroy (priv->expanded_bg_surface);
	priv->expanded_bg_surface = NULL;
	height = metric_converter_em2px (priv->mc, 1.0);
	height += MAX (metric_converter_em2px (priv->mc, 3.0),
		           cairo_image_surface_get_height (priv->summary_surface) +
		           cairo_image_surface_get_height (priv->body_surface));
	height += split_button_get_expanded_height (priv->split_button);
	height += metric_converter_em2px (priv->mc, 2.0);
	priv->expanded_bg_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                        width + 2.0 * priv->settings->drop_shadow->size,
	                                                        height + 2.0 * priv->settings->drop_shadow->size);
	cr = cairo_create (priv->expanded_bg_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings->drop_shadow->size +
	                        priv->settings->drop_shadow->offset_x, FALSE),
	                 align (priv->settings->drop_shadow->size +
	                        priv->settings->drop_shadow->offset_y, FALSE),
	                 priv->settings->background->outline_corner_radius,
	                 width,
	                 height);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->drop_shadow->color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->drop_shadow->color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->drop_shadow->color.blue / (gdouble) 0xFFFF,
	                       priv->settings->drop_shadow->opacity);
	cairo_fill_preserve (cr);
	raico_blur_apply (blur, priv->expanded_bg_surface);
	draw_round_rect (cr,
	                 1.0,
	                 align (priv->settings->drop_shadow->size, FALSE),
	                 align (priv->settings->drop_shadow->size, FALSE),
	                 priv->settings->background->outline_corner_radius,
	                 width,
	                 height);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_fill_preserve (cr);

	/* expanded background - pattern */
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	_pattern_fill (cr,
	               &priv->settings->background->pattern_color,
	               priv->settings->background->pattern_opacity);

	/* expanded background - fill */
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->fill_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->fill_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->fill_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->fill_opacity);
	cairo_fill_preserve (cr);

	/* expanded background - spotlight */
	_fill_spotlight (cr,
	                 width,
	                 height,
	                 priv->settings->drop_shadow->size,
	                 priv->settings->drop_shadow->size,
	                 priv->settings->background->spotlight_position_x,
	                 priv->settings->background->spotlight_position_y,
	                 priv->settings->background->spotlight_radius,
	                 &priv->settings->background->spotlight_center_color,
	                 priv->settings->background->spotlight_center_opacity,
	                 &priv->settings->background->spotlight_end_color,
	                 priv->settings->background->spotlight_end_opacity,
	                 priv->settings->background->spotlight_opacity);

	/* expanded background - outline */
	draw_round_rect (cr,
                 1.0,
                 align (priv->settings->drop_shadow->size, TRUE),
                 align (priv->settings->drop_shadow->size, TRUE),
                 priv->settings->background->outline_corner_radius,
                 width - priv->settings->background->outline_thickness,
                 height - priv->settings->background->outline_thickness);
	cairo_set_line_width (cr, priv->settings->background->outline_thickness);
	cairo_set_source_rgba (cr,
	                       (gdouble) priv->settings->background->outline_color.red / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->outline_color.green / (gdouble) 0xFFFF,
	                       (gdouble) priv->settings->background->outline_color.blue / (gdouble) 0xFFFF,
	                       priv->settings->background->outline_opacity);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_stroke_preserve (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	cairo_stroke (cr);
	cairo_destroy (cr);
}

void
_refresh_sd_icon (SdBubble* self)
{
	SdBubblePrivate* priv = GET_PRIVATE (self);

	/* small icon */
	cairo_surface_destroy (priv->icon_small_surface);
	priv->icon_small_surface = NULL;
	gdouble width  = priv->settings->background->contracted_icon_size;
	gdouble height = width;
	priv->icon_small_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                       width,
	                                                       height);
	cairo_t* cr = cairo_create (priv->icon_small_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	if (priv->icon_pixbuf)
	{
		GdkPixbuf* pixbuf = gdk_pixbuf_scale_simple (priv->icon_pixbuf,
		                                             (gint) width,
		                                             (gint) height,
		                                             GDK_INTERP_HYPER);
		gdk_cairo_set_source_pixbuf (cr, pixbuf, 0.0, 0.0);
		cairo_paint (cr);
		g_object_unref (pixbuf);
	}
	cairo_destroy (cr);
	cr = NULL;

	/* big icon */
	cairo_surface_destroy (priv->icon_big_surface);
	priv->icon_big_surface = NULL;
	width  = priv->settings->background->icon_size;
	height = width;
	priv->icon_big_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                     width,
	                                                     height);
	cr = cairo_create (priv->icon_big_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	if (priv->icon_pixbuf)
	{
		GdkPixbuf* pixbuf = gdk_pixbuf_scale_simple (priv->icon_pixbuf,
		                                             (gint) width,
		                                             (gint) height,
		                                             GDK_INTERP_HYPER);
		gdk_cairo_set_source_pixbuf (cr, pixbuf, 0.0, 0.0);
		cairo_paint (cr);
		g_object_unref (pixbuf);
	}
	cairo_destroy (cr);
}

void
_refresh_sd_summary (SdBubble* self)
{
	SdBubblePrivate* priv             = GET_PRIVATE (self);
	gdouble          summary_color[4] = {1.0, 1.0, 1.0, 1.0};
	gdouble          width            = 0.0;
	gdouble          height           = 0.0;
	PangoWeight      summary_weight   = PANGO_WEIGHT_NORMAL;

	setup_color (summary_color,
	             &priv->settings->text->summary_color,
	             priv->settings->text->summary_opacity);
	setup_font_weight (&summary_weight, priv->settings->text->summary_weight);

	width  = priv->settings->background->text_width;
	if (!priv->summary)
		height = metric_converter_em2px (priv->mc, 1.0);
	else
	{
		height = (gdouble) get_text_height (priv->summary->str,
			                                width,
			                                1,
			                                priv->settings->text->summary_family,
			                                priv->settings->text->summary_size,
			                                summary_weight);
	}

	/* title text */
	cairo_surface_destroy (priv->summary_surface);
	priv->summary_surface = NULL;	
	priv->summary_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                    width,
	                                                    height);
	cairo_t* cr = cairo_create (priv->summary_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

	if (priv->summary)
		draw_text (cr,
		           priv->summary->str,
		           0,
		           0,
		           width,
		           height,
		           1,
		           priv->settings->text->summary_family,
		           priv->settings->text->summary_size,
		           summary_weight,
		           summary_color);
	cairo_destroy (cr);
}

void
_refresh_sd_body (SdBubble* self)
{
	SdBubblePrivate* priv          = GET_PRIVATE (self);
	gdouble          body_color[4] = {1.0, 1.0, 1.0, 1.0};
	gdouble          width         = 0.0;
	gdouble          height        = 0.0;
	PangoWeight      body_weight   = PANGO_WEIGHT_NORMAL;

	setup_color (body_color,
	             &priv->settings->text->body_color,
	             priv->settings->text->body_opacity);
	setup_font_weight (&body_weight, priv->settings->text->body_weight);

	width  = priv->settings->background->text_width;
	if (!priv->body)
		height = metric_converter_em2px (priv->mc, 1.0);
	else
	{
		height = (gdouble) get_text_height (priv->body->str,
			                                width,
			                                priv->settings->text->body_max_lines,
			                                priv->settings->text->body_family,
			                                priv->settings->text->body_size,
			                                body_weight);
	}

	/* body text */
	cairo_surface_destroy (priv->body_surface);
	priv->body_surface = NULL;
	priv->body_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                 width,
	                                                 height);
	cairo_t* cr = cairo_create (priv->body_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);

	if (priv->body)
		draw_text (cr,
		           priv->body->str,
		           0,
		           0,
			       width,
			       height,
		           priv->settings->text->body_max_lines,
			       priv->settings->text->body_family,
			       priv->settings->text->body_size,
			       body_weight,
			       body_color);
	cairo_destroy (cr);
}

void
_refresh_sd_buttons (SdBubble* self)
{
	SdBubblePrivate* priv = GET_PRIVATE (self);

	/* right button */
	if (priv->accept_button != NULL)
		button_del (priv->accept_button);

	priv->accept_button = button_new (0,
	                                  0,
	                                  priv->settings->button->width,
	                                  priv->settings->button->height,
	                                  TRUE,
									  priv->settings->button,
	                                  sd_bubble_get_action_by_index (self, 1));
	button_set_button_tint (priv->accept_button,
	                        sd_bubble_get_button_tint (self));

	/* left button */
	if (priv->uses_split_button)
	{
		if (priv->split_button != NULL)
			split_button_del (priv->split_button);

		priv->split_button = split_button_new (0,
			                                   0,
		                                       priv->settings->button,
		                                       priv->settings->split_button,
			                                   &priv->actions[2],
			                                   priv->num_actions - 2);
	}
	else
	{
		if (priv->reject_button != NULL)
			button_del (priv->reject_button);

		priv->reject_button = button_new (0,
			                              0,
			                              priv->settings->button->width,
			                              priv->settings->button->height,
		                                  FALSE,
		                                  priv->settings->button,
			                              sd_bubble_get_action_by_index (self, 3));
		button_set_button_tint (priv->reject_button, FALSE);
	}

	/* close button */
	if (priv->close_button != NULL)
		close_button_del (priv->close_button);

	priv->close_button = close_button_new (priv->settings->close_button);
}

gboolean
_check_num_actions (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), FALSE);

	SdBubblePrivate* priv = GET_PRIVATE (self);

	if (priv->num_actions == 4)
		return FALSE;
	else if (priv->num_actions == 8  ||
	         priv->num_actions == 10 ||
	         priv->num_actions == 12)
		return TRUE;
	else
		g_print ("%s() - Uses too many or too few actions!", G_STRFUNC);

	return FALSE;
}

/*-- public API --------------------------------------------------------------*/

SdBubble*
sd_bubble_new (gchar**   actions,
               guint     size,
               Settings* settings)
{
	g_return_val_if_fail (actions && size >= 4 && settings, NULL);

	SdBubble*        self        = NULL;
	GtkWidget*       window      = NULL;
	SdBubblePrivate* priv        = NULL;
	GdkRGBA          transparent = {0.0, 0.0, 0.0, 0.0};

	self = g_object_new (SD_BUBBLE_TYPE, NULL);
	if (!self)
		return NULL;

	sd_bubble_set_actions (self, actions, size);

	priv = GET_PRIVATE (self);

	priv->widget = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	window = priv->widget;
	if (!window)
		return NULL;
	
	gtk_window_set_type_hint (GTK_WINDOW (window),
				  GDK_WINDOW_TYPE_HINT_NOTIFICATION);
	gtk_window_set_skip_pager_hint (GTK_WINDOW (window), TRUE);
	gtk_window_set_skip_taskbar_hint (GTK_WINDOW (window), TRUE);
	gtk_window_stick (GTK_WINDOW (window));

	gtk_widget_set_events (window,
	                       GDK_POINTER_MOTION_MASK |
	                       GDK_BUTTON_PRESS_MASK   |
	                       GDK_BUTTON_RELEASE_MASK |
	                       GDK_ENTER_NOTIFY_MASK   |
	                       GDK_LEAVE_NOTIFY_MASK);

	gtk_window_move (GTK_WINDOW (window), 0, 0);

	// make sure the window opens with a RGBA-visual
	_screen_changed_handler (GTK_WINDOW (window), NULL, NULL);
	gtk_widget_realize (window);
	gdk_window_set_background_rgba (gtk_widget_get_window (window),
									&transparent);


	// "clear" input-mask, set title/icon/attributes
	gtk_widget_set_app_paintable (window, TRUE);
	gtk_window_set_title (GTK_WINDOW (window), "notify-osd");
	gtk_window_set_decorated (GTK_WINDOW (window), FALSE);
	gtk_window_set_keep_above (GTK_WINDOW (window), TRUE);
	gtk_window_set_resizable (GTK_WINDOW (window), TRUE);
	gtk_window_set_focus_on_map (GTK_WINDOW (window), FALSE);
	gtk_window_set_accept_focus (GTK_WINDOW (window), TRUE);
	gtk_window_set_opacity (GTK_WINDOW (window), 1.0f);

	// TODO: fold some of that back into bubble_init
	self->priv             = GET_PRIVATE (self);
	self->priv->composited = gdk_screen_is_composited (gtk_widget_get_screen (window));
	self->priv->focused                  = FALSE;
	self->priv->hovered                  = FALSE;
	self->priv->visible                  = FALSE;
	self->priv->icon_pixbuf              = NULL;
	self->priv->summary                  = NULL;
	self->priv->body                     = NULL;
	self->priv->dbus_sender              = NULL;
	self->priv->icon_small_surface       = NULL;
	self->priv->icon_big_surface         = NULL;
	self->priv->summary_surface          = NULL;
	self->priv->body_surface             = NULL;
	self->priv->accept_button            = NULL;
	self->priv->reject_button            = NULL;
	self->priv->split_button             = NULL;
	self->priv->close_button             = NULL;
	self->priv->mc                       = metric_converter_new ();
	self->priv->timer_id                 = 0;
	self->priv->uses_split_button        = _check_num_actions (self);
	self->priv->uses_button_tint         = FALSE;
	self->priv->settings                 = settings;
	self->priv->alpha                    = NULL;
	self->priv->timeline                 = NULL;

	/* hook up input/event handlers to window */
	g_signal_connect (G_OBJECT (window),
	                  "draw",
	                  G_CALLBACK (_draw_handler),
	                  self);
	g_signal_connect (G_OBJECT (window),
	                  "screen-changed",
	                  G_CALLBACK (_screen_changed_handler),
	                  self);
	g_signal_connect (G_OBJECT (window),
	                  "composited-changed",
	                  G_CALLBACK (_composited_changed_handler),
	                  self);
	/*g_signal_connect (G_OBJECT (window),
	                  "focus-in-event",
	                  G_CALLBACK (_focus_in_handler),
	                  self);
	g_signal_connect (G_OBJECT (window),
	                  "focus-out-event",
	                  G_CALLBACK (_focus_out_handler),
	                  self);*/
	g_signal_connect (G_OBJECT (window),
	                  "enter-notify-event",
	                  G_CALLBACK (_enter_notify_handler),
	                  self);
	g_signal_connect (G_OBJECT (window),
	                  "leave-notify-event",
	                  G_CALLBACK (_leave_notify_handler),
	                  self);
	g_signal_connect (G_OBJECT (window),
	                  "button-press-event",
	                  G_CALLBACK (_button_press_handler),
	                  self);
	g_signal_connect (G_OBJECT (window),
	                  "button-release-event",
	                  G_CALLBACK (_button_release_handler),
	                  self);
	g_signal_connect (G_OBJECT (window),
	                  "motion-notify-event",
	                  G_CALLBACK (_motion_notify_handler),
	                  self);

	_refresh_sd_icon (self);
	_refresh_sd_body (self);
	_refresh_sd_summary (self);
	_refresh_sd_buttons (self);
	_refresh_sd_background (self);

	return self;
}

const gchar*
sd_bubble_get_summary (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), NULL);

	return GET_PRIVATE (self)->summary->str;
}

void
sd_bubble_set_summary (SdBubble* self,
                       gchar*    summary_text)
{
	g_return_if_fail (self && IS_SD_BUBBLE (self) && summary_text);

	SdBubblePrivate* priv = GET_PRIVATE (self);

	if (priv->summary)
	{
		g_string_free (priv->summary, TRUE);
		priv->summary = NULL;
	}

	priv->summary = g_string_new (summary_text);
	_refresh_sd_summary (self);
}

const gchar*
sd_bubble_get_body (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), NULL);

	return GET_PRIVATE (self)->body->str;
}

void
sd_bubble_set_body (SdBubble* self,
                    gchar*    body_text)
{
	g_return_if_fail (self && IS_SD_BUBBLE (self) && body_text);

	SdBubblePrivate* priv = GET_PRIVATE (self);

	if (priv->body)
	{
		g_string_free (priv->body, TRUE);
		priv->body = NULL;
	}

	priv->body = g_string_new (body_text);
	_refresh_sd_body (self);
	_refresh_sd_background (self);
}

void
sd_bubble_set_icon_pixbuf (SdBubble*        self,
                           const GdkPixbuf* icon_pixbuf)
{
	g_return_if_fail (self && IS_SD_BUBBLE (self) && icon_pixbuf);

	SdBubblePrivate* priv = GET_PRIVATE (self);

	if (priv->icon_pixbuf)
	{
		g_object_unref (priv->icon_pixbuf);
		priv->icon_pixbuf = NULL;
	}

	priv->icon_pixbuf = gdk_pixbuf_copy (icon_pixbuf);
	_refresh_sd_icon (self);
}

gboolean
sd_bubble_set_actions (SdBubble* self,
                       gchar**   actions,
                       guint     size)
{
	g_return_val_if_fail (self &&
	                      IS_SD_BUBBLE (self) &&
	                      actions &&
	                      size > 0,
	                      FALSE);

	SdBubblePrivate* priv   = GET_PRIVATE (self);
	guint            i      = 0;

	if (priv->actions != NULL)
	{
		for (i = 0; i < priv->num_actions; i++)
			g_free (priv->actions[i]);

		g_free (priv->actions);
		priv->actions = NULL;
		priv->num_actions = 0;
	}

	priv->num_actions = size;
	priv->actions = (gchar**) g_malloc0 (priv->num_actions * sizeof (gchar*));
	for (i = 0; i < priv->num_actions; i++)
		priv->actions[i] = g_strdup (actions[i]);

	return TRUE;
}

guint
sd_bubble_get_num_actions (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), 0);

	return GET_PRIVATE (self)->num_actions;
}

const gchar*
sd_bubble_get_action_by_index (SdBubble* self,
                               guint     index)
{
	g_return_val_if_fail (self &&
	                      IS_SD_BUBBLE (self) &&
	                      GET_PRIVATE (self)->num_actions - 1 >= index,
	                      NULL);

	return GET_PRIVATE (self)->actions[index];
}

gint
sd_bubble_get_x (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), -1);

	gint x;
	gtk_window_get_position (GTK_WINDOW (GET_PRIVATE (self)->widget), &x, NULL);

	return x;
}

gint
sd_bubble_get_y (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), -1);

	gint y;
	gtk_window_get_position (GTK_WINDOW (GET_PRIVATE (self)->widget), NULL, &y);

	return y;
}

gint
sd_bubble_get_width (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), 0);

	gint width = 0;

	gtk_widget_get_size_request (GET_PRIVATE (self)->widget, &width, NULL);

	return width;
}

gint
sd_bubble_get_height (SdBubble* self)
{
	if (!self | !IS_SD_BUBBLE (self))
		return 0;

	gint height = 0;

	gtk_widget_get_size_request (GET_PRIVATE (self)->widget, NULL, &height);

	return height;
}

gboolean
sd_bubble_set_state (SdBubble*     self,
                     SdBubbleState state)
{
	SdBubblePrivate* priv = NULL;

	g_return_val_if_fail (self && IS_SD_BUBBLE (self), TRUE);

	if (sd_bubble_get_state (self) == state)
		return TRUE;

	priv = GET_PRIVATE (self);

	if (state == SD_BUBBLE_STATE_CONTRACTED && priv->uses_split_button)
		split_button_set_form (priv->split_button, SPLIT_BUTTON_FORM_COLLAPSED);

	priv->state = state;

	_recalc_sd_bubble_size (self);
	_update_input_shape (priv->widget,
						 (guint) priv->settings->drop_shadow->size,
						 (guint) priv->settings->drop_shadow->size,
						 (guint) sd_bubble_get_width (self) - 2 * (guint) priv->settings->drop_shadow->size,
						 (guint) sd_bubble_get_height (self) - 2 * (guint) priv->settings->drop_shadow->size);
	gtk_widget_queue_draw (priv->widget);

	g_signal_emit (self, g_sd_bubble_signals[STATE_CHANGED], 0);

	return FALSE;
}

SdBubbleState
sd_bubble_get_state (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), SD_BUBBLE_STATE_NONE);

	return GET_PRIVATE(self)->state;
}

gboolean
sd_bubble_set_notification_id (SdBubble* self,
                               guint     notification_id)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), FALSE);

	GET_PRIVATE (self)->notification_id = notification_id;

	return TRUE;
}

guint
sd_bubble_get_notification_id (SdBubble* self)
{
	if (!self || !IS_SD_BUBBLE (self))
		return 0;

	return GET_PRIVATE(self)->notification_id;
}

void
sd_bubble_show (SdBubble* self)
{
	SdBubblePrivate* priv = NULL;

	g_return_if_fail (self && IS_SD_BUBBLE (self));

	priv = GET_PRIVATE (self);

	priv->visible = TRUE;
	gtk_widget_show_all (priv->widget);

	// FIXME: do nasty busy-polling rendering in the drawing-area
	priv->redraw_handler_id = g_timeout_add (1000/FPS,
	                                         (GSourceFunc) _redraw_handler,
	                                         self);

	/* disable this if _fade_in_completed_cb() is used */
	sd_bubble_start_timer (self);
}

void
sd_bubble_close (SdBubble* self)
{
	g_return_if_fail (self && IS_SD_BUBBLE (self));

	gtk_widget_hide (GET_PRIVATE (self)->widget);
	GET_PRIVATE (self)->visible = FALSE;
	dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
	                        sd_bubble_get_notification_id (self),
	                        1);

	g_signal_emit (self, g_sd_bubble_signals[CLOSED], 0);
}

void
sd_bubble_move (SdBubble* self,
                gint      x,
                gint      y)
{
	if (!self || !IS_SD_BUBBLE (self))
		return;

	gtk_window_move (GTK_WINDOW (GET_PRIVATE (self)->widget), x, y);
}

gboolean
sd_bubble_is_visible (SdBubble* self)
{
	if (!self || !IS_SD_BUBBLE (self))
		return FALSE;

	return GET_PRIVATE (self)->visible;
}

gboolean
sd_bubble_is_focused (SdBubble* self)
{
	if (!self || !IS_SD_BUBBLE (self))
		return FALSE;

	return GET_PRIVATE (self)->focused;
}

gboolean
sd_bubble_is_hovered (SdBubble* self)
{
	if (!self || !IS_SD_BUBBLE (self))
		return FALSE;

	return GET_PRIVATE (self)->hovered;
}

void
sd_bubble_start_timer (SdBubble* self)
{
	SdBubblePrivate* priv = NULL;

	g_return_if_fail (self && IS_SD_BUBBLE (self));

	priv = GET_PRIVATE (self);

	if (priv->timer_id > 0)
		g_source_remove (priv->timer_id);

	priv->timer_id = g_timeout_add_seconds (SD_BUBBLE_TIMEOUT,
	                                        (GSourceFunc) sd_bubble_timed_out,
	                                        self);
}

gboolean
sd_bubble_timed_out (SdBubble* self)
{
	SdBubblePrivate* priv = NULL;

	g_return_val_if_fail (self && IS_SD_BUBBLE (self), FALSE);

	priv = GET_PRIVATE (self);

	g_source_remove (priv->redraw_handler_id);
	sd_bubble_fade_out (self);

	dbus_send_close_signal (sd_bubble_get_dbus_sender (self),
	                        sd_bubble_get_notification_id (self),
	                        1);

	g_signal_emit (self, g_sd_bubble_signals[TIMED_OUT], 0);

	return FALSE;
}

void
sd_bubble_set_dbus_sender (SdBubble*    self,
                           const gchar* dbus_sender)
{
	SdBubblePrivate* priv = NULL;

	g_return_if_fail (self && IS_SD_BUBBLE (self) && dbus_sender);

	priv = GET_PRIVATE (self);

	if (priv->dbus_sender != NULL)
	{
		g_string_free (priv->dbus_sender, TRUE);
		priv->dbus_sender = NULL;
	}

	priv->dbus_sender = g_string_new (dbus_sender);
}

gchar*
sd_bubble_get_dbus_sender (SdBubble* self)
{
	if (!self || !IS_SD_BUBBLE (self))
		return NULL;

	return GET_PRIVATE (self)->dbus_sender->str;
}

void
sd_bubble_set_button_tint (SdBubble* self,
                           gboolean  uses_tint)
{
	g_return_if_fail (self && IS_SD_BUBBLE (self));

	GET_PRIVATE (self)->uses_button_tint = uses_tint;
	_refresh_sd_buttons (self);
}

gboolean
sd_bubble_get_button_tint (SdBubble* self)
{
	g_return_val_if_fail (self && IS_SD_BUBBLE (self), FALSE);

	return GET_PRIVATE (self)->uses_button_tint;
}

void
_fade_cb (EggTimeline* timeline,
          gint         frame_no,
          SdBubble*    self)
{
	float            opacity = 0.0;
	SdBubblePrivate* priv    = NULL;

	g_return_if_fail (IS_SD_BUBBLE (self));

	priv = GET_PRIVATE (self);

	opacity = (float) egg_alpha_get_alpha (priv->alpha) /
			  (float) EGG_ALPHA_MAX_ALPHA *
			  WINDOW_MAX_OPACITY;

	gtk_window_set_opacity (GTK_WINDOW (priv->widget), opacity);
}

void
_fade_in_completed_cb (EggTimeline* timeline,
                       SdBubble*    self)
{
	SdBubblePrivate* priv = NULL;

	g_return_if_fail (IS_SD_BUBBLE (self));

	priv = GET_PRIVATE (self);

	if (priv->alpha)
	{
		g_object_unref (priv->alpha);
		priv->alpha = NULL;
	}

	if (priv->timeline)
	{
		g_object_unref (priv->timeline);
		priv->timeline = NULL;
	}

	gtk_window_set_opacity (GTK_WINDOW (priv->widget), WINDOW_MAX_OPACITY);

	sd_bubble_start_timer (self);
}

static void
_fade_out_completed_cb (EggTimeline* timeline,
                        SdBubble*    self)
{
	g_return_if_fail (IS_SD_BUBBLE (self));

	sd_bubble_close (self);
}

void
_update_input_shape (GtkWidget* window,
                     guint      offset_x,
                     guint      offset_y,
                     guint      width,
                     guint      height)
{
	cairo_region_t*       region = NULL;
	cairo_rectangle_int_t rect   = {offset_x, offset_y, width, height};

	/* sanity check */
	if (!window)
		return;
	
	/* set input-region to ignore the drop-shadow area surrounding a
	** snap-decision bubble */ 
	region = cairo_region_create_rectangle (&rect);
	if (cairo_region_status (region) == CAIRO_STATUS_SUCCESS)
	{
		/* clear any old input-region */
		gtk_widget_input_shape_combine_region (window, NULL);

		/* set new input-region */
		gtk_widget_input_shape_combine_region (window, region);
	}
	cairo_region_destroy (region);
}

void
sd_bubble_fade_in (SdBubble* self)
{
	SdBubblePrivate* priv = NULL;

	g_return_if_fail (IS_SD_BUBBLE (self));

	priv = GET_PRIVATE (self);

	priv->timeline = egg_timeline_new_for_duration (UNITY_SLOW_BEAT);
	egg_timeline_set_speed (priv->timeline, FPS);

	if (priv->alpha != NULL)
	{
		g_object_unref (priv->alpha);
		priv->alpha = NULL;
	}

	priv->alpha = egg_alpha_new_full (priv->timeline,
	                                  EGG_ALPHA_RAMP_INC,
	                                  NULL,
	                                  NULL);

	g_signal_connect (G_OBJECT (priv->timeline),
	                  "completed",
	                  G_CALLBACK (_fade_in_completed_cb),
	                  self);

	g_signal_connect (G_OBJECT (priv->timeline),
	                  "new-frame",
	                  G_CALLBACK (_fade_cb),
	                  self);

	egg_timeline_start (priv->timeline);

	gtk_window_set_opacity (GTK_WINDOW (priv->widget), 1.0f);

	sd_bubble_show (self);
}

void
sd_bubble_fade_out (SdBubble* self)
{
	SdBubblePrivate* priv = NULL;

	g_return_if_fail (IS_SD_BUBBLE (self));

	priv = GET_PRIVATE (self);

	priv->timeline = egg_timeline_new_for_duration (UNITY_FAST_BEAT);
	egg_timeline_set_speed (priv->timeline, FPS);

	if (priv->alpha != NULL)
	{
		g_object_unref (priv->alpha);
		priv->alpha = NULL;
	}

	priv->alpha = egg_alpha_new_full (priv->timeline,
	                                  EGG_ALPHA_RAMP_DEC,
	                                  NULL,
	                                  NULL);

	g_signal_connect (G_OBJECT (priv->timeline),
	                  "completed",
	                  G_CALLBACK (_fade_out_completed_cb),
	                  self);
	g_signal_connect (G_OBJECT (priv->timeline),
	                  "new-frame",
	                  G_CALLBACK (_fade_cb),
	                  self);

	egg_timeline_start (priv->timeline);
}
