/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** metric-converter.c - implements conversion between EM, pixel, centimeter and
**                      inch
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <math.h>
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "metric-converter.h"

G_DEFINE_TYPE (MetricConverter, metric_converter, G_TYPE_OBJECT);

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), METRIC_CONVERTER_TYPE, MetricConverterPrivate))

struct _MetricConverterPrivate {
	gdouble dpi;
	gdouble points;
	gdouble pixels_per_em;
	gdouble pixels_per_inch;
	gdouble pixels_per_cm;
};

#define DEFAULT_DPI          96.0
#define DEFAULT_POINTS       10.0
#define GNOME_DESKTOP_SCHEMA "org.gnome.desktop.interface"
#define GSETTINGS_FONT_KEY   "font-name"

/*-- internal API ------------------------------------------------------------*/

static void
metric_converter_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (metric_converter_parent_class)->dispose (gobject);
}

static void
metric_converter_finalize (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (metric_converter_parent_class)->finalize (gobject);
}

static void
metric_converter_init (MetricConverter* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */
}

static void
metric_converter_get_property (GObject*    gobject,
                               guint       prop,
                               GValue*     value,
                               GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
metric_converter_class_init (MetricConverterClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (MetricConverterPrivate));

	gobject_class->dispose      = metric_converter_dispose;
	gobject_class->finalize     = metric_converter_finalize;
	gobject_class->get_property = metric_converter_get_property;
}

gdouble
_get_font_size()
{
	GSettings*            gnome_settings = NULL;
	gchar*                font_name      = NULL;
	PangoFontDescription* desc           = NULL;
	gdouble               points         = 0.0f;

	gnome_settings = g_settings_new (GNOME_DESKTOP_SCHEMA);
	font_name = g_settings_get_string (gnome_settings, GSETTINGS_FONT_KEY);
	desc = pango_font_description_from_string (font_name);
	points = (gdouble) pango_font_description_get_size (desc) /
		     (gdouble) PANGO_SCALE;
	pango_font_description_free (desc);
	g_free ((gpointer) font_name);
	g_object_unref (gnome_settings);

	return points;
}

/*-- public API --------------------------------------------------------------*/

MetricConverter*
metric_converter_new ()
{
	MetricConverter*        self = NULL;
	MetricConverterPrivate* priv = NULL;

	self = g_object_new (METRIC_CONVERTER_TYPE, NULL);
	if (!self)
		return NULL;

	priv = GET_PRIVATE (self);

	priv->points = _get_font_size();

	GdkScreen*   screen        = gdk_screen_get_default(); // not ref'ed
	gint         max_monitors  = gdk_screen_get_n_monitors (screen);
	gint         monitor       = 0;
	GdkRectangle geo           = {0, 0, 0, 0};
	gint         width         = 0;
	gint         height        = 0;
	gdouble      dpi_vert      = 0.0;
	gdouble      dpi_horiz     = 0.0;

	for (monitor = 0; monitor < max_monitors; monitor++)
	{
		width  = gdk_screen_get_monitor_width_mm (screen, monitor);
		height = gdk_screen_get_monitor_height_mm (screen, monitor);
		if (width == -1 || height == -1)
		{
			GdkDisplay* display     = gdk_screen_get_display(screen); // not ref'ed
			Display*    display_x11 = gdk_x11_display_get_xdisplay(display); // not ref'ed

			g_debug ("Could not determine geometry in mm of each monitor. "
			         "You might happen to run a TwinView-setup.\n");
			g_debug ("screen 0: %dmm * %dmm, %dpx * %dpx\n",
			         DisplayWidthMM(display_x11, 0),
			         DisplayHeightMM(display_x11, 0),
			         DisplayWidth(display_x11, 0),
			         DisplayHeight(display_x11, 0));

			/* assume a default DPI of 96.0 */
			priv->dpi = 96.0;

			/* since we can't get the physical size only iterate once */
			monitor = max_monitors;
		}
		else
		{
			gdk_screen_get_monitor_geometry (screen, monitor, &geo);

			/* determine current system DPI */
			/* remember: 1 inch == 2.54 cm == 25.4 mm */
			dpi_horiz  = (gdouble) geo.width / ((gdouble) width / 25.4);
			dpi_vert = (gdouble) geo.height / ((gdouble) height / 25.4);
			if(dpi_horiz != dpi_vert)
				priv->dpi = (dpi_horiz + dpi_vert) / 2.0;
			else
				priv->dpi = dpi_horiz;
		}
	}

	priv->pixels_per_em   = priv->points / 72.0f * priv->dpi;
	priv->pixels_per_inch = priv->dpi;
	priv->pixels_per_cm   = priv->dpi / 2.54;

	return self;
}

gdouble
metric_converter_get_dpi (MetricConverter* self)
{
	g_return_val_if_fail (self && IS_METRIC_CONVERTER(self), DEFAULT_DPI);

	return GET_PRIVATE (self)->dpi;
}

gdouble
metric_converter_get_points (MetricConverter* self)
{
	g_return_val_if_fail (self && IS_METRIC_CONVERTER(self), DEFAULT_POINTS);

	return GET_PRIVATE (self)->points;
}

gint
metric_converter_em2px (MetricConverter* self,
                        gdouble          em)
{
	g_return_val_if_fail (self && IS_METRIC_CONVERTER(self), 0);

	return (gint) roundl (em * GET_PRIVATE (self)->pixels_per_em);
}

gdouble
metric_converter_em2cm (MetricConverter* self,
                        gdouble          em)
{
	g_return_val_if_fail (self &&
	                      IS_METRIC_CONVERTER (self) &&
	                      GET_PRIVATE (self)->pixels_per_cm != 0.0,
	                      0.0);

	return (metric_converter_em2px (self, em) / GET_PRIVATE (self)->pixels_per_cm);
}

gdouble
metric_converter_em2inch (MetricConverter* self,
                          gdouble          em)
{
	g_return_val_if_fail (self &&
	                      IS_METRIC_CONVERTER (self) &&
	                      GET_PRIVATE (self)->pixels_per_inch != 0.0,
	                      0.0);

	return (metric_converter_em2px (self, em) / GET_PRIVATE (self)->pixels_per_inch);
}

gdouble
metric_converter_px2em (MetricConverter* self,
                        gint             px)
{
	g_return_val_if_fail (self &&
	                      IS_METRIC_CONVERTER (self) &&
	                      GET_PRIVATE (self)->pixels_per_em != 0.0,
	                      0.0);

	return ((gdouble) px / GET_PRIVATE (self)->pixels_per_em);
}

gdouble
metric_converter_px2cm (MetricConverter* self,
                        gint             px)
{
	g_return_val_if_fail (self &&
	                      IS_METRIC_CONVERTER (self) &&
	                      GET_PRIVATE (self)->pixels_per_cm != 0.0,
	                      0.0);

	return ((gdouble) px / GET_PRIVATE (self)->pixels_per_cm);
}

gdouble
metric_converter_px2inch (MetricConverter* self,
                          gint             px)
{
	g_return_val_if_fail (self &&
	                      IS_METRIC_CONVERTER (self) &&
	                      GET_PRIVATE (self)->pixels_per_inch != 0.0,
	                      0.0);

	return ((gdouble) px / GET_PRIVATE (self)->pixels_per_inch);
}

gdouble
metric_converter_cm2em (MetricConverter* self,
                        gdouble          cm)
{
	g_return_val_if_fail (self && IS_METRIC_CONVERTER (self), 0.0);

	return (metric_converter_px2em (self, metric_converter_cm2px (self, cm)));
}

gint
metric_converter_cm2px (MetricConverter* self,
                        gdouble          cm)
{
	g_return_val_if_fail (self && IS_METRIC_CONVERTER (self), 0);

	return (gint) roundl (cm * GET_PRIVATE (self)->pixels_per_cm);
}

gdouble
metric_converter_inch2em (MetricConverter* self,
                          gdouble          inch)
{
	g_return_val_if_fail (self && IS_METRIC_CONVERTER (self), 0.0);

	return (metric_converter_px2em (self, metric_converter_inch2px (self, inch)));
}

gint
metric_converter_inch2px (MetricConverter* self,
                          gdouble          inch)
{
	g_return_val_if_fail (self && IS_METRIC_CONVERTER (self), 0);

	return (gint) roundl (inch * GET_PRIVATE (self)->pixels_per_inch);
}
