/*  VER 218   TAB P   $Id: putarticle.c,v 1.23.2.19 2003/05/18 10:44:37 egil Exp $
 *
 *  POST articles via an NNTP server
 *
 *  copyright 1996, 1997, 1999 Egil Kvaleberg, egil@kvaleberg.no
 *  the GNU General Public License applies
 *
 *  $Log: putarticle.c,v $
 *  Revision 1.23.2.19  2003/05/18 10:44:37  egil
 *  Fix for bug #237
 *
 *  Revision 1.23.2.18  2003/01/22 10:18:25  egil
 *  Implemented hostconfig, removed storageapi
 *
 *  Revision 1.23.2.17  2003/01/21 09:47:25  egil
 *  Renamed MAXHEADERSIZE to MAX_HEADER_SIZE due to name collision
 *
 *  Revision 1.23.2.16  2002/10/31 11:51:39  egil
 *  Fix for bug #133, bang_paths
 *
 *  Revision 1.23.2.15  2002/10/02 10:08:29  egil
 *  New filter for SMretrieve
 *
 *  Revision 1.23.2.14  2002/10/02 09:05:28  egil
 *  Do not flush stream when posting
 *
 *  Revision 1.23.2.13  2002/10/01 13:09:11  egil
 *  Silly mistakes
 *
 *  Revision 1.23.2.12  2002/10/01 13:07:18  egil
 *  Silly bug: Omits empty lines!
 *
 *  Revision 1.23.2.11  2002/10/01 12:54:35  egil
 *  Fix for bug #86
 *
 *  Revision 1.23.2.10  2002/10/01 09:21:54  egil
 *  Made function for storage API
 *
 *  Revision 1.23.2.9  2002/10/01 08:54:59  egil
 *  Changed sprintf to snprintf
 *
 *  Revision 1.23.2.8  2002/09/21 17:20:35  egil
 *  A long range of patches incorporated..
 *
 *  Revision 1.23.2.7  2002/02/06 20:11:35  egil
 *  Debian Bug#88795: newsx: Inn 2.3.1 storage API compilation
 *
 *  Revision 1.23.2.6  2002/01/29 21:10:16  egil
 *  More sane handling of articles with CR
 *
 *  Revision 1.23.2.5  2002/01/29 20:46:49  egil
 *  Better error message
 *
 *  Revision 1.23.2.4  2002/01/29 20:43:35  egil
 *  Fixed problems with articles with wrong CRs
 *
 *  Revision 1.23.2.3  2002/01/29 07:46:39  egil
 *  Initialize storage earlier
 *
 *  Revision 1.23.2.2  2002/01/29 07:39:03  egil
 *  Storage API detection fixed
 *
 *  Revision 1.23.2.1  2001/02/14 06:55:40  egil
 *  Fixes from Winston Edmond
 *
 *  Revision 1.23  1999/04/05 19:01:51  src
 *  Lock for scanlogs
 *
 *  Revision 1.22  1999/04/05 13:07:34  src
 *  Various fixes for storage-API
 *
 *  Revision 1.21  1999/03/31 03:28:20  src
 *  Also removing "NNTP-Posting-Date" when posting
 *
 *  Revision 1.20  1999/03/19 13:01:50  src
 *  Removed comma from log file format
 *
 *  Revision 1.19  1999/03/19 08:02:26  src
 *  Fixed problem with newlines in logfile introduced in 1.4pre
 *
 *  Revision 1.18  1999/03/16 08:03:31  src
 *  Renamed lock() to do_lock()
 *
 *  Revision 1.17  1999/03/15 08:50:28  src
 *  Fixed posting with new storeageAPI
 *
 *  Revision 1.16  1999/03/14 08:47:55  src
 *  Uprated log for innreport.
 *
 *  Revision 1.15  1999/03/11 07:54:47  src
 *  Moved storage to lib and src
 *
 *  Revision 1.14  1999/03/07 14:58:19  src
 *  Read newsconfig supported. Storage API supported.
 *
 *  Revision 1.13  1999/03/04 17:43:20  src
 *  Removing X-trace etc.
 *
 *  Revision 1.12  1998/11/22 08:00:08  src
 *  Option --no-queue
 *
 *  Revision 1.11  1998/09/11 16:37:44  src
 *  Lockfile for logfile of posted articles and for posted article folder.
 *
 *  Revision 1.10  1998/09/11 09:17:43  src
 *  Check path consistency (--no-path) and length (--max-path)
 *  GNU style option --help, --version, --dry-run, changed --noxx to --no-xx
 *  Check for putenv and setenv, added xstrcpy
 *
 *  Revision 1.9  1998/09/09 07:32:13  src
 *  Version 1.1
 *
 *  Revision 1.8  1998/09/02 06:50:31  src
 *  newsx version 1.0
 *
 *  Revision 1.7  1998/08/24 06:17:15  src
 *
 *  Revision 1.6  1998/07/22 10:56:31  src
 *  Implemented ad hoc support for "441 Duplicate" type response.
 */

#include "common.h"
#include "proto.h"
#include "options.h"
#include "statistics.h"
#include "news.h"
#include "nntp.h"
#include "self.h"
#include "sim.h"

/*
 *  local stuff
 */
#define MAILFOLDER_TAG "From "
#define MAILFOLDER_TAG_LEN 5

/* for log etc. */
static char msgid[NNTP_STRLEN];
static char msgsender[NNTP_STRLEN];

static int article_lines = 0;

static int SM_open = 0;

/*
 *  check if header tag
 */
int
is_tag_n(char *line, int len, char *tag)
{
    char c,d;
    char *end = line+len;

    while ((c = *tag++)) {
	if (line >= end) return 0;
	d = *line++;
	if (toupper(c) != toupper(d)) return 0;
    }
    return 1;
}
	
/*
 *  get current local time
 *  return static pointer
 */
static char *
current_time(void)
{
    time_t t;

    time(&t);
    return text_time(t);
}

/*
 *  check out header...
 *  cr/lf is not included
 *  update skip state
 */
static void 
check_header(char *line, int len, int *skip)
{
    char *p;

    if (len > 0) switch (toupper(line[0])) {
#if 0
    case 'C':
	if (is_tag_n(line,len,p="Content-Type: ")) {
	    if (!contenttype[0])
		strncpy(contenttype,line+strlen(p),sizeof(contenttype)-1);
	}
	break;
#endif
    case 'F':
	if (is_tag_n(line,len,p="From: ")) {
	    /* use From: if no Sender: */
	    if (!msgsender[0])
		strncpy(msgsender,line+strlen(p),sizeof(msgsender)-1);
	}
	break;
    case 'M':
	if (is_tag_n(line,len,p="Message-ID: ")) {
	    /* record local Message-ID */
	    strncpy(msgid,line+strlen(p),sizeof(msgid)-1);
	    if (nomsgid_opt && !ihave_opt) {
		/* but don't transmit it */
		*skip = 1;
		return;
	    }
	}
	break;
    case 'N':
	if ((is_tag_n(line,len,"NNTP-Posting-Host:")
	  || is_tag_n(line,len,"NNTP-Posting-Date:")) && !ihave_opt) {
	    /*
	     * when POSTing, these headers may cause
	     * messages to be rejected
	     * fix by: Riku Saikkonen <rjs@isil.lloke.dna.fi>               
	     *         and Simon J. Mudd <sjmudd@bitmailer.net>
	     */
	    *skip = 1;
	    return;
	}
	break;
    case 'P':
	if (is_tag_n(line,len,"Path:")) {
	    /* skip "Path: " */
	    int bangs = path_bangs(line+6,len-6);

	    if (bangs > max_path) {
		/* BUG: do something better */
		log_msg(L_ERR,"article path is %d steps long, max is %d",
					     bangs,max_path);
		/* BUG: don't send article */
		/* BUG: document */
	    }
	    if (!keep_path_opt && !ihave_opt) {
		/*
		 * skip local Path when POSTing:
		 * the main reason is that the local client
		 * may not have a registered host name
		 */
		*skip = 1;
		return;
	    }
	}
	break;
    case 'R':
	if (is_tag_n(line,len,p="Reply-to: ")) {
	    /* record Reply-to: */
	    strncpy(msgsender,line+strlen(p),sizeof(msgsender)-1);
	}
	break;
    case 'S':
	if (is_tag_n(line,len,p="Sender: ")) {
	    /* BUG: use Sender if no Reply-to */
	    strncpy(msgsender,line+strlen(p),sizeof(msgsender)-1);
	}
	break;
    case 'X':
	if (is_tag_n(line,len,"Xref:")) {
	    /* always omit local Xref */
	    *skip = 1;
	    return;
	} else if (is_tag_n(line,len,"X-Server-Date:")
		|| is_tag_n(line,len,"X-Trace:")
		|| is_tag_n(line,len,"X-Complaints-To:")) {
	    if (!ihave_opt) {
		/* should not occur in postings */
		*skip = 1;
		return;
	    }
	    return;
	}
	break;
    case ' ':
    case '\t':
	/* continued header line, maintain skip state */
	return;
    }
    /* keep header line */
    *skip = 0;
}

/*
 *  sniff an article to see if it is ok
 *  return false and return reason if problems
 *  set pointer to message ID if available
 */
static int
sniff_article(ARTHANDLE *article,char **id_p,char **reason)
{
    char *line = article->data;
    char *end = line + article->len;
    int linelen;
    int dummy;
    int hdr = 1;
    char *p;

    msgid[0] = '\0';
    msgsender[0] = '\0';
	
    /* read the article, header and body */
    progtitle("find message-ID");
    while (line < end) {
	if (!(p = memchr(line,'\r',end-line)) || p+1 >= end || p[1]!='\n') {
	    *reason = "inconsistent line termination";
	    log_msg(L_ERR,*reason);
	    /* article no good */
	    return 0;
	}
	linelen = (p+2) - line;

	if (linelen == 1+2 && line[0] == '.') {
	    /* proper end-of-file */
	    break;
	}

	if (linelen == 2) { /* empty line: end of header */
	    hdr = 0;
	} else if (hdr) {
	    check_header(line, p-line, &dummy);
	}

	line += linelen;
    }
    if (id_p) {
	*id_p = msgid[0] ? msgid : 0;
    }
    return 1;
}

/*
 *  transfer an article...
 *  set static variable article_lines
 */
static void
write_article(ARTHANDLE *article)
{
    char *line = article->data;
    char *end = line + article->len;
    int linelen;
    int hdr = 1;
    int skiphdr = 0;
    FILE *folder_file = 0;
    int lock_id = -1;
    char *p;

    msgid[0] = '\0';
    msgsender[0] = '\0';
    article_lines = 0;

    /* transfer the article, header and body */
    progtitle("post: transfer article");
    while (line < end) {
	if (!(p = memchr(line,'\r',end-line)) || p+1 >= end || p[1]!='\n') {
	    /* should not happen: see sniff_article() */
	    log_msg(L_ERR,"internal article inconsistency");
	    exit_cleanup(1);
	}

	linelen = (p+2) - line;
	++article_lines;

	if (linelen == 1+2 && line[0] == '.') {
	    /* proper end-of-file */
	    /* BUG: verify with end */
	    break;
	}

	if (linelen == 2) {
	    /* empty line: end of header */
	    hdr = skiphdr = 0;
	} else if (hdr) {
	    /* update skip state */
	    check_header(line, p-line, &skiphdr);
	}

	if (!skiphdr) {
	    if (!put_server_line(line,p-line,0)) exit_cleanup(9);

	    /* save to folder also */
	    if (folder) {
		if (!folder_file) {
		    char lockname[PATH_MAX];

		    /*
		     * lock other newsxes accessing same file
		     */
		    progtitle("locking folder");
		    build_filename(lockname,folder,LOCK_SUFFIX,NULL,NULL);
		    lock_id = do_lock("",lockname,0);

		    if (!(folder_file = fopen(folder,"a"))) {
			log_msg(L_ERRno,"can't open folder: %s",folder);
			folder = 0;
		    } else {
			/* folder item header, as for mail folders */
			fprintf(folder_file, "%s%s %s\n",
				MAILFOLDER_TAG, spoolname, current_time());
		    }
		}
		if (folder_file) {
		    /* save to folder too */
		    if (strncmp(line, MAILFOLDER_TAG, MAILFOLDER_TAG_LEN)==0) {
			/* mail folder hack */
			fputc('>', folder_file);
		    }
		    /* newline to Unix convention */
		    fwrite(line, sizeof(char), p-line, folder_file);
		    fputc('\n', folder_file);
		}
	    }
	    net_bytecount += linelen;
	}
	line += linelen;
    }

    if (folder_file) {
	/* BUG: what if posting failed... */
	/* BUG: or if noaction_opt.. */
	/* always a trailing blank line */
	fputc('\n',folder_file);
	fflush(folder_file);
	if (ferror(folder_file)) {
	    log_msg(L_ERRno,"error writing folder: %s",folder);
	}
	fclose(folder_file);
    }

    /* time to release lock */
    if (lock_id >= 0)
	unlock_one(lock_id);

    /* send termination */
    progtitle("post: send termination");
    if (!put_server_line(".",1,1)) exit_cleanup(9);

    /* article successfully read */
    log_msg(L_DEBUG,"(%d lines)", article_lines);
}

/*
 *  put_article following POST or IHAVE
 *  return string if all right
 *  return reason for failure
 */
static char *
put_article(ARTHANDLE *article, char **reason)
{
    static char status[NNTP_STRLEN+1];
    char *p;
    char *endptr;
    char *ok = 0;

    *reason = "?";

    /* read status line from server */
    if (!noaction_opt) {
	if (!get_server_nntp(status, sizeof(status))) {
	    exit_cleanup(9);
	}
    } else {
	snprintf(status,sizeof(status),"%d",CONT_POST); /* fake OK */
    }

    switch (strtoul(status,&endptr,10)) {
    case CONT_POST:                 /* post: posting allowed, continue */
    case CONT_XFER:                 /* ihave: go ahead */
	write_article(article);
	break;

    case ERR_POSTFAIL:              /* post: failed for some other reason */
	log_msg(L_ERR,"posting failed: got \"%s\"", status);
	++failed_articles;
	*reason = status;
	return 0; /* try again */

    case ERR_GOTIT:                 /* ihave: already got it */
	log_msg(L_DEBUG,"already got it");
	++duplicate_articles;
	return "Already got it";

    case ERR_XFERFAIL:              /* ihave: try again later */
	log_msg(L_ERR,"try again later");
	++failed_articles;
	*reason = status;
	return 0; /* try again */

    case ERR_NOPOST:                /* post: not allowed */
    case ERR_GOODBYE:               /* ihave: you do not have permission */
	log_msg(L_ERR,"sending prohibited: got \"%s\"", status);
	++failed_articles;
	*reason = status;
	return 0; /* don't give up */

    case ERR_XFERRJCT:              /* ihave: rejected - do not try again */
	log_msg(L_ERR,"rejected: got \"%s\"", status);
	++failed_articles;
	*reason = status;
	return 0; /* BUG: don't give up */

    /* otherwise must be a protocol error */
    default:
	log_msg(L_ERR,"NNTP send negotiation error: got \"%s\"", status);
	exit_cleanup(4);
    }

    /* get status of posting */
    progtitle("post: get status");
    if (!noaction_opt) {
	if (!get_server_nntp(status, sizeof(status))) {
	    exit_cleanup(9);
	} 
    } else {
	snprintf(status,sizeof(status),"%d",OK_POSTED);
    }

    switch (strtoul(status,&endptr,10)) {
    case OK_XFERED:                 /* ihave: transferred OK */
    case OK_POSTED:                 /* post: article posted OK */
	++posted_articles;
	if (!noaction_opt)
	    ok = "OK";
	else
	    ok = "TEST";
	break;

    case ERR_NOPOST:                /* post: not allowed */
    case ERR_GOODBYE:               /* ihave: you do not have permission */
	log_msg(L_ERR,"sending not allowed: got \"%s\"", status);
	*reason = status;
	break; /* don't give up */

    case ERR_GOTIT:                 /* ihave: already got it */
	/* we'll just continue */
	log_msg(L_DEBUG,"already got it");
	++duplicate_articles;
	ok = "Already got it";
	break;

    case ERR_POSTFAIL:              /* post: failed */
    case ERR_XFERFAIL:              /* ihave: try again later */
    case ERR_XFERRJCT:              /* ihave: rejected - do not try again */
	/* see if secondary error message */
	if (strtoul(endptr,NULL,10) == ERR_GOTIT) {
	    /* already gotit: we'll just continue */
	    log_msg(L_DEBUG,"duplicate");
	    ++duplicate_articles;
	    ok = "Duplicate";

	} else if ((p = strchr(endptr,'D'))
		   && strncmp(p,"Duplicate",9)==0) {
	    /*
	     * this is really adhoc - the response is something like:
	     *  441 Posting Failed (Duplicate Message-ID)
	     */
	    log_msg(L_DEBUG,"assumed duplicate");
	    ++duplicate_articles;
	    ok = "Assumed duplicate";

	} else {
	    /* posting failed, some other reason */
	    log_msg(L_ERR,"article rejected: got \"%s\"", status);
	    ++failed_articles;
	    *reason = status;
	    break;
	}
	break;

    default:                        /* otherwise, protocol error */
	log_msg(L_ERR,"NNTP sending protocol error: got \"%s\"", status);
	exit_cleanup(4);
    }

    return ok;
}

/*
 *  submit an article with specified id to the server
 */
static void 
post_article(void)
{
    progtitle("post: issuing POST");
    if (!put_server_cmd("POST")) exit_cleanup(9);
}

/*
 *  as post_article, but use IHAVE mechanism 
 */
static void 
ihave_article(char *msgid)
{
    char buf[NNTP_STRLEN+1];
    progtitle("post: issuing IHAVE");
    snprintf(buf,sizeof(buf),"IHAVE %s", msgid);
    if (!put_server_cmd(buf)) exit_cleanup(9);
}

/*
 *  log to file
 *  BUG: rather inefficient
 */
static void
log_to_file(char *articlename, char *ok)
{
    FILE *fp;
    int lock_id;
    char lockname[PATH_MAX];

    if (!logfile) return;

    /*
     *  lock other newsxes accessing same file
     */
    progtitle("locking log file");
    build_filename(lockname,logfile,LOCK_SUFFIX,NULL,NULL);
    lock_id = do_lock("",lockname,0);

    progtitle("post: logging");
    if (!(fp = fopen(logfile,"a"))) {
	log_msg(L_ERRno,"can't open logfile: %s",logfile);
    } else {
	/* for innreport; Uli Zappe <uli@tallowcross.uni-frankfurt.de> */
	fprintf(fp,"%s %s newsx[%i]: %s\t%s\t%s\t%s\t%s\t%d lines\n",
		    current_time() + 4, /* skip "Day " */
		       my_hostname,
				my_pid,
				     spoolname,
					 msgid[0] ? msgid : "<?>",
					     articlename,
						 msgsender[0] ? msgsender:"?",
						     ok,
							 article_lines);
	if (fclose(fp) == EOF) {
	    log_msg(L_ERRno,"can't write to logfile: %s",logfile);
	}
    }
    unlock_one(lock_id);
}

/*
 *  kill article, for various reasons
 */
void
kill_article(char *articlename, ARTHANDLE *article, char *reason)
{
    char *to;

    log_msg(L_ERR,"failed article \"%s\" killed",articlename);
    if (!bounce || !bounce[0] || strcmp(bounce,"poster")==0) {
	/* return to sender */
	if (!msgsender[0]) {
	    log_msg(L_ERR,"no From-specification in \"%s\"",
						  articlename);
	    to = 0;
	} else {
	    to = msgsender;
	}
    } else if (strcmp(bounce,"none")==0) {
	to = 0;
    } else {
	/* bounce address specified */
	to = bounce;
    }
    if (to) {
	++bounced_articles;
	log_msg(L_ERR,"bounce message to %s",to);
	if (!bounce_msg(to,article->data,article->len,
		      article->arrived,articlename,msgid,reason)) {
	    /* BUG: what ever shall we do? */
	    log_msg(L_ERR,"bounce failed");
	}
    }
}

/*
 *  we need storage management now
 *  return string if there are problems
 */
static char *
need_sm(void)
{
    if (!SM_open) {
	/* NOTE: below, "int" should be "BOOL", but various installations
	 *       may or may not typedef BOOL
	 */
	static int value = 0; /* read/only access */

	if (!SMsetup(SM_RDWR,(void *)&value)) {
	    log_msg(L_ERR,"can't setup storage-API: %s",SMerrorstr);
	    return "Storage-API error";
	}
	if (!SMinit()) {
	    log_msg(L_ERR,"can't initialize storage-API: %s",SMerrorstr);
	    return "Storage-API error";
	}
	SM_open = 1;
    }
    return NULL;
}

/*
 *  submit articles to the currently open NNTP server socket.
 *  return string if article should be removed from outgoing batch
 */
char *
submit_article(char *articlename)
{
    ARTHANDLE *article;
    long age;
    char *err;
    char *id;
    char *reason;
    int is_api = 0;

    progtitle("post: reading article");

    if (IsToken(articlename)) {
	/* use  Storage-API */
	TOKEN art_token;

	if ((err = need_sm())) return err;

	is_api = 1;
	art_token = TextToToken(articlename);

	log_msg(L_DEBUG,"reading API %s", articlename);

	/* better error reporting, by Andres Metzler <ametzler@logic.univie.ac.at> */
	if ((article = SMretrieve(art_token, RETR_ALL)) == NULL) {
	    /* filter by Jochen Schmitt <Jochen@herr-schmitt-de> */
	    if ((SMerrno != SMERR_NOENT) && (SMerrno != SMERR_UNINIT)) {
		log_msg(L_ERR,"Could not retrieve article: %s", SMerrorstr);
		return "Retrieve error";
	    }
	}

    } else if (articlename[0] == '@' && 
	       articlename[strlen(articlename)-1] == '@') {
	/* storage-API type of entry, but no support */
	log_msg(L_ERR,"no storage-API support for \"%s\"",articlename);
	return "No support for storage-API";

    } else {
	/* work with full and partial paths */
	char fullname[PATH_MAX];

	build_alt_filename(fullname,spooldir,articlename,NULL);

	log_msg(L_DEBUG,"reading %s", fullname);

	article = SIMretrieve(fullname);
    }

    if (!article) {
	/* article is missing - throw it away */
	log_msg(L_ERR,"can't find article \"%s\"",articlename);
	++missing_articles;
	err = "Missing";
    } else {
	if (!sniff_article(article,&id,&reason)) {
	    /* something is not like it should be: kill and possibly bounce */
	    ++failed_articles;
	    kill_article(articlename,article,reason);
	    err = "Malformed";
	} else {
	    if (ihave_opt) {
		/* use ihave */
		if (!id) {
		    log_msg(L_ERR,"article \"%s\" lacks a message ID",articlename);
		    ++missing_articles;
		    err = "Missing-ID";
		} else {
		    ihave_article(id);
		    err = put_article(article,&reason);
		}
	    } else {
		/* request post */
		/* BUG: check if post_allowed? */
		post_article();

		/* and do it */
		err = put_article(article,&reason);
	    }
	}
    }

    if (err) {
	log_to_file(articlename,err);
    } else {
	if (failtime) {
	    /* timeout message... */
	    time_t now;

	    time(&now);
	    age = now - article->arrived;

	    log_msg(L_DEBUG,"article is %ds old, limit is %ds", age, failtime);
	    if (age > failtime) {
		kill_article(articlename,article,reason);
		err = "Failed";
	    }
	}
    }
    if (article) {
	if (is_api) SMfreearticle(article);
	else SIMfreearticle(article);
    }

    return err;
}

/*
 *  close article interface
 */
void
close_article(void)
{
    if (SM_open) {
	SMshutdown();
	SM_open = 0;
    }
}
