
#include "wlan.h"
#include <stdlib.h>
#include "wlanacmevent.h"

#include "WlanScanner.h"
#include "DataCenter.h"

wxString MacAddressToString(const unsigned char* mac)
{
	int a[6];
	for (int i = 0 ; i < WLAN_MAC_ADDRESS_LENGTH ; i ++)
	{
		a[i] = 0 + mac[i];
	}
	return wxString::Format(_T("%02X:%02X:%02X:%02X:%02X:%02X"), a[0], a[1], a[2], a[3], a[4], a[5]);
}

bool StringToMacAddress(unsigned char* mac, const char* str, size_t strLen)
{
    if (str == NULL || mac == NULL || strLen < 17)
        return false;

    const char* colon = ":";
    size_t index = 0;
    const char* first = str;
    const char* second = NULL;
    while (index < 6)
    {
        second = strstr(first, colon);
        unsigned char hex = 0;
        const char* p = first;
        int count = 0;
        while (count < 2)
        {
            char n = *p - '0';
            if (n > 9) n -= 7;
            hex = hex * 16 + n;
            p++;
            count++;
        }
        mac[index] = hex;
        index++;
        if (second != NULL)
        {
            first = strstr(second, colon);
            first++;
        }
    }
    return true;
}

bool operator == (const WlanSsid& ssid1, const WlanSsid& ssid2)
{
    bool result = true;
    result &= (ssid1.length == ssid2.length);
    for (int i = 0 ; i < AP_SSID_MAX_LENGTH ; i ++)
    {
        result &= (ssid1.ucSsid[i] == ssid2.ucSsid[i]);
    }
    return result;
}

bool operator != (const WlanSsid& ssid1, const WlanSsid& ssid2)
{
    return !(ssid1 == ssid2);
}

int CompareMacAddress(unsigned char* mac1, unsigned char* mac2)
{
    if ( mac1 == NULL || mac2 == NULL )
        return 0;

    int differ = 0;
    int index = 0;
    do
    {
        differ = mac1[index] - mac2[index];
        index++;
    }
    while ( differ == 0 && index < WLAN_MAC_ADDRESS_LENGTH );
    return differ;
}

#if defined(__WXMSW__)

wxString NetworkNameToString(const PWLAN_AVAILABLE_NETWORK pNetwork)
{
    unsigned char* pChar = &pNetwork->dot11Ssid.ucSSID[0];
    return wxString::FromUTF8((char*)pChar, pNetwork->dot11Ssid.uSSIDLength);
}

wxString GuidToString(GUID guid)
{
    wxString str = wxString::Format(_T("{%08X-"), guid.Data1);
    str += wxString::Format(_T("%04X-"), guid.Data2);
    str += wxString::Format(_T("%04X-"), guid.Data3);
    str += wxString::Format(_T("%02X%02X-"), guid.Data4[0], guid.Data4[1]);
    str += wxString::Format(_T("%02X%02X%02X%02X%02X%02X}"), guid.Data4[2], guid.Data4[3], guid.Data4[4], guid.Data4[5], guid.Data4[6], guid.Data4[7]);
    return str;
}

VOID UpdateCurrentAP(const PDOT11_SSID pSsid, bool connected)
{
    DataCenter* pDataCenter = DataCenter::GetInstance();
    if (pDataCenter == NULL)
        return;

    if (!connected)
    {
        pDataCenter->UpdateCurrentAP(NULL, false);
    }
    else
    {
        WlanSsid ssid;
        ssid.length = pSsid->uSSIDLength;
        memcpy(ssid.ucSsid, pSsid->ucSSID, AP_SSID_MAX_LENGTH);
        pDataCenter->UpdateCurrentAP(&ssid, true);
    }
}

VOID WINAPI OnWlanNotify(PWLAN_NOTIFICATION_DATA pData, PVOID pVoid)
{
    wxEvtHandler* pEvtHandler = pVoid == NULL ? NULL : static_cast<wxEvtHandler*>(pVoid);
    PWLAN_CONNECTION_NOTIFICATION_DATA pNotifyData = NULL;
    DWORD source = pData->NotificationSource;
    if ( source & WLAN_NOTIFICATION_SOURCE_ACM )
    {
        switch ( pData->NotificationCode )
        {
        case wlan_notification_acm_scan_complete:
            if ( pEvtHandler != NULL )
                ::wxPostEvent(pEvtHandler, WlanAcmEvent(ID_WLAN_ACM_SCAN_COMPLETE));
            break;

        case wlan_notification_acm_connection_complete:
            pNotifyData = static_cast<PWLAN_CONNECTION_NOTIFICATION_DATA>(pData->pData);
            UpdateCurrentAP(&pNotifyData->dot11Ssid, true);
//          ::wxPostEvent(pEvtHandler, WlanAcmEvent(ID_WLAN_ACM_CONNECT));
            break;

        case wlan_notification_acm_interface_arrival:
        case wlan_notification_acm_interface_removal:
            if ( pEvtHandler != NULL )
                ::wxPostEvent(pEvtHandler, WlanAcmEvent(ID_WLAN_ACM_INTERFACE));
            break;

        case wlan_notification_acm_network_not_available:
        case wlan_notification_acm_network_available:
            if ( pEvtHandler != NULL )
                ::wxPostEvent(pEvtHandler, WlanAcmEvent(ID_WLAN_ACM_NETWORK));
            break;

        case wlan_notification_acm_disconnected:
            UpdateCurrentAP(NULL, false);
            break;

        default:
            break;
        }
    }
}

DWORD ENMWlanGetAPStatus(PWLAN_AVAILABLE_NETWORK pNetwork, PAPStatus pAPStatus)
{
	if (pNetwork == NULL || pAPStatus == NULL)
		return ERROR_INVALID_PARAMETER;

	// get name
	mbstowcs(pAPStatus->name, (const char*)pNetwork->dot11Ssid.ucSSID, pNetwork->dot11Ssid.uSSIDLength);

	// ssid
	pAPStatus->ssid.length = pNetwork->dot11Ssid.uSSIDLength;
	memcpy(&pAPStatus->ssid.ucSsid[0], &pNetwork->dot11Ssid.ucSSID[0], sizeof(pNetwork->dot11Ssid.ucSSID));

	// get signal quality
	pAPStatus->signalQuality = pNetwork->wlanSignalQuality;

	// get security status
	if (pNetwork->bSecurityEnabled)
	{
	    pAPStatus->auth = WLAN_SECURITY_UNKNOWN;
	    switch (pNetwork->dot11DefaultAuthAlgorithm)
	    {
        case DOT11_AUTH_ALGO_WPA_PSK:
            pAPStatus->auth = WLAN_SECURITY_WPA_PSK;
            break;
        case DOT11_AUTH_ALGO_80211_OPEN:
            pAPStatus->auth = WLAN_SECURITY_WEP;
            break;
	    }
	}
	else
	{
	    pAPStatus->auth = WLAN_SECURITY_NONE;
	}
#if GETTYPE_THROUGH_WLANAPI
	// get type
    pAPStatus->type = WLAN_UNKNOWN;
    if ( pNetwork->uNumberOfPhyTypes > 0 )
    {
        DOT11_PHY_TYPE phyType = pNetwork->dot11PhyTypes[0];
        for ( ULONG i = 0 ; i < pNetwork->uNumberOfPhyTypes; i++ )
        {
            if ( pNetwork->dot11PhyTypes[i] > phyType )
                phyType = pNetwork->dot11PhyTypes[i];
        }
        if (phyType == dot11_phy_type_hrdsss)
            pAPStatus->type = WLAN_802_11_B;
        else if (phyType == dot11_phy_type_erp)
            pAPStatus->type = WLAN_802_11_G;
        else if (phyType == dot11_phy_type_ht)
            pAPStatus->type = WLAN_802_11_N;
    }
#endif
    // get bss type
    if (pNetwork->dot11BssType == dot11_BSS_type_infrastructure)
		pAPStatus->bssType = BSS_TYPE_INFRASTRUCTURE;
	else if (pNetwork->dot11BssType == dot11_BSS_type_independent)
		pAPStatus->bssType = BSS_TYPE_INDEPENDENT;
    else if (pNetwork->dot11BssType == dot11_BSS_type_any)
		pAPStatus->bssType = BSS_TYPE_ANY;
	else
		pAPStatus->bssType = BSS_TYPE_UNKNOWN;

	return ERROR_SUCCESS;
}

#endif // __WXMSW__
