
#include "ipv4.h"
#include <string.h>

#if defined(__WXMSW__)
#include <Winsock2.h>
#else
#include <arpa/inet.h>
#endif

IPv4AddressUtil::IPv4AddressUtil()
{
}

IPv4AddressUtil::~IPv4AddressUtil()
{
}

bool IPv4AddressUtil::ValidateMask(Inetv4Address mask)
{
    const unsigned long nMask = ntohl(mask.addr_l);
    unsigned long nBit = 0x80000000L;
    int iSubnetLength = 1;
    while ( iSubnetLength < 31 )
    {
        if ( (nMask ^ nBit) == 0 )
            return true;
        iSubnetLength++;
        nBit = (nBit >> 1) | 0x80000000L;
    }
    return false;
}

bool IPv4AddressUtil::IsBroadcastAddress(Inetv4Address inetAddr, Inetv4Address mask)
{
    if ( !ValidateMask(mask) )
        return false;

    unsigned long nIp = ntohl(inetAddr.addr_l);
    unsigned long nMask = ntohl(mask.addr_l);

    return (nIp | nMask) == 0xFFFFFFFFL;
}

bool IPv4AddressUtil::IsMulticastAddress(Inetv4Address inetAddr)
{
    unsigned long nIp = ntohl(inetAddr.addr_l);
    return (nIp & 0xF0000000L) == 0xE0000000L;
}

InetResult IPv4AddressUtil::ValidateHostAddress(const Inetv4Address& hostAddr)
{
    unsigned long nIp = ntohl(hostAddr.addr_l);
    if ( (nIp & 0xFF000000L) == 0L )
        return INET_INVALID_HOST;
    if ( (nIp & 0xFF000000L) == 0x7F000000L )
        return INET_RESERVED_ADDRESS;
    if ( (nIp & 0xF0000000L) == 0xE0000000L )
        return INET_MULTICAST_ADDRESS;
    if ( (nIp & 0xF8000000L) == 0xF0000000L )
        return INET_RESERVED_ADDRESS;
    return INET_OK;
}

InetResult IPv4AddressUtil::ValidateHostAddress(const char* pszHostAddr)
{
    if ( pszHostAddr == NULL )
        return INET_INVALID_PARAMETER;

    Inetv4Address inetAddr;
    InetResult result = StringToInetAddress(pszHostAddr, &inetAddr);
    if ( result != INET_OK )
        return INET_INVALID_HOST;
    else
        return ValidateHostAddress(inetAddr);
}

InetResult IPv4AddressUtil::StringToInetAddress(const char* pszIpAddr, Inetv4Address* pInetAddr)
{
    if ( pszIpAddr == NULL || pInetAddr == NULL )
        return INET_INVALID_PARAMETER;

    int iDotCount = 0;
    for ( size_t i = 0; i < strlen(pszIpAddr); i++ )
    {
        char c = pszIpAddr[i];
        if ( (c < '0' || c > '9') && c != '.' )
            return INET_INVALID_ADDRESS;
        if ( c == '.' )
            iDotCount++;
    }
    if ( iDotCount != 3 )
        return INET_INVALID_ADDRESS;

    unsigned long ip = inet_addr(pszIpAddr);
    if ( ip == INADDR_NONE || ip == 0L )
        return INET_INVALID_ADDRESS;

    pInetAddr->addr_l = ip;
    return INET_OK;
}

InetResult IPv4AddressUtil::InetAddressToString(const Inetv4Address* pInetAddr, char* pszIpAddr)
{
    if ( pszIpAddr == NULL || pInetAddr == NULL )
        return INET_INVALID_PARAMETER;

	in_addr addr;
	memcpy(&addr, pInetAddr, sizeof(Inetv4Address));
    char* pAddr = inet_ntoa(addr);
    if ( pAddr == NULL )
        return INET_INVALID_ADDRESS;

    strcpy(pszIpAddr, pAddr);
    return INET_OK;
}

InetResult IPv4AddressUtil::ValidateAdapterAddress(const AdapterAddress* pAdptAddr, bool strictType)
{
    if ( pAdptAddr == NULL )
        return INET_INVALID_PARAMETER;

    if ( !ValidateMask(pAdptAddr->mask) )
        return INET_INVALID_MASK;

	unsigned long nIp = ntohl(pAdptAddr->ip.addr_l);
    unsigned long nMask = ntohl(pAdptAddr->mask.addr_l);
    unsigned long nGw = ntohl(pAdptAddr->gw.addr_l);

    if ( ValidateHostAddress(pAdptAddr->ip) != INET_OK )
        return INET_INVALID_HOST;

    if ( strictType )
    {
        if ( (nIp & 0x80000000L) == 0x00000000L && ((nMask & 0xFF000000L) ^ 0xFF000000L) != 0x00000000L )
            return INET_INVALID_ADDRESS_TYPE;
        if ( (nIp & 0xC0000000L) == 0x80000000L && ((nMask & 0xFFFF0000L) ^ 0xFFFF0000L) != 0x00000000L )
            return INET_INVALID_ADDRESS_TYPE;
        if ( (nIp & 0xE0000000L) == 0xC0000000L && ((nMask & 0xFFFFFF00L) ^ 0xFFFFFF00L) != 0x00000000L )
            return INET_INVALID_ADDRESS_TYPE;
    }

    if ( (nIp & nMask) != (nGw & nMask) )
        return INET_ADDRESS_NOT_MATCH;

    if ( (nIp | nMask) == 0xFFFFFFFFL )
        return INET_BROADCAST_HOST_ADDRESS;

    if ( (nIp & ~nMask) == 0L )
        return INET_NETWORK_HOST_ADDRESS;

    if ( (nIp & 0xF0000000L) == 0xE0000000L )
        return INET_MULTICAST_ADDRESS;

    if ( (nIp & 0xF8000000L) == 0xF0000000L )
        return INET_RESERVED_ADDRESS;

    if ( (nGw | nMask) == 0xFFFFFFFFL )
        return INET_BROADCAST_GATEWAY_ADDRESS;

    if ( (nGw & ~nMask) == 0L )
        return INET_NETWORK_GATEWAY_ADDRESS;

    return INET_OK;
}

InetResult IPv4AddressUtil::ValidateAdapterAddress(const char* pszHost, const char* pszMask, const char* pszGateway, bool strictType)
{
    if ( pszHost == NULL || pszMask == NULL || pszGateway == NULL )
        return INET_INVALID_PARAMETER;

    AdapterAddress adptAddr;
    if ( StringToInetAddress(pszHost, &adptAddr.ip) != INET_OK )
        return INET_INVALID_HOST;
    if ( StringToInetAddress(pszMask, &adptAddr.mask) != INET_OK )
        return INET_INVALID_MASK;
    if ( StringToInetAddress(pszGateway, &adptAddr.gw) != INET_OK )
        return INET_INVALID_GATEWAY;

    return ValidateAdapterAddress(&adptAddr, strictType);
}
