/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include "nmn-device-handler.h"
#include "nmn-network-item.h"

G_DEFINE_TYPE (NmnDeviceHandler, nmn_device_handler, G_TYPE_OBJECT)

enum {
    PROP_0,
    PROP_NM_DATA,
    PROP_DEVICE,

    LAST_PROP
};

enum {
    ITEM_ADDED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL];

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_DEVICE_HANDLER, NmnDeviceHandlerPrivate))

typedef struct {
    NmnNMData *nm_data;
    NMDevice *device;
    GSList *items;
    gboolean started;
    gulong device_state_changed_id;
    gulong user_connection_added_id;
    gulong system_connection_added_id;

    gboolean disposed;
} NmnDeviceHandlerPrivate;

static void
connection_added (NMSettings *settings,
                  NMExportedConnection *exported,
                  gpointer user_data)
{
    NmnDeviceHandler *handler = NMN_DEVICE_HANDLER (user_data);

    if (NMN_DEVICE_HANDLER_GET_CLASS (handler)->connection_added)
        NMN_DEVICE_HANDLER_GET_CLASS (handler)->connection_added (handler, exported);
}

static void
add_one_connection (gpointer data, gpointer user_data)
{
    connection_added (NULL, NM_EXPORTED_CONNECTION (data), user_data);
}

void
nmn_device_handler_add_items (NmnDeviceHandler *self)
{
    GSList *list;

    g_return_if_fail (NMN_IS_DEVICE_HANDLER (self));
    g_return_if_fail (GET_PRIVATE (self)->started);

    if (nm_device_get_state (nmn_device_handler_get_device (self)) < NM_DEVICE_STATE_DISCONNECTED)
        return;

    list = nmn_device_handler_get_connections (self);
    g_slist_foreach (list, add_one_connection, self);
    g_slist_free (list);
}

static void
device_state_changed (NMDevice *device,
                      NMDeviceState new_state,
                      NMDeviceState old_state,
                      NMDeviceStateReason reason,
                      gboolean user_data)
{
    NmnDeviceHandler *handler = NMN_DEVICE_HANDLER (user_data);

    if (new_state < NM_DEVICE_STATE_DISCONNECTED)
        nmn_device_handler_remove_items (handler);
    else if (old_state < NM_DEVICE_STATE_DISCONNECTED && new_state >= NM_DEVICE_STATE_DISCONNECTED)
        nmn_device_handler_add_items (handler);
}

void
nmn_device_handler_start (NmnDeviceHandler *self)
{
    NMSettings *settings;
    NmnDeviceHandlerPrivate *priv;

    g_return_if_fail (NMN_IS_DEVICE_HANDLER (self));

    priv = GET_PRIVATE (self);
    if (priv->started)
        return;

    priv->started = TRUE;

    priv->device_state_changed_id = g_signal_connect (priv->device, "state-changed",
                                                      G_CALLBACK (device_state_changed), self);

    settings = nmn_nm_data_get_user_settings (priv->nm_data);
    if (settings)
        priv->user_connection_added_id = g_signal_connect (settings, "new-connection",
                                                           G_CALLBACK (connection_added), self);

    settings = nmn_nm_data_get_system_settings (priv->nm_data);
    if (settings)
        priv->system_connection_added_id = g_signal_connect (settings, "new-connection",
                                                             G_CALLBACK (connection_added), self);

    nmn_device_handler_add_items (self);
}

NmnNMData *
nmn_device_handler_get_nm_data (NmnDeviceHandler *self)
{
    g_return_val_if_fail (NMN_IS_DEVICE_HANDLER (self), NULL);

    return GET_PRIVATE (self)->nm_data;
}

NMDevice *
nmn_device_handler_get_device (NmnDeviceHandler *self)
{
    g_return_val_if_fail (NMN_IS_DEVICE_HANDLER (self), NULL);

    return GET_PRIVATE (self)->device;
}

static void
item_removed (NmnItem *item, gpointer user_data)
{
    NmnDeviceHandlerPrivate *priv = GET_PRIVATE (user_data);

    priv->items = g_slist_remove (priv->items, item);
}

void
nmn_device_handler_add_item (NmnDeviceHandler *self,
                             NmnItem *item)
{
    NmnDeviceHandlerPrivate *priv;

    g_return_if_fail (NMN_IS_DEVICE_HANDLER (self));
    g_return_if_fail (NMN_IS_ITEM (item));

    g_signal_connect (item, "remove-requested", G_CALLBACK (item_removed), self);

    priv = GET_PRIVATE (self);
    priv->items = g_slist_prepend (priv->items, item);
    g_signal_emit (self, signals[ITEM_ADDED], 0, item);
}

GSList *
nmn_device_handler_get_items (NmnDeviceHandler *self)
{
    g_return_val_if_fail (NMN_IS_DEVICE_HANDLER (self), NULL);

    return GET_PRIVATE (self)->items;
}

GSList *
nmn_device_handler_get_connections (NmnDeviceHandler *self)
{
    NmnDeviceHandlerPrivate *priv;
    NMSettings *settings;
    GSList *list = NULL;

    g_return_val_if_fail (NMN_IS_DEVICE_HANDLER (self), NULL);

    priv = GET_PRIVATE (self);

    settings = nmn_nm_data_get_user_settings (priv->nm_data);
    if (settings)
        list = nm_settings_list_connections (settings);

    settings = nmn_nm_data_get_system_settings (priv->nm_data);
    if (settings)
        list = g_slist_concat (list, nm_settings_list_connections (settings));

    return list;
}

NmnItem *
nmn_device_handler_get_item_for_connection (NmnDeviceHandler *self,
                                            NMExportedConnection *connection)
{
    GSList *list;
    GSList *iter;

    g_return_val_if_fail (NMN_IS_DEVICE_HANDLER (self), NULL);
    g_return_val_if_fail (NM_IS_EXPORTED_CONNECTION (connection), NULL);

    list = GET_PRIVATE (self)->items;
    for (iter = list; iter; iter = iter->next) {
        if (nmn_network_item_get_connection (NMN_NETWORK_ITEM (iter->data)) == connection)
            return NMN_ITEM (iter->data);
    }

    return NULL;
}

void
nmn_device_handler_remove_items (NmnDeviceHandler *self)
{
    GSList *list;
    GSList *iter;

    g_return_if_fail (NMN_IS_DEVICE_HANDLER (self));

    list = nmn_device_handler_get_items (self);
    for (iter = list; iter; iter = iter->next) {
        NmnItem *item = NMN_ITEM (iter->data);

        if (nmn_item_get_status (item) != NMN_ITEM_STATUS_DISCONNECTED)
            nmn_item_disconnect_request (item);

        nmn_item_remove_request (item);
    }
}

static void
nmn_device_handler_init (NmnDeviceHandler *handler)
{
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnDeviceHandlerPrivate *priv;

    object = G_OBJECT_CLASS (nmn_device_handler_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    if (!priv->nm_data || !priv->device) {
        g_warning ("Missing constructor arguments");
        g_object_unref (object);
        return NULL;
    }

    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    NmnDeviceHandlerPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        /* Construct only */
        priv->nm_data = g_value_dup_object (value);
        break;
    case PROP_DEVICE:
        /* Construct only */
        priv->device = g_value_dup_object (value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    NmnDeviceHandlerPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        g_value_set_object (value, priv->nm_data);
        break;
    case PROP_DEVICE:
        g_value_set_object (value, priv->device);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
dispose (GObject *object)
{
    NmnDeviceHandlerPrivate *priv = GET_PRIVATE (object);
    NMSettings *settings;

    if (priv->disposed)
        return;

    g_signal_handler_disconnect (settings, priv->device_state_changed_id);

    settings = nmn_nm_data_get_user_settings (priv->nm_data);
    g_signal_handler_disconnect (settings, priv->user_connection_added_id);

    settings = nmn_nm_data_get_system_settings (priv->nm_data);
    g_signal_handler_disconnect (settings, priv->system_connection_added_id);

    g_slist_foreach (priv->items, (GFunc) g_object_unref, NULL);
    g_slist_free (priv->items);

    g_object_unref (priv->device);
    g_object_unref (priv->nm_data);
    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_device_handler_parent_class)->dispose (object);
}

static void
nmn_device_handler_class_init (NmnDeviceHandlerClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnDeviceHandlerPrivate));

    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->dispose = dispose;

    /* properties */
    g_object_class_install_property
        (object_class, PROP_NM_DATA,
         g_param_spec_object (NMN_DEVICE_HANDLER_NM_DATA,
                              "NmnNMData",
                              "NmnNMData",
                              NMN_TYPE_NM_DATA,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    g_object_class_install_property
        (object_class, PROP_DEVICE,
         g_param_spec_object (NMN_DEVICE_HANDLER_DEVICE,
                              "NMDevice",
                              "NMDevice",
                              NM_TYPE_DEVICE,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    /* signals */
    signals[ITEM_ADDED] = g_signal_new 
        ("item-added",
         G_OBJECT_CLASS_TYPE (class),
         G_SIGNAL_RUN_LAST,
         G_STRUCT_OFFSET (NmnDeviceHandlerClass, item_added),
         NULL, NULL,
         g_cclosure_marshal_VOID__OBJECT,
         G_TYPE_NONE, 1,
         NMN_TYPE_ITEM);
}
