/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-app.h"

#include <glib.h>
#include <glib/gi18n.h>
#include <gio/gio.h>
#include <string.h>
#include <stdlib.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <launcher/launcher.h>

#include <netbook-launcher/netbook-launcher.h>
#include "nl-window.h"

#define LINK_WATCH_DIR "/tmp/webfav"

G_DEFINE_TYPE (NlApp, nl_app, UNIQUE_TYPE_APP);

#define NL_APP_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_APP, NlAppPrivate))

struct _NlAppPrivate
{
  GtkWidget    *window;
  NlShell      *shell;
  
  GFile        *watch_dir;
  GFileMonitor *dir_monitor;
};

/* Forwards */
UniqueResponse on_unique_message_received (UniqueApp         *self,
                                           gint               command,
                                           UniqueMessageData *data,
                                           guint              time_,
                                           gpointer           udata);

static void    watch_links                (NlApp             *app);

/* GObject stuff */
static void
nl_app_finalize (GObject *object)
{
  NlAppPrivate *priv;

  priv = NL_APP (object)->priv;

  if (priv->watch_dir)
    {
      g_object_unref (priv->watch_dir);
      priv->watch_dir = NULL;
    }
  if (priv->dir_monitor)
    {
      g_object_unref (priv->dir_monitor);
      priv->dir_monitor = NULL;
    }

  G_OBJECT_CLASS (nl_app_parent_class)->finalize (object);
}

static void
nl_app_class_init (NlAppClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = nl_app_finalize;

  g_type_class_add_private (obj_class, sizeof (NlAppPrivate));
}

static void
nl_app_init (NlApp *self)
{
  NlAppPrivate *priv;

  priv = self->priv = NL_APP_GET_PRIVATE (self);

  unique_app_add_command (UNIQUE_APP (self),
                          "add-favorite",
                          COMMAND_ADD_FAVORITE);
                          
  unique_app_add_command (UNIQUE_APP (self),
                          "show-favorites",
                          COMMAND_SHOW_FAVORITES);

  g_signal_connect (self, "message-received",
                    G_CALLBACK (on_unique_message_received), NULL);
}

NlApp *
nl_app_get_default (void)

{
  static NlApp *app = NULL;

  if (G_UNLIKELY (app == NULL))
    app = g_object_new (NL_TYPE_APP,
                        "name", "com.canonical.NetbookLauncher",
                        NULL);

  return app;
}


void
nl_app_init_real (NlApp *app)
{
  NlAppPrivate *priv;

  g_return_if_fail (NL_IS_APP (app));
  priv = app->priv;

  priv->window = nl_window_get_default ();
  priv->shell = NL_SHELL (priv->window);

  /* Set everything else up here */
  watch_links (app);
}

/*
 * Private Methods
 */
static void
create_favorite_from_path (NlApp *self, gchar *path)
{
  NlAppPrivate *priv = self->priv;
  GSList       *loaders, *l;
  gchar        *uid = NULL;

  g_return_if_fail (path);

  loaders = nl_shell_get_favorites_loaders (priv->shell);
  for (l = loaders; l; l = l->next)
    {
      NlFavoritesLoader *loader = l->data;

      uid = nl_favorites_loader_uid_for_path (loader, path);
      if (uid)
        break;
    }

  if (uid)
    nl_shell_add_favorite (priv->shell, uid);
  else
    g_warning ("Unable to handle favorite of path: %s", path);

  g_free (uid);

  g_free (path);
}

UniqueResponse
on_unique_message_received (UniqueApp         *self,
                            gint               command,
                            UniqueMessageData *data,
                            guint              time_,
                            gpointer           udata)
{
  g_return_val_if_fail (NL_IS_APP (self), UNIQUE_RESPONSE_INVALID);

  switch (command)
    {
    case COMMAND_ADD_FAVORITE:
      create_favorite_from_path (NL_APP (self),
                                 unique_message_data_get_text (data));
      break;
    case COMMAND_SHOW_FAVORITES:
      nl_shell_show_favorites (NL_APP (self)->priv->shell);
    default:
      g_debug ("Cannot handle Unique message of type: %d", command);
      return UNIQUE_RESPONSE_INVALID;
    }

  return UNIQUE_RESPONSE_OK;
}

/*
 * STUFF THAT NEEDS TO BE PORTED
 */

/*
 * External link watching
 */
static void
on_watch_dir_changed (GFileMonitor      *monitor,
                      GFile             *file_changed,
                      GFile             *other_file,
                      GFileMonitorEvent  event,
                      NlApp       *app)
{
  NlAppPrivate *priv;
  gchar *link_path;

  g_return_if_fail (NL_IS_APP (app));
  priv = app->priv;

  if (event != G_FILE_MONITOR_EVENT_CREATED)
    return;

  link_path = g_file_get_path (file_changed);
  if (link_path && g_str_has_suffix (link_path, ".desktop"))
    {
      GKeyFile *file;
      gchar    *name;
      gchar    *url;
      gchar    *uid;
      LauncherFavorites *favs;

      file = g_key_file_new ();
      if (!g_key_file_load_from_file (file, link_path, 0, NULL))
        {
          g_warning ("Unable to open %s, cannot add web favorite", link_path);
          return;
        }

      name = g_key_file_get_string (file,
                                    G_KEY_FILE_DESKTOP_GROUP,
                                    G_KEY_FILE_DESKTOP_KEY_NAME,
                                    NULL);
      url = g_key_file_get_string (file,
                                   G_KEY_FILE_DESKTOP_GROUP,
                                   G_KEY_FILE_DESKTOP_KEY_URL,
                                   NULL);

      uid = g_strdup_printf ("uri-%d", (gint)time (NULL));
     
      favs = launcher_favorites_get_default ();

      launcher_favorites_set_string (favs, uid, "type", "uri");
      launcher_favorites_set_string (favs, uid, "uri", url);
      launcher_favorites_set_string (favs, uid, "name", name);
      launcher_favorites_add_favorite (favs, uid);

      nl_shell_add_favorite (priv->shell, uid);

      g_object_unref (favs);
      g_key_file_free (file);
      g_free (name);
      g_free (url);
      g_free (uid);
    }
  g_free (link_path);
}

static void
watch_links (NlApp *app)
{
  NlAppPrivate *priv = app->priv;
  GFile *dir;

  g_mkdir_with_parents (LINK_WATCH_DIR, 0700);

  dir = priv->watch_dir = g_file_new_for_path (LINK_WATCH_DIR);

  priv->dir_monitor = g_file_monitor_directory (dir, 0, NULL, NULL);

  g_signal_connect (priv->dir_monitor, "changed",
                    G_CALLBACK (on_watch_dir_changed), app);
}

#if 0
static gboolean
launch_dist_page (ClutterActor *actor, ClutterButtonEvent *event)
{
  const gchar *url;
  gchar *exec;

  if (event->button != 1)
    return FALSE;

  url = clutter_actor_get_name (actor);
  exec = g_strdup_printf ("%s %s", XDG_OPEN, url);

  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    exec,
                                    NULL);
  g_free (exec);

  return FALSE;
}

static void
load_dist_logo (NlApp *app)
{
#define DIST_DIR "/apps/netbook-launcher"
#define DIST_DIR_LEGACY "/apps/ume-nl"
#define DIST_IMG DIST_DIR"/dist_image"
#define DIST_IMG_LEGACY DIST_DIR_LEGACY"/dist_image"
#define DIST_URL DIST_DIR"/dist_url"
#define DIST_URL_LEGACY DIST_DIR_LEGACY"/dist_url"

  NlAppPrivate *priv = app->priv;
  NlConfig *cfg = nl_config_get_default ();
  GConfClient *client = gconf_client_get_default ();
  ClutterActor *tex;
  GdkPixbuf *logo;
  gint width, height;
  const gchar *imageurl;
  const gchar *url;

  width = cfg->bar_width;
  height = cfg->shortcut_height * 1.1;
  height *= 1;

  gconf_client_add_dir (client, DIST_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
  imageurl = gconf_client_get_string (client, DIST_IMG, NULL);
  url = gconf_client_get_string (client, DIST_URL, NULL);

  /* if new gconf key exists, use it, otherwise try legacy key */
  if (!imageurl)
    {
      gconf_client_remove_dir (client, DIST_DIR, NULL);
      gconf_client_add_dir (client, DIST_DIR_LEGACY, GCONF_CLIENT_PRELOAD_NONE, NULL);
      imageurl = gconf_client_get_string (client, DIST_IMG_LEGACY, NULL);
      url = gconf_client_get_string (client, DIST_URL_LEGACY, NULL);

      if (!imageurl)
        return;
    }

  logo = gdk_pixbuf_new_from_file_at_scale (imageurl,
         width, height, TRUE, NULL);

  if (logo)
    {
      tex = nl_util_texture_new_from_pixbuf (logo);
      clutter_container_add_actor (CLUTTER_CONTAINER (app), tex);
      clutter_actor_set_anchor_point_from_gravity (tex, CLUTTER_GRAVITY_CENTER);
      clutter_actor_set_position (tex,
                                  clutter_actor_get_x (priv->window) + (width/2),
                                  CSH() - PADDING*2 - (height/2));
      clutter_actor_set_name (tex, url);
      clutter_actor_set_reactive (tex, TRUE);
      clutter_actor_show (tex);

      if (url)
        {
          g_signal_connect (tex, "button-release-event",
                            G_CALLBACK (launch_dist_page), NULL);
        }
      g_object_unref (logo);
    }

  g_object_unref (client);
}
#endif
