/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <new>
#include <nsServiceManagerUtils.h>
#include <nsNetUtil.h>
#include <nsIPermission.h>
#include <nsIPermissionManager.h>

#include "mhs-permission-manager.h"
#include "mhs-error-private.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

/* D-Bus method implementations */

static gboolean mhs_pm_add (MhsPermissionManager *pm,
                            const gchar *uri,
                            const gchar *type,
                            guint32 permission,
                            GError **error);

static gboolean mhs_pm_remove (MhsPermissionManager *pm,
                               const gchar *host,
                               const gchar *type,
                               GError **error);

static gboolean mhs_pm_remove_all (MhsPermissionManager *pm,
                                   GError **error);

static gboolean mhs_pm_test_permission (MhsPermissionManager *pm,
                                        const gchar *uri,
                                        const gchar *type,
                                        PRUint32 *result,
                                        GError **error);

static gboolean mhs_pm_test_exact_permission (MhsPermissionManager *pm,
                                              const gchar *uri,
                                              const gchar *type,
                                              PRUint32 *result,
                                              GError **error);

static gboolean mhs_pm_get_all (MhsPermissionManager *pm,
                                GPtrArray **permissions,
                                GError **error);

#include "mhs-permission-manager-glue.h"

/* End D-Bus method implementations */

static void mhs_permission_manager_finalize (GObject *object);

G_DEFINE_TYPE (MhsPermissionManager,
               mhs_permission_manager,
               G_TYPE_OBJECT);

#define MHS_PERMISSION_MANAGER_GET_PRIVATE(obj)                     \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MHS_TYPE_PERMISSION_MANAGER, \
                                MhsPermissionManagerPrivate))

struct _MhsPermissionManagerPrivate
{
  nsCOMPtr<nsIPermissionManager> pm_service;
};

static void
mhs_permission_manager_class_init (MhsPermissionManagerClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->finalize = mhs_permission_manager_finalize;

  g_type_class_add_private (klass, sizeof (MhsPermissionManagerPrivate));

  dbus_g_object_type_install_info
    (MHS_TYPE_PERMISSION_MANAGER,
     &dbus_glib_mhs_pm_object_info);
}

static void
mhs_permission_manager_init (MhsPermissionManager *self)
{
  MhsPermissionManagerPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;
  nsresult rv;

  priv = self->priv = MHS_PERMISSION_MANAGER_GET_PRIVATE (self);

  /* 'placement new' to call the constructor for
     MhsPermissionManagerPrivate */
  new (reinterpret_cast<void *> (priv)) MhsPermissionManagerPrivate;

  priv->pm_service
    = do_GetService (NS_PERMISSIONMANAGER_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    g_warning ("Failed to retrieve permission manager service");

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      dbus_g_connection_register_g_object
        (connection,
         MHS_SERVICE_PERMISSION_MANAGER_PATH,
         G_OBJECT (self));
      dbus_g_connection_unref (connection);
    }
}

static void
mhs_permission_manager_finalize (GObject *object)
{
  MhsPermissionManager *self = MHS_PERMISSION_MANAGER (object);
  MhsPermissionManagerPrivate *priv = self->priv;

  /* Explicitly call the destructor for the private data (so that it
     destructs without trying to free the memory) */
  priv->~MhsPermissionManagerPrivate ();

  G_OBJECT_CLASS (mhs_permission_manager_parent_class)->finalize (object);
}

MhsPermissionManager *
mhs_permission_manager_new (void)
{
  return (MhsPermissionManager *) g_object_new (MHS_TYPE_PERMISSION_MANAGER,
                                                NULL);
}

static gboolean
mhs_pm_add (MhsPermissionManager *pm,
            const gchar *uri,
            const gchar *type,
            guint32 permission,
            GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (pm), FALSE);

  MhsPermissionManagerPrivate *priv = pm->priv;

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = priv->pm_service->Add (nsuri, type, permission);
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_pm_remove (MhsPermissionManager *pm,
               const gchar *host,
               const gchar *type,
               GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (pm), FALSE);

  MhsPermissionManagerPrivate *priv = pm->priv;

  rv = priv->pm_service->Remove (nsDependentCString (host), type);
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_pm_remove_all (MhsPermissionManager *pm,
                   GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (pm), FALSE);

  MhsPermissionManagerPrivate *priv = pm->priv;

  rv = priv->pm_service->RemoveAll ();
  if (NS_FAILED (rv))
    goto rv_error;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_pm_test_permission (MhsPermissionManager *pm,
                        const gchar *uri,
                        const gchar *type,
                        guint32 *result,
                        GError **error)
{
  nsresult rv;
  PRUint32 result_ns;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (pm), FALSE);

  MhsPermissionManagerPrivate *priv = pm->priv;

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = priv->pm_service->TestPermission (nsuri, type, &result_ns);
  if (NS_FAILED (rv))
    goto rv_error;

  *result = result_ns;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_pm_test_exact_permission (MhsPermissionManager *pm,
                              const gchar *uri,
                              const gchar *type,
                              guint32 *result,
                              GError **error)
{
  nsresult rv;
  PRUint32 result_ns;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (pm), FALSE);

  MhsPermissionManagerPrivate *priv = pm->priv;

  nsCOMPtr<nsIURI> nsuri;
  rv = NS_NewURI (getter_AddRefs (nsuri), uri);
  if (NS_FAILED (rv))
    goto rv_error;

  rv = priv->pm_service->TestExactPermission (nsuri, type, &result_ns);
  if (NS_FAILED (rv))
    goto rv_error;

  *result = result_ns;

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}

static gboolean
mhs_pm_get_all (MhsPermissionManager *pm,
                GPtrArray **permissions,
                GError **error)
{
  nsresult rv;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (pm), FALSE);

  MhsPermissionManagerPrivate *priv = pm->priv;

  GValue empty_string_value = { 0 };

  nsCOMPtr<nsISimpleEnumerator> penum;
  rv = priv->pm_service->GetEnumerator (getter_AddRefs (penum));
  if (NS_FAILED (rv))
    goto rv_error;

  g_value_init (&empty_string_value, G_TYPE_STRING);

  *permissions = g_ptr_array_new ();

  while (true)
    {
      int value_num = 0;

      /* Check whether there are more elements */
      PRBool has_more_elements;
      rv = penum->HasMoreElements (&has_more_elements);
      if (NS_FAILED (rv))
        break;
      if (!has_more_elements)
        break;

      /* Get the next element as an nsISupports */
      nsCOMPtr<nsISupports> perm_supports;
      rv = penum->GetNext (getter_AddRefs (perm_supports));
      if (NS_FAILED (rv))
        break;

      /* Convert it to an nsIPermission */
      nsCOMPtr<nsIPermission> perm
        = do_QueryInterface (perm_supports, &rv);
      if (NS_FAILED (rv))
        break;

      /* Create a new value array and add it to the pointer array. If
         any of the below fail then the value array will be freed when
         the pointer array is freed */
      GValueArray *value_array = g_value_array_new (3);
      g_ptr_array_add (*permissions, value_array);

      /* Get each member of the permission instance and add it to the
         value array */
      GValue *value;

      nsCAutoString host;
      rv = perm->GetHost (host);
      if (NS_FAILED (rv))
        break;

      g_value_array_append (value_array, &empty_string_value);
      value = g_value_array_get_nth (value_array, value_num++);
      g_value_set_string (value, host.get ());

      nsCAutoString type;
      rv = perm->GetType (type);
      if (NS_FAILED (rv))
        break;

      g_value_array_append (value_array, &empty_string_value);
      value = g_value_array_get_nth (value_array, value_num++);
      g_value_set_string (value, type.get ());

      PRUint32 capability;
      rv = perm->GetCapability (&capability);
      if (NS_FAILED (rv))
        break;

      GValue capability_value = { 0 };
      g_value_init (&capability_value, G_TYPE_UINT);
      g_value_set_uint (&capability_value, capability);
      g_value_array_append (value_array, &capability_value);
      g_value_unset (&capability_value);
    }

  g_value_unset (&empty_string_value);

  if (NS_FAILED (rv))
    {
      g_ptr_array_foreach (*permissions, (GFunc) g_value_array_free, NULL);
      g_ptr_array_free (*permissions, TRUE);
      goto rv_error;
    }

  return TRUE;

 rv_error:
  mhs_error_set_from_nsresult (rv, error);
  return FALSE;
}
