/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "mhs-error.h"
#include "mhs-error-private.h"

#include "mhs-prefs.h"
#include "mhs-prefs-bindings.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

G_DEFINE_TYPE (MhsPrefs, mhs_prefs, G_TYPE_OBJECT);

#define MHS_PREFS_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MHS_TYPE_PREFS, \
                                MhsPrefsPrivate))

struct _MhsPrefsPrivate
{
  DBusGConnection *connection;
  DBusGProxy *proxy;
};

enum
  {
    READ_SIGNAL,
    RESET_SIGNAL,
    APP_DEFAULTS_SIGNAL,
    BRANCH_CHANGED_SIGNAL,

    LAST_SIGNAL
  };

static guint prefs_signals[LAST_SIGNAL] = { 0, };

static void
mhs_prefs_dispose (GObject *object)
{
  MhsPrefs *self = (MhsPrefs *) object;
  MhsPrefsPrivate *priv = self->priv;

  if (priv->connection)
    dbus_g_connection_flush (priv->connection);

  if (priv->proxy)
    {
      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }

  if (priv->connection)
    {
      dbus_g_connection_unref (priv->connection);
      priv->connection = NULL;
    }

  G_OBJECT_CLASS (mhs_prefs_parent_class)->dispose (object);
}

static void
mhs_prefs_class_init (MhsPrefsClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->dispose = mhs_prefs_dispose;

  prefs_signals[READ_SIGNAL] =
    g_signal_new ("read",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, read),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[RESET_SIGNAL] =
    g_signal_new ("reset",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, reset),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[APP_DEFAULTS_SIGNAL] =
    g_signal_new ("app-defaults",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, app_defaults),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[BRANCH_CHANGED_SIGNAL] =
    g_signal_new ("branch-changed",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsPrefsClass, branch_changed),
                  NULL, NULL,
                  mhs_marshal_VOID__INT_STRING,
                  G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (MhsPrefsPrivate));

  dbus_g_object_register_marshaller (mhs_marshal_VOID__INT_STRING,
                                     G_TYPE_NONE,
                                     G_TYPE_INT,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
}

static void
mhs_prefs_read_cb (DBusGProxy  *proxy,
                   gpointer     data)
{
  g_signal_emit (data, prefs_signals[READ_SIGNAL], 0);
}

static void
mhs_prefs_reset_cb (DBusGProxy  *proxy,
                    gpointer     data)
{
  g_signal_emit (data, prefs_signals[RESET_SIGNAL], 0);
}

static void
mhs_prefs_app_defaults_cb (DBusGProxy  *proxy,
                           gpointer     data)
{
  g_signal_emit (data, prefs_signals[APP_DEFAULTS_SIGNAL], 0);
}

static void
mhs_prefs_branch_changed_cb (DBusGProxy  *proxy,
                             gint         id,
                             const gchar *domain,
                             gpointer     data)
{
  g_signal_emit (data, prefs_signals[BRANCH_CHANGED_SIGNAL], 0, id, domain);
}

static void
mhs_prefs_init (MhsPrefs *self)
{
  MhsPrefsPrivate *priv;
  GError *error = NULL;

  priv = self->priv = MHS_PREFS_GET_PRIVATE (self);

  if ((priv->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      priv->proxy
        = dbus_g_proxy_new_for_name (priv->connection,
                                     MHS_SERVICE_PREFS,
                                     MHS_SERVICE_PREFS_PATH,
                                     MHS_SERVICE_PREFS_INTERFACE);

      dbus_g_proxy_add_signal (priv->proxy, "Read",
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "Reset",
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "AppDefaults",
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "BranchChanged",
                               G_TYPE_INT,
                               G_TYPE_STRING,
                               G_TYPE_INVALID);

      dbus_g_proxy_connect_signal
        (priv->proxy, "Read",
         G_CALLBACK (mhs_prefs_read_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "Reset",
         G_CALLBACK (mhs_prefs_reset_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "AppDefaults",
         G_CALLBACK (mhs_prefs_app_defaults_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "BranchChanged",
         G_CALLBACK (mhs_prefs_branch_changed_cb),
         self, NULL);
    }
}

MhsPrefs *
mhs_prefs_new (void)
{
  MhsPrefs *self = (MhsPrefs *) g_object_new (MHS_TYPE_PREFS,
                                                    NULL);

  return self;
}

static gboolean
mhs_prefs_check_proxy (MhsPrefs  *self,
                          GError      **error)
{
  if (self->priv->proxy)
    return TRUE;

  g_set_error (error, MHS_ERROR,
               MHS_ERROR_PROXY,
               "Failed to initialize DBUS proxy");

  return FALSE;
}

gboolean
mhs_prefs_read_user (MhsPrefs  *self,
                        const gchar  *path,
                        GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_read_user (priv->proxy,
                                           path,
                                           error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_reset (MhsPrefs  *self,
                    GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_reset (priv->proxy, error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_reset_user (MhsPrefs  *self,
                         GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_reset_user (priv->proxy, error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_save_pref_file (MhsPrefs  *self,
                             const gchar  *path,
                             GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_save_pref_file (priv->proxy,
                                             path,
                                             error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_get_branch (MhsPrefs  *self,
                         const gchar  *root,
                         gint         *id,
                         GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_get_branch (priv->proxy,
                                            root,
                                            id,
                                            error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_get_default_branch (MhsPrefs  *self,
                                 const gchar  *root,
                                 gint         *id,
                                 GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_get_default_branch (priv->proxy,
                                                    root,
                                                    id,
                                                    error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_release_branch (MhsPrefs  *self,
                             gint          id,
                             GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_release_branch (priv->proxy,
                                                id,
                                                error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_get_type (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gint         *type,
                              GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_get_type (priv->proxy,
                                                 id,
                                                 name,
                                                 type,
                                                 error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_get_bool (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gboolean     *value,
                              GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_get_bool (priv->proxy,
                                                 id,
                                                 name,
                                                 value,
                                                 error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_set_bool (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gboolean      value,
                              GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_set_bool (priv->proxy,
                                                 id,
                                                 name,
                                                 value,
                                                 error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_get_int (MhsPrefs  *self,
                             gint          id,
                             const gchar  *name,
                             gint         *value,
                             GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_get_int (priv->proxy,
                                                id,
                                                name,
                                                value,
                                                error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_set_int (MhsPrefs  *self,
                             gint          id,
                             const gchar  *name,
                             gint          value,
                             GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_set_int (priv->proxy,
                                                id,
                                                name,
                                                value,
                                                error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_get_char (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gchar       **value,
                              GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_get_char (priv->proxy,
                                                 id,
                                                 name,
                                                 value,
                                                 error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_set_char (MhsPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              const gchar  *value,
                              GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_set_char (priv->proxy,
                                                 id,
                                                 name,
                                                 value,
                                                 error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_has_user_value (MhsPrefs  *self,
                                    gint          id,
                                    const gchar  *name,
                                    gboolean     *result,
                                    GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_has_user_value (priv->proxy,
                                                       id,
                                                       name,
                                                       result,
                                                       error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_lock (MhsPrefs  *self,
                          gint          id,
                          const gchar  *name,
                          GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_lock (priv->proxy,
                                             id,
                                             name,
                                             error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_is_locked (MhsPrefs  *self,
                               gint          id,
                               const gchar  *name,
                               gboolean     *result,
                               GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_is_locked (priv->proxy,
                                                  id,
                                                  name,
                                                  result,
                                                  error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_unlock (MhsPrefs  *self,
                            gint          id,
                            const gchar  *name,
                            GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_unlock (priv->proxy,
                                               id,
                                               name,
                                               error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_get_child_list (MhsPrefs   *self,
                                    gint           id,
                                    const gchar   *start,
                                    guint         *len,
                                    gchar       ***array,
                                    GError       **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_get_child_list (priv->proxy,
                                                       id,
                                                       start,
                                                       len,
                                                       array,
                                                       error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_add_observer (MhsPrefs  *self,
                              gint          id,
                              const gchar  *domain,
                              GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_add_observer (priv->proxy,
                                                  id,
                                                  domain,
                                                  error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_prefs_branch_remove_observer (MhsPrefs  *self,
                                  gint          id,
                                  const gchar  *domain,
                                  GError      **error)
{
  MhsPrefsPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PREFS (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_prefs_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_Prefs_branch_remove_observer (priv->proxy,
                                                     id,
                                                     domain,
                                                     error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

