/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "mhs-error.h"
#include "mhs-error-private.h"

#include "mhs-permission-manager.h"
#include "mhs-permission-manager-bindings.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

G_DEFINE_TYPE (MhsPermissionManager,
               mhs_permission_manager,
               G_TYPE_OBJECT);

#define MHS_PERMISSION_MANAGER_GET_PRIVATE(obj)                         \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MHS_TYPE_PERMISSION_MANAGER,     \
                                MhsPermissionManagerPrivate))

struct _MhsPermissionManagerPrivate
{
  DBusGProxy *proxy;
};

static void
mhs_permission_manager_dispose (GObject *object)
{
  MhsPermissionManager *self = (MhsPermissionManager *) object;
  MhsPermissionManagerPrivate *priv = self->priv;

  if (priv->proxy)
    {
      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }

  G_OBJECT_CLASS (mhs_permission_manager_parent_class)->dispose (object);
}

static void
mhs_permission_manager_class_init (MhsPermissionManagerClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->dispose = mhs_permission_manager_dispose;

  g_type_class_add_private (klass, sizeof (MhsPermissionManagerPrivate));
}

static void
mhs_permission_manager_init (MhsPermissionManager *self)
{
  MhsPermissionManagerPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = MHS_PERMISSION_MANAGER_GET_PRIVATE (self);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      priv->proxy = dbus_g_proxy_new_for_name
        (connection,
         MHS_SERVICE_PERMISSION_MANAGER,
         MHS_SERVICE_PERMISSION_MANAGER_PATH,
         MHS_SERVICE_PERMISSION_MANAGER_INTERFACE);

      dbus_g_connection_unref (connection);
    }
}

MhsPermissionManager *
mhs_permission_manager_new (void)
{
  return g_object_new (MHS_TYPE_PERMISSION_MANAGER, NULL);
}

static gboolean
mhs_pm_check_proxy (MhsPermissionManager  *self,
                    GError               **error)
{
  if (self->priv->proxy)
    return TRUE;

  g_set_error (error, MHS_ERROR, MHS_ERROR_PROXY,
               "Failed to initialize DBUS proxy");

  return FALSE;
}

gboolean
mhs_pm_add (MhsPermissionManager *self,
            const gchar *uri,
            const gchar *type,
            guint32 permission,
            GError **error)
{
  MhsPermissionManagerPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_pm_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_PermissionManager_add (priv->proxy,
                                              uri,
                                              type,
                                              permission,
                                              error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_pm_remove (MhsPermissionManager *self,
               const gchar *host,
               const gchar *type,
               GError **error)
{
  MhsPermissionManagerPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_pm_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_PermissionManager_remove (priv->proxy,
                                                 host,
                                                 type,
                                                 error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_pm_remove_all (MhsPermissionManager *self,
                   GError **error)
{
  MhsPermissionManagerPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_pm_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_PermissionManager_remove_all (priv->proxy,
                                                     error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_pm_test_permission (MhsPermissionManager *self,
                        const gchar *uri,
                        const gchar *type,
                        guint32 *result,
                        GError **error)
{
  MhsPermissionManagerPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_pm_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_PermissionManager_test_permission (priv->proxy,
                                                          uri,
                                                          type,
                                                          result,
                                                          error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_pm_test_exact_permission (MhsPermissionManager *self,
                              const gchar *uri,
                              const gchar *type,
                              guint32 *result,
                              GError **error)
{
  MhsPermissionManagerPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_pm_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mhs_PermissionManager_test_exact_permission (priv->proxy,
                                                                uri,
                                                                type,
                                                                result,
                                                                error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_pm_get_all (MhsPermissionManager *self,
                guint *n_permissions,
                MhsPermission **permissions,
                GError **error)
{
  MhsPermissionManagerPrivate *priv;
  GPtrArray *permission_array;

  g_return_val_if_fail (MHS_IS_PERMISSION_MANAGER (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mhs_pm_check_proxy (self, error))
    return FALSE;

  if (org_moblin_mhs_PermissionManager_get_all (priv->proxy,
                                                &permission_array,
                                                error))
    {
      int permission_num;

      /* Convert the pointer array to a C array of structs */
      *n_permissions = permission_array->len;
      *permissions = g_new (MhsPermission, permission_array->len);

      for (permission_num = 0;
           permission_num < permission_array->len;
           permission_num++)
        {
          MhsPermission *permission = (* permissions) + permission_num;
          GValueArray *value_array = g_ptr_array_index (permission_array,
                                                        permission_num);
          GValue *value;

          value = g_value_array_get_nth (value_array, 0);
          permission->host = g_value_dup_string (value);

          value = g_value_array_get_nth (value_array, 1);
          permission->type = g_value_dup_string (value);

          value = g_value_array_get_nth (value_array, 2);
          permission->capability = g_value_get_uint (value);

          g_value_array_free (value_array);
        }

      g_ptr_array_free (permission_array, TRUE);

      return TRUE;
    }
  else
    {
      _mhs_error_translate_from_dbus (error);

      return FALSE;
    }
}

void
mhs_pm_free_permissions (guint n_permissions,
                         MhsPermission *permissions)
{
  guint i;

  for (i = 0; i < n_permissions; i++)
    {
      g_free (permissions[i].host);
      g_free (permissions[i].type);
    }

  g_free (permissions);
}
