/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mwb-favicon-loader.h"
#include <nsCOMPtr.h>
#include <nsIOutputStream.h>
#include <nsStringAPI.h>
#include <nsNetCID.h>
#include <nsComponentManagerUtils.h>
#include <nsIURI.h>
#include <nsIPipe.h>
#include <nsIInputStream.h>
#include <nsNetUtil.h>

#define DEFAULT_ICON           PKGDATADIR "/default-favicon.png"
#define DEFAULT_ICON_MIME_TYPE "image/png"

NS_INTERFACE_MAP_BEGIN(MwbFaviconLoader)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIProtocolHandler)
  NS_INTERFACE_MAP_ENTRY(nsIProtocolHandler)
NS_INTERFACE_MAP_END

struct MwbFaviconClosure
{
  MwbFaviconClosure (nsIChannel *_channel,
                     nsIOutputStream *_output_stream)
    : channel (_channel),
      output_stream (_output_stream)
  {
  }

  static void Destroy (gpointer data);
  static void LoadedCb (MhsHistory *history,
                        const gchar *mime_type,
                        const guint8 *data,
                        guint data_len,
                        const GError *error,
                        gpointer user_data);

private:
  nsCOMPtr<nsIChannel> channel;
  nsCOMPtr<nsIOutputStream> output_stream;
};

MwbFaviconLoader *MwbFaviconLoader::favicon_loader = nsnull;

MwbFaviconLoader *
MwbFaviconLoader::GetSingleton(void)
{
  if (!favicon_loader)
    favicon_loader = new MwbFaviconLoader ();

  return favicon_loader;
}

MwbFaviconLoader::MwbFaviconLoader ()
{
  history = mhs_history_new ();
  default_favicon = NULL;
}

MwbFaviconLoader::~MwbFaviconLoader ()
{
  g_free (default_favicon);
  g_object_unref (history);
  favicon_loader = nsnull;
}

NS_IMETHODIMP_(nsrefcnt)
MwbFaviconLoader::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MwbFaviconLoader::Release ()
{
  return 1;
}

NS_IMETHODIMP
MwbFaviconLoader::GetScheme (nsACString &aScheme)
{
  aScheme.AssignLiteral ("mwb-favicon");

  return NS_OK;
}

NS_IMETHODIMP
MwbFaviconLoader::GetDefaultPort (PRInt32 *aDefaultPort)
{
  /* There is no default port */
  *aDefaultPort = -1;

  return NS_OK;
}

NS_IMETHODIMP
MwbFaviconLoader::GetProtocolFlags (PRUint32 *aProtocolFlags)
{
  *aProtocolFlags = (URI_DANGEROUS_TO_LOAD
                     | URI_IS_LOCAL_RESOURCE
                     | URI_NORELATIVE
                     | URI_NOAUTH);

  return NS_OK;
}

NS_IMETHODIMP
MwbFaviconLoader::NewURI (const nsACString &aSpec,
                          const char *aOriginCharset,
                          nsIURI *aBaseURI,
                          nsIURI **_retval NS_OUTPARAM)
{
  nsresult rv;
  nsCOMPtr<nsIURI> uri = do_CreateInstance (NS_SIMPLEURI_CONTRACTID, &rv);
  NS_ENSURE_SUCCESS (rv, rv);
  rv = uri->SetSpec (aSpec);
  NS_ENSURE_SUCCESS(rv, rv);

  *_retval = nsnull;
  uri.swap (*_retval);

  return NS_OK;
}

NS_IMETHODIMP
MwbFaviconLoader::NewChannel (nsIURI *aURI,
                              nsIChannel **_retval NS_OUTPARAM)
{
  nsresult rv;

  /* Create a pipe which we can drop the data on when the favicon is
     loaded */
  nsCOMPtr<nsIPipe> pipe = do_CreateInstance ("@mozilla.org/pipe;1", &rv);
  NS_ENSURE_SUCCESS (rv, rv);
  rv = pipe->Init (PR_TRUE, PR_TRUE, 8 * 1024, 1, nsnull);
  NS_ENSURE_SUCCESS (rv, rv);
  /* Get the streams for the pipe */
  nsCOMPtr<nsIAsyncInputStream> input_stream;
  nsCOMPtr<nsIAsyncOutputStream> output_stream;
  rv = pipe->GetInputStream (getter_AddRefs (input_stream));
  NS_ENSURE_SUCCESS (rv, rv);
  rv = pipe->GetOutputStream (getter_AddRefs (output_stream));
  NS_ENSURE_SUCCESS (rv, rv);

  /* Get a channel for the pipe */
  nsCOMPtr<nsIChannel> channel;
  rv = NS_NewInputStreamChannel (getter_AddRefs (channel), aURI,
                                 input_stream, EmptyCString ());
  NS_ENSURE_SUCCESS (rv, rv);

  /* Extract the URL for which we want the favicon */
  nsCAutoString path;
  const gchar *path_p;
  rv = aURI->GetPath (path);
  NS_ENSURE_SUCCESS (rv, rv);
  path_p = path.get ();
  while (*path_p == '/')
    path_p++;

  /* Start fetching the favicon from MHS */
  MwbFaviconClosure *closure = new MwbFaviconClosure (channel, output_stream);
  mhs_history_get_favicon (history, path_p, FALSE,
                           MwbFaviconClosure::LoadedCb,
                           closure,
                           MwbFaviconClosure::Destroy);

  channel.forget (_retval);

  return NS_OK;
}

NS_IMETHODIMP
MwbFaviconLoader::AllowPort (PRInt32 port,
                             const char *scheme,
                             PRBool *_retval NS_OUTPARAM)
{
  *_retval = PR_FALSE;

  return NS_OK;
}

void
MwbFaviconClosure::Destroy (gpointer data)
{
  delete (MwbFaviconClosure *) data;
}

void
MwbFaviconClosure::LoadedCb (MhsHistory *history,
                             const gchar *mime_type,
                             const guint8 *data,
                             guint data_len,
                             const GError *error,
                             gpointer user_data)
{
  MwbFaviconClosure *self = (MwbFaviconClosure *) user_data;
  nsresult rv;
  MwbFaviconLoader *loader = MwbFaviconLoader::favicon_loader;

  /* If the error is just that the icon isn't available then
     resort to the default favicon */
  if (error &&
      error->domain == MHS_ERROR &&
      error->code == MHS_ERROR_NOTAVAILABLE &&
      loader)
    {
      if (loader->default_favicon == NULL)
        {
          GError *file_error = NULL;

          if (!g_file_get_contents (DEFAULT_ICON,
                                    &loader->default_favicon,
                                    &loader->default_favicon_length,
                                    &file_error))
            {
              loader->default_favicon = NULL;
              g_warning ("Error getting default favicon: %s", error->message);
              g_clear_error (&file_error);
            }
        }

      if (loader->default_favicon)
        {
          mime_type = DEFAULT_ICON_MIME_TYPE;
          data = (guint8 *) loader->default_favicon;
          data_len = loader->default_favicon_length;
          error = NULL;
        }
    }

  if (error)
    g_warning ("%s", error->message);
  else
    {
      PRUint32 written;

      self->channel->SetContentType (nsDependentCString (mime_type));
      while (data_len > 0)
        {
          rv = self->output_stream->Write ((const char *) data,
                                           data_len, &written);
          if (NS_FAILED (rv))
            break;
          data += written;
          data_len -= written;
        }
    }

  self->output_stream->Close ();
}
