/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include "mwb-find-bar.h"
#include "mwb-separator.h"

G_DEFINE_TYPE (MwbFindBar, mwb_find_bar, NBTK_TYPE_BOX_LAYOUT)

#define FIND_BAR_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_FIND_BAR, MwbFindBarPrivate))

enum
{
  CHANGED,
  FIND_NEXT,
  FIND_PREV,
  CLOSE,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

struct _MwbFindBarPrivate
{
  NbtkWidget *entry;
  gboolean    backwards;
};


static void
mwb_find_bar_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_find_bar_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_find_bar_dispose (GObject *object)
{
  G_OBJECT_CLASS (mwb_find_bar_parent_class)->dispose (object);
}

static void
mwb_find_bar_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_find_bar_parent_class)->finalize (object);
}

static void
mwb_find_bar_class_init (MwbFindBarClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbFindBarPrivate));

  object_class->get_property = mwb_find_bar_get_property;
  object_class->set_property = mwb_find_bar_set_property;
  object_class->dispose = mwb_find_bar_dispose;
  object_class->finalize = mwb_find_bar_finalize;

  signals[CHANGED] =
    g_signal_new ("changed",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbFindBarClass, changed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  signals[FIND_NEXT] =
    g_signal_new ("find-next",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbFindBarClass, find_next),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  signals[FIND_PREV] =
    g_signal_new ("find-prev",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbFindBarClass, find_prev),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  signals[CLOSE] =
    g_signal_new ("close",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbFindBarClass, close),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

static void
mwb_find_bar_activate_cb (ClutterText *text,
                          MwbFindBar  *bar)
{
  const gchar *string;
  MwbFindBarPrivate *priv = bar->priv;

  string = clutter_text_get_text (text);
  if (priv->backwards)
    g_signal_emit (bar, signals[FIND_PREV], 0, string);
  else
    g_signal_emit (bar, signals[FIND_NEXT], 0, string);
}

static void
mwb_find_bar_text_changed_cb (ClutterText *text,
                              MwbFindBar  *bar)
{
  bar->priv->backwards = FALSE;
  g_signal_emit (bar, signals[CHANGED], 0, clutter_text_get_text (text));
}

static void
mwb_find_bar_prev_clicked_cb (NbtkButton *button,
                              MwbFindBar *bar)
{
  const gchar *string;
  MwbFindBarPrivate *priv = bar->priv;

  string = clutter_text_get_text (CLUTTER_TEXT (
             nbtk_entry_get_clutter_text (NBTK_ENTRY (priv->entry))));

  priv->backwards = TRUE;
  g_signal_emit (bar, signals[FIND_PREV], 0, string);
}

static void
mwb_find_bar_next_clicked_cb (NbtkButton *button,
                              MwbFindBar *bar)
{
  const gchar *string;
  MwbFindBarPrivate *priv = bar->priv;

  string = clutter_text_get_text (CLUTTER_TEXT (
             nbtk_entry_get_clutter_text (NBTK_ENTRY (priv->entry))));

  priv->backwards = FALSE;
  g_signal_emit (bar, signals[FIND_NEXT], 0, string);
}

static void
mwb_find_bar_close_clicked_cb (NbtkButton *button,
                               MwbFindBar *bar)
{
  bar->priv->backwards = FALSE;
  g_signal_emit (bar, signals[CLOSE], 0);
}

static void
mwb_find_bar_init (MwbFindBar *self)
{
  NbtkWidget *label, *prev, *next, *sep, *close;

  MwbFindBarPrivate *priv = self->priv = FIND_BAR_PRIVATE (self);

  nbtk_box_layout_set_spacing (NBTK_BOX_LAYOUT (self), 16);

  /* Create widgets */
  label = nbtk_label_new (_("Find in Page"));
  priv->entry = nbtk_entry_new ("");
  nbtk_entry_set_hint_text (NBTK_ENTRY (priv->entry),
                            _("Type to search"));
  next = nbtk_button_new_with_label (_("Next result"));
  prev = nbtk_button_new_with_label (_("Previous result"));
  sep = mwb_separator_new ();
  close = nbtk_button_new ();
  clutter_actor_set_name (CLUTTER_ACTOR (close), "close");

  /* Pack */
  clutter_container_add (CLUTTER_CONTAINER (self),
                         CLUTTER_ACTOR (label),
                         CLUTTER_ACTOR (priv->entry),
                         CLUTTER_ACTOR (prev),
                         CLUTTER_ACTOR (next),
                         CLUTTER_ACTOR (sep),
                         CLUTTER_ACTOR (close),
                         NULL);
  clutter_container_child_set (CLUTTER_CONTAINER (self),
                               CLUTTER_ACTOR (label),
                               "y-fill", FALSE,
                               NULL);
  clutter_container_child_set (CLUTTER_CONTAINER (self),
                               CLUTTER_ACTOR (priv->entry),
                               "expand", TRUE,
                               NULL);
  clutter_container_child_set (CLUTTER_CONTAINER (self),
                               CLUTTER_ACTOR (close),
                               "x-fill", FALSE,
                               "y-fill", FALSE,
                               NULL);

  /* Connect signals */
  g_signal_connect (nbtk_entry_get_clutter_text (NBTK_ENTRY (priv->entry)),
                    "activate",
                    G_CALLBACK (mwb_find_bar_activate_cb),
                    self);
  g_signal_connect (nbtk_entry_get_clutter_text (NBTK_ENTRY (priv->entry)),
                    "text-changed",
                    G_CALLBACK (mwb_find_bar_text_changed_cb),
                    self);
  g_signal_connect (prev, "clicked",
                    G_CALLBACK (mwb_find_bar_prev_clicked_cb), self);
  g_signal_connect (next, "clicked",
                    G_CALLBACK (mwb_find_bar_next_clicked_cb), self);
  g_signal_connect (close, "clicked",
                    G_CALLBACK (mwb_find_bar_close_clicked_cb), self);
}

NbtkWidget *
mwb_find_bar_new (void)
{
  return g_object_new (MWB_TYPE_FIND_BAR, NULL);
}

void
mwb_find_bar_focus (MwbFindBar *bar)
{
  MwbFindBarPrivate *priv = bar->priv;
  clutter_actor_grab_key_focus (CLUTTER_ACTOR (priv->entry));
}
