/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

const MwbIStartPageObserver
  = Components.interfaces.MwbIStartPageObserver;
const nsISupports = Components.interfaces.nsISupports;
const nsISupportsWeakReference = Components.interfaces.nsISupportsWeakReference;

const N_PINNED_PAGES = 5;

function FetchFavsObserver()
{
}

FetchFavsObserver.prototype =
{
  favoritesReceived : function(favorites, nFavorites)
  {
    // Remove all existing favorite links
    var favDiv = document.getElementById("favorites");
    while (favDiv.childNodes.length > 0)
      favDiv.removeChild(favDiv.childNodes[0]);

    if (nFavorites < 1)
      {
        // If there are no favorites then display a helpful page instead
        var req = new XMLHttpRequest();
        req.open('GET',
                 'no-history-fallback.xhtml',
                 false);
        req.send(null);
        favDiv.innerHTML = "";
        favDiv.appendChild(req.responseXML.getElementById("no-history-root"));
      }
    else
      {
        var h1 = document.createElement('h1');
        h1.appendChild(document.createTextNode(_("Favorite pages")));
        favDiv.appendChild(h1);

        var ioService = Components.classes["@mozilla.org/network/io-service;1"]
        .getService(Components.interfaces.nsIIOService);

        // Get the user's home directory
        var dirService
        = (Components.classes["@mozilla.org/file/directory_service;1"]
           .getService(Components.interfaces.nsIProperties));
        var homeDir = dirService.get("Home", Components.interfaces.nsIFile);
        var thumbnailDir = homeDir.clone();

        thumbnailDir.append(".thumbnails");
        thumbnailDir.append("large");

        // Get a crypto hash component instance for calculating MD5s
        var cryptoHash = Components.classes["@mozilla.org/security/hash;1"]
        .createInstance(Components.interfaces.nsICryptoHash);

        for (fav = 0; fav < nFavorites; fav++)
          {
            var div = document.createElement('div');
            div.setAttribute('class', 'favorite');
            var a = document.createElement('a');
            div.appendChild(a);
            var img = document.createElement('img');
            img.setAttribute('width', '223');
            img.setAttribute('height', '111');
            a.appendChild(img);
            a.appendChild(document.createElement('br'));
            var p = document.createElement('p');
            a.appendChild(p);

            p.appendChild(document.createTextNode(favorites[fav].title));
            a.setAttribute("href", favorites[fav].url);

            // Get an MD5 sum of the URL
            cryptoHash.init(cryptoHash.MD5);
            cryptoHash.update([favorites[fav].url.charCodeAt(i)
                               for (i in favorites[fav].url)],
                              favorites[fav].url.length);
            var binHash = cryptoHash.finish(false);
            var strHash = "";
            // Convert it to ascii hex
            for (i = 0; i < binHash.length; i++)
              {
                if (binHash.charCodeAt(i) < 16)
                  strHash += "0";
                strHash += binHash.charCodeAt(i).toString(16);
              }
            // Build a filename of the thumbnail using the hash
            var thumbnailFile = thumbnailDir.clone();
            thumbnailFile.append(strHash + ".png");
            // If the file doesn't exist then use a replacement instead
            if (thumbnailFile.exists())
              {
                // Convert it to a URI
                var thumbnailURI = ioService.newFileURI(thumbnailFile);
                img.setAttribute("src", thumbnailURI.spec);
              }
            else
              img.setAttribute("src", "fallback-page.png");

            favDiv.appendChild(div);
          }
      }
  },

  linkVisited : function(uri, visit_time)
  {
    var i;

    // Check if this page is pinned
    for (i = 0; i < document.pinnedPages.length; i++)
      if (document.pinnedPages[i].uri == uri)
        {
          // Update the visit time and rebuilt the list
          document.pinnedPages[i].visit_time = visit_time;
          rebuildPinnedPagesList();
          break;
        }
  },

  pinnedPage : function(title, uri, visit_time)
  {
    var i;

    // Check if we already have the page in the list
    for (i = 0; i < document.pinnedPages.length; i++)
      if (document.pinnedPages[i].uri == uri)
        {
          // Just update the details
          document.pinnedPages[i].title = title;
          document.pinnedPages[i].visit_time = visit_time;
          break;
        }
    if (i >= document.pinnedPages.length)
      // New entry
      document.pinnedPages.push({ title : title,
                                  uri : uri,
                                  visit_time : visit_time });

    rebuildPinnedPagesList();
  },

  unpinnedPage : function(uri)
  {
    var i;

    // Find the list item with this uri
    for (i = 0; i < document.pinnedPages.length; i++)
      if (document.pinnedPages[i].uri == uri)
        {
          document.pinnedPages.splice(i, 1);
          rebuildPinnedPagesList();
          break;
        }
  },

  QueryInterface : function(aIID)
  {
    if (!aIID.equals(MwbIStartPageObserver)
        && !aIID.equals(nsISupports)
        && !aIID.equals(nsISupportsWeakReference))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

function formatTimeForDisplay(time)
{
  // This is based on format_time_for_display from Tweet

  var now = new Date();
  var secsDiff = (now.getTime() - time.getTime()) / 1000;

  if (secsDiff < 60)
    return _("Less than a minute ago");

  var minsDiff = secsDiff / 60;

  if (minsDiff < 60)
    // Translators: The two underscores here will get replaced with a
    // number so they should be in the translation too. Eg 'About __
    // minutes ago' will become 'About 10 minutes ago'.
    return ngettext ("About a minute ago",
                     "About __ minutes ago",
                     minsDiff).replace("__", Math.floor(minsDiff));
  if (minsDiff < 360)
    // Translators: The two underscores here will get replaced with a
    // number so they should be in the translation too. Eg 'About __
    // minutes ago' will become 'About 10 minutes ago'.
    return ngettext ("About an hour ago",
                     "About __ hours ago",
                     minsDiff / 60).replace("__", Math.floor(minsDiff / 60));

  // Convert both times to midday on the day they represent so we can
  // compare in whole days
  var nowMidday = new Date(now.getYear(), now.getMonth(), now.getDay(), 12);
  var timeMidday = new Date(time.getYear(), time.getMonth(), time.getDay(), 12);
  var daysDiff = ((nowMidday.getTime() - timeMidday.getTime())
                  / (24 * 3600 * 1000));
  var format;

  if (daysDiff < 1)
    format = _("Today at %H:%M");
  else if (daysDiff < 2)
    format = _("Yesterday at %H:%M");
  else if (daysDiff < 7)
    format = _("Last %A at %H:%M"); // day of the week
  else
    format = _("%x at %H:%M"); // any other date

  return time.toLocaleFormat(format);
}

function rebuildPinnedPagesList()
{
  var div = document.getElementById('pinned_pages_div');
  var i;

  // Resort the list according to descending visit time order
  document.pinnedPages.sort(function(a, b)
                            {
                              return b.visit_time - a.visit_time;
                            });

  // Clear all existing entries
  while (div.childNodes.length > 0)
    div.removeChild(div.childNodes[0]);

  var nPages = document.pinnedPages.length;
  if (!document.showMorePinnedPages && nPages > N_PINNED_PAGES)
    nPages = N_PINNED_PAGES;

  // If there are no pages then display some instructions instead
  if (nPages <= 0)
    div.innerHTML = (_("<p>Push the pin icon in the location bar above to "
                       + "add pages to this list.</p>"));
  else
    {
      // Add each new entry
      for (i = 0; i < nPages; i++)
        {
          var page = document.pinnedPages[i];
          var a = document.createElement('a');
          var pageDiv = document.createElement('div');
          var imgDiv = document.createElement('div');
          var img = document.createElement('img');
          var innerDiv = document.createElement('div');
          var titleSpan = document.createElement('span');
          var visitTimeSpan = document.createElement('span');

          a.setAttribute('href', page.uri);
          pageDiv.setAttribute('class', 'pinned_page_div');
          imgDiv.setAttribute('class', 'pinned_favicon_div');
          img.setAttribute('class', 'pinned_favicon');
          img.setAttribute('src', 'mwb-favicon://' + page.uri);
          imgDiv.appendChild(img);
          pageDiv.appendChild(imgDiv);
          innerDiv.setAttribute('class', 'pinned_inner_div');
          titleSpan.setAttribute('class', 'pinned_title');
          titleSpan.appendChild(document.createTextNode(page.title + " "));
          innerDiv.appendChild(titleSpan);
          visitTimeSpan.setAttribute('class', 'pinned_visit_time');
          var visitTime = formatTimeForDisplay(new Date(page.visit_time
                                                        * 1000));
          visitTimeSpan.appendChild(document.createTextNode(visitTime));
          innerDiv.appendChild(visitTimeSpan);
          pageDiv.appendChild(innerDiv);
          a.appendChild(pageDiv);
          div.appendChild(a);
        }

      if (document.pinnedPages.length > N_PINNED_PAGES)
        {
          var a = document.createElement('a');
          var func, text;
          if (document.showMorePinnedPages)
            {
              func = showLessPinnedPages;
              text = _("Fewer pinned pages");
            }
          else
            {
              func = showMorePinnedPages;
              text = _("More pinned pages");
            }
          a.setAttribute('class', 'pinned_more_link');
          a.appendChild(document.createTextNode(text));
          a.addEventListener("click", func, false);
          var p = document.createElement('p');
          p.appendChild(a);
          div.appendChild(p);
        }
    }
}

function showMorePinnedPages()
{
  document.showMorePinnedPages = true;
  rebuildPinnedPagesList();
}

function showLessPinnedPages()
{
  document.showMorePinnedPages = false;
  rebuildPinnedPagesList();
}

function makeTabPrivate()
{
  var startPageService
    = Components.classes["@moblin.org/mwb-start-page-service;1"]
    .getService(Components.interfaces.MwbIStartPage);
  startPageService.startPrivateBrowsing();
}

var observer;

// Run the initialisation once the page has finished loading
document.addEventListener("DOMContentLoaded", function(event)
{
  // List of pinned pages. Initially empty
  document.pinnedPages = [];
  // Whether to show all pinned pages or just a subset
  document.showMorePinnedPages = false;

  var startPageService
    = Components.classes["@moblin.org/mwb-start-page-service;1"]
    .getService(Components.interfaces.MwbIStartPage);

  observer = new FetchFavsObserver();

  // Add our observer for when the favorites become available
  startPageService.addStartPageObserver(observer);

  // Start fetching the favorites immediately
  startPageService.getFavorites();
  // and the pinned pages
  startPageService.getPinnedPages();
  rebuildPinnedPagesList();

  var privateButton = document.getElementById("make_private_button");
  privateButton.addEventListener("click", makeTabPrivate, false);

  // Set a style on the body tag if we are in private mode so that we
  // can style the page differently
  if (location.search.match(/^\?(?:|.+&)private(?:$|&)/))
    {
      var body = document.getElementsByTagName("body")[0];
      body.setAttribute("class", "private_mode");
    }

  // Unregister the load event so we don't initialize more than once
  event.currentTarget.removeEventListener("DOMContentLoaded",
                                          arguments.callee,
                                          false);
}, false);
