#!/usr/bin/perl -w

# Moblin-Web-Browser: The web browser for Moblin
# Copyright (c) 2009, Intel Corporation.
#
# This program is free software; you can redistribute it and/or modify it
# under the terms and conditions of the GNU Lesser General Public License,
# version 2.1, as published by the Free Software Foundation.
#
# This program is distributed in the hope it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
# License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

use strict;
use warnings;
use File::Basename;
use XML::Parser;

# This script looks up the values for all the entities defined in a
# DTD and generates a new DTD containing the translations of those
# values.

my %entities;
my %translations;

my $msgid = "";
my $msgstr = "";
my $got_msgid = 0;
my $got_msgstr = 0;
my @flags;

sub unescape_char
{
    my ($ch) = @_;

    return ($ch eq "n" ? "\n" :
            $ch eq "t" ? "\t" :
            $ch eq "r" ? "\r" :
            $ch);
}

sub unescape_string
{
    my ($str) = @_;

    $str =~ s/\\(.)/unescape_char($1)/eg;

    return $str;
}

sub handle_entity
{
    my ($parser, $name, $value) = @_;

    $entities{$name} = $value;
}

sub flush_entry
{
    if ($got_msgid && $got_msgstr && !grep(/\Afuzzy\z/, @flags))
    {
        $translations{$msgid} = $msgstr;
    }
    @flags = ();
    $got_msgid = 0;
    $got_msgstr = 0;
}

sub parse_po
{
    my ($fn) = @_;

    %translations = ();

    my $fin;

    # Use msgconv to convert the po file to UTF-8
    open $fin, "-|", "msgconv -t UTF-8 '$fn'"
        or die("Failed to open $fn");

    # Loose parser for the .po file
    while (my $line = <$fin>)
    {
        if ($line =~ /^\s*$/)
        {
            flush_entry();
        }
        elsif ($line =~ /^msgid \"(.*)\"\s*$/)
        {
            $msgid = unescape_string($1);
            $got_msgid = 1;
        }
        elsif ($line =~ /^msgstr \"(.*)\"\s*$/)
        {
            $msgstr = unescape_string($1);
            $got_msgstr = 1;
        }
        elsif ($line =~ /^\"(.*)\"\s*$/)
        {
            if ($got_msgstr)
            {
                $msgstr .= unescape_string($1);
            }
            else
            {
                $msgid .= unescape_string($1);
            }
        }
        elsif ($line =~ /^#, (.*?)\s*$/)
        {
            @flags = split(/,/, $1);
        }
    }

    flush_entry();

    close $fin;
}

sub parse_dtd
{
    my ($dtd_file) = @_;
    %entities = ();

    my $parser = new XML::Parser(Handlers => { Entity => \&handle_entity });

    my $fin;
    open $fin, "<", $dtd_file or die("Failed to open $dtd_file");
    # Get the whole file contents
    my $contents;
    read $fin, $contents, 1 << 30;
    close $fin;

    # Wrap the contents up as a complete XHTML document
    $contents = ("<?xml version=\"1.0\"?>\n" .
                 "<!DOCTYPE html PUBLIC " .
                 "\"-//W3C//DTD XHTML 1.0 Strict//EN\"\n" .
                 "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\" [\n" .
                 "$contents\n" .
                 "]>\n" .
                 "<html />\n");

    # Give it to the XML parser so it will tell us about the entities
    $parser->parse($contents);
}

die("usage: $0 <LINGUAS-file> <locale-dir> [dtd]...") unless @ARGV >= 2;

my $all_linguas = shift(@ARGV);
my $locale_dir = shift(@ARGV);
my @linguas = split(/\s+/, $all_linguas);

foreach my $lingua (@linguas)
{
    parse_po(dirname($0) . "/../../po/$lingua.po");

    foreach my $dtd (@ARGV)
    {
        parse_dtd($dtd);

        my $fout;

        my $lingua_fn = $lingua;
        $lingua_fn =~ s/_/-/g;

        my $lingua_dir = "$locale_dir/$lingua_fn";
        unless (-d $lingua_dir)
        {
            mkdir($lingua_dir) or die("Failed to mkdir $lingua_dir");
        }

        $lingua_fn = "$lingua_dir/" . basename($dtd);

        open $fout, ">", "$lingua_fn" or die("failed to $lingua_fn");

        # Write each entity back out
        while (my ($entity, $value) = each %entities)
        {
            $value = $translations{$value} || $value;
            $value =~ s/\"/&quot;/g;

            print $fout "<!ENTITY $entity \"$value\">\n";
        }

        close $fout;

        %entities = ();
    }
}
