/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <gtk/gtk.h>
#include <clutter-gtk/clutter-gtk.h>
#include <gdk/gdkx.h>
#include <clutter/x11/clutter-x11.h>

#include "mwb-window.h"
#include "mwb-browser.h"

G_DEFINE_TYPE (MwbWindow, mwb_window, GTK_TYPE_WINDOW)

#define WINDOW_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_WINDOW, MwbWindowPrivate))

enum
{
  PROP_0,
  PROP_INTERNAL_ACTOR,
};

struct _MwbWindowPrivate
{
  GtkWidget       *layout;
  GtkClutterEmbed *embed;
  ClutterActor    *internal;
};

static GdkFilterReturn
gdk_to_clutter_event_forward (GdkXEvent *xevent,
                              GdkEvent *event,
                              gpointer mwb_window);

static void
mwb_window_set_internal_actor (MwbWindow *self, ClutterActor *actor);


static void
mwb_window_get_property (GObject *object, guint property_id,
                         GValue *value, GParamSpec *pspec)
{
  MwbWindowPrivate *priv = MWB_WINDOW (object)->priv;

  switch (property_id)
    {
    case PROP_INTERNAL_ACTOR:
      g_value_set_object (value, priv->internal);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_window_set_property (GObject *object, guint property_id,
                        const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    case PROP_INTERNAL_ACTOR:
      mwb_window_set_internal_actor (MWB_WINDOW (object),
                                    g_value_get_object (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_window_dispose (GObject *object)
{
  MwbWindowPrivate *priv = MWB_WINDOW (object)->priv;

  gdk_window_remove_filter (NULL, gdk_to_clutter_event_forward, NULL);

  if (priv->internal && priv->embed)
    mwb_window_set_internal_actor (MWB_WINDOW (object), NULL);

  if (priv->embed)
    {
      gtk_container_remove (GTK_CONTAINER (priv->layout),
                            GTK_WIDGET (priv->embed));
      priv->embed = NULL;
    }

  if (priv->layout)
    {
      gtk_container_remove (GTK_CONTAINER (object),
                            GTK_WIDGET (priv->layout));
      priv->layout = NULL;
    }

  G_OBJECT_CLASS (mwb_window_parent_class)->dispose (object);
}

static void
mwb_window_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_window_parent_class)->finalize (object);
}

/* XXX: The intention is that we could seperate this class out - potentially
 * into NBTK - and the default internal actor would be somthing like a
 * ClutterGroup or just NULL, and we would instead sub-class and override this
 * for the browser. */
static void
mwb_window_create_internal_actor_real (MwbWindow *self)
{
  MwbBrowser *mwb_browser = MWB_BROWSER (mwb_browser_new (self));
  mwb_window_set_internal_actor (self, CLUTTER_ACTOR (mwb_browser));
}

static gint
mwb_window_key_press_event (GtkWidget   *widget,
                            GdkEventKey *event)
{
  GtkWindow *window = GTK_WINDOW (widget);

  return gtk_window_propagate_key_event (window, event);
}

static void
mwb_window_class_init (MwbWindowClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  MwbWindowClass *window_class = MWB_WINDOW_CLASS (klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbWindowPrivate));

  object_class->get_property = mwb_window_get_property;
  object_class->set_property = mwb_window_set_property;
  object_class->dispose = mwb_window_dispose;
  object_class->finalize = mwb_window_finalize;
  /* we don't want the default keyboard accelerators since, for example, they
   * break tabbing through web forms... */
  widget_class->key_press_event = mwb_window_key_press_event;

  window_class->create_internal_actor =
    mwb_window_create_internal_actor_real;

  g_object_class_install_property (object_class,
                                   PROP_INTERNAL_ACTOR,
                                   g_param_spec_object ("internal-actor",
                                                        "Embedded actor",
                                                        "Then internally "
                                                        "embedded actor",
                                                        CLUTTER_TYPE_ACTOR,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

}

GtkWidget *
mwb_window_new (void)
{
  return g_object_new (MWB_TYPE_WINDOW, NULL);
}

static void
on_stage_resize (GObject *object, GParamSpec *pspec, gpointer data)
{
  ClutterActor *stage = CLUTTER_ACTOR (object);
  GList *children = clutter_container_get_children (CLUTTER_CONTAINER (stage));
  ClutterActor *internal_child;
  ClutterGeometry stage_geom;

  if (!children)
    return;

  clutter_actor_get_geometry (stage, &stage_geom);

  internal_child = children->data;
  g_list_free (children);

  clutter_actor_set_size (internal_child, stage_geom.width, stage_geom.height);
}

static GdkFilterReturn
gdk_to_clutter_event_forward (GdkXEvent *xevent,
                              GdkEvent *event,
                              gpointer mwb_window)
{
  MwbWindowPrivate *priv = MWB_WINDOW (mwb_window)->priv;

  if (((XEvent*)xevent)->type == ButtonPress)
    gtk_widget_grab_focus (GTK_WIDGET (priv->embed));

  if (((XAnyEvent*)xevent)->window ==
      GDK_WINDOW_XWINDOW (gtk_widget_get_window (priv->layout)))
    {
      /* Forward the events on layout to GtkClutterEmbed */
      ((XAnyEvent*)xevent)->window =
        GDK_WINDOW_XWINDOW (gtk_widget_get_window (GTK_WIDGET (priv->embed)));
    }

  switch (clutter_x11_handle_event ((XEvent*)xevent))
    {
    default:
    case CLUTTER_X11_FILTER_CONTINUE:
      return GDK_FILTER_CONTINUE;
    case CLUTTER_X11_FILTER_TRANSLATE:
      return GDK_FILTER_TRANSLATE;
    case CLUTTER_X11_FILTER_REMOVE:
      return GDK_FILTER_REMOVE;
    }

  return GDK_FILTER_CONTINUE;
}

static void
on_layout_size_allocated (GtkWidget *widget,
                          GtkAllocation *allocation,
                          gpointer clutter_embed)
{
  gtk_widget_size_allocate (GTK_WIDGET (clutter_embed), allocation);
}

static void
mwb_window_init (MwbWindow *self)
{
  MwbWindowPrivate *priv = self->priv = WINDOW_PRIVATE (self);
  MwbWindowClass *window_class = MWB_WINDOW_GET_CLASS (self);
  ClutterActor *stage;
  gint event_mask;

  /* create a GtkFixed to contain GtkClutterEmbed and GtkPluginViewport */
  priv->layout = gtk_fixed_new ();
  gtk_fixed_set_has_window (GTK_FIXED (priv->layout), TRUE);
  priv->embed = GTK_CLUTTER_EMBED (gtk_clutter_embed_new ());

  gtk_container_add (GTK_CONTAINER (priv->layout), GTK_WIDGET (priv->embed));
  gtk_container_add (GTK_CONTAINER (self), priv->layout);

  g_signal_connect (G_OBJECT (priv->layout), "size-allocate",
                    G_CALLBACK (on_layout_size_allocated), priv->embed);

  gtk_widget_show (GTK_WIDGET (priv->embed));

  if (window_class->create_internal_actor)
    window_class->create_internal_actor (self);

  stage = gtk_clutter_embed_get_stage (priv->embed);
  g_signal_connect (G_OBJECT (stage),
                    "notify::allocation",
                    G_CALLBACK (on_stage_resize),
                    NULL);

  /* proxy the events to clutter stage */
  event_mask = GDK_BUTTON_PRESS_MASK
    | GDK_BUTTON_RELEASE_MASK
    | GDK_KEY_PRESS_MASK
    | GDK_KEY_RELEASE_MASK
    | GDK_POINTER_MOTION_MASK;

  gtk_widget_add_events (priv->layout, event_mask);

  gdk_window_add_filter (NULL, gdk_to_clutter_event_forward, self);
}

/**
 * mwb_window_fullscreen:
 * @self: A #MwbWindow object
 *
 * Attempts to fullscreen the window.
 */
void
mwb_window_fullscreen (MwbWindow *self)
{
#ifndef DISABLE_MUTTER_MOBLIN_FULLSCREEN_HACK
  GdkScreen *screen = gdk_screen_get_default ();
  gint width, height;

  /* XXX: This is a hack so that mutter doesn't disable the panel as it would
   * for a real fullscreen window.
   *
   * I don't know what would happen if the user were e.g. using RandR to group
   * multiple displays under one screen.
   *
   * It might be better to instead set a special _MOBLIN_FULLSCREEN_PROPERTY
   */
  gtk_window_set_decorated (GTK_WINDOW (self), FALSE);
  width = gdk_screen_get_width (screen);
  height = gdk_screen_get_height (screen);
  gtk_window_resize (GTK_WINDOW (self), width, height);
#else
  gtk_window_fullscreen (GTK_WINDOW (self));
#endif
}

static void
mwb_window_set_internal_actor (MwbWindow *self, ClutterActor *actor)
{
  MwbWindowPrivate *priv = self->priv;
  ClutterActor *stage = gtk_clutter_embed_get_stage (priv->embed);

  if (priv->internal == actor)
    return;

  if (priv->internal)
    {
      g_object_set_data (G_OBJECT (priv->internal), "MwbWindow", NULL);
      clutter_container_remove_actor (CLUTTER_CONTAINER (stage),
                                      CLUTTER_ACTOR (priv->internal));
    }

  if (actor)
    {
      priv->internal = actor;
      clutter_container_add_actor (CLUTTER_CONTAINER (stage),
                                   CLUTTER_ACTOR (priv->internal));

      /* Not an ideal mechanism but it's used to allow the internal actor to
       * be able to query it's containing MbwStage.
       *
       * Potentially we could require implementation of a special interface
       * that would simply let us set a property on the child. */
      g_object_set_data (G_OBJECT (actor), "MwbWindow", self);
    }
}

/**
 * mwb_window_get_internal_actor:
 * @self: A #MwbWindow object
 *
 * Returns: the internal actor contained by the window (no reference is taken)
 */
ClutterActor *
mwb_window_get_internal_actor (MwbWindow *self)
{
  return self->priv->internal;
}

GtkWidget *
mwb_window_get_layout (MwbWindow *self)
{
  return self->priv->layout;
}
