/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <clutter-gtk/clutter-gtk.h>
#include <clutter-mozembed/clutter-mozembed.h>

#include "mwb-download.h"
#include "mwb-browser.h"
#include "mwb-utils.h"

enum
{
  PROP_0,

  PROP_DOWNLOAD,
  PROP_SRC,
  PROP_DEST,
  PROP_PROGRESS,
  PROP_MAX_PROGRESS,
  PROP_COMPLETED,
  PROP_CANCELLED
};

G_DEFINE_TYPE (MwbDownload, mwb_download, NBTK_TYPE_TABLE)

#define DOWNLOAD_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_DOWNLOAD, MwbDownloadPrivate))

struct _MwbDownloadPrivate
{
  ClutterActor *icon;
  NbtkWidget   *file_label;
  NbtkWidget   *progress_label;
  NbtkWidget   *progress_bar;
  NbtkWidget   *open_button;
  NbtkWidget   *cancel_button;
  ClutterActor *separator;

  goffset       previous_file_size;
  gint64        progress;
  gint64        max_progress;
  GFile        *file;
  GFile        *file_part;

  gboolean      completed;
  gboolean      cancelled;

  ClutterMozEmbedDownload *download;
};

static void
mwb_download_get_property (GObject *object, guint property_id,
                           GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static GtkWidget *
mwb_download_get_window (MwbDownload *download)
{
  GtkWidget *window = NULL;
  ClutterActor *stage = clutter_actor_get_stage (CLUTTER_ACTOR (download));

  if (stage)
    {
      ClutterActor *browser =
        clutter_container_find_child_by_name (
          CLUTTER_CONTAINER (stage), "browser");
      if (browser && MWB_IS_BROWSER (browser))
        g_object_get (G_OBJECT (browser),
                      "window", &window, NULL);
    }

  return window;
}

static GFileInfo *
mwb_download_query_file_info (MwbDownload *download,
                              const char *attributes)
{
  MwbDownloadPrivate *priv = download->priv;
  GFileInfo *info;

  info = g_file_query_info (priv->file, attributes, 0, NULL, NULL);
  if (info)
    return info;

  /* If that failed try getting info for the same filename but with
     ".part" appended */
  return g_file_query_info (priv->file_part, attributes, 0, NULL, NULL);
}

static void
mwb_download_update_icon (MwbDownload *download)
{
  MwbDownloadPrivate *priv = download->priv;

  if (priv->file)
    {
      GFileInfo *info;

      gboolean icon_set = FALSE;
      GtkWidget *window = mwb_download_get_window (download);

      /* If we can't get the window, we can't set the icon */
      if (!window)
        return;

      info = mwb_download_query_file_info (download,
                                           G_FILE_ATTRIBUTE_STANDARD_ICON);

      if (info)
        {
          GIcon *icon = g_file_info_get_icon (info);

          if (icon)
            {
              if (G_IS_THEMED_ICON (icon))
                {
                  gint i;
                  GStrv names;

                  g_object_get (G_OBJECT (icon), "names", &names, NULL);

                  for (i = 0; names[i]; i++)
                    {
                      if (gtk_clutter_texture_set_from_icon_name (
                            CLUTTER_TEXTURE (priv->icon),
                            window,
                            names[i],
                            GTK_ICON_SIZE_DIALOG,
                            NULL))
                        {
                          icon_set = TRUE;
                          break;
                        }
                    }
                  g_strfreev (names);
                }
              else if (G_IS_LOADABLE_ICON (icon))
                g_warning ("GLoadableIcon is unhandled");
              else if (G_IS_FILE_ICON (icon))
                g_warning ("GFileIcon is unhandled");
            }

          g_object_unref (info);
        }

      if (!icon_set)
        gtk_clutter_texture_set_from_stock (CLUTTER_TEXTURE (priv->icon),
                                            window,
                                            GTK_STOCK_FILE,
                                            GTK_ICON_SIZE_DIALOG,
                                            NULL);
    }
}

static void
mwb_download_update_progress (MwbDownload *download)
{
  gchar *progress_text;
  gdouble progress;

  MwbDownloadPrivate *priv = download->priv;

  progress = priv->progress / (gdouble)priv->max_progress;

  /* If the file has grown to over 4KB then try to get the icon
     again. We can't use priv->progress to determine the file size
     because Mozilla may not have flushed the data to disk yet */
  if (priv->file && priv->previous_file_size < 4096)
    {
      GFileInfo *info =
        mwb_download_query_file_info (download, G_FILE_ATTRIBUTE_STANDARD_SIZE);

      if (info)
        {
          gint64 size = g_file_info_get_size (info);

          if (size >= 4096)
            mwb_download_update_icon (download);

          priv->previous_file_size = size;

          g_object_unref (info);
        }
    }

  if (priv->max_progress < 4096)
    {
      progress_text = g_strdup_printf ("%d%%  %lld/%lldb",
                                       (gint)(progress * 100.0),
                                       priv->progress,
                                       priv->max_progress);
    }
  else if (priv->max_progress < (1024 * 1024))
    {
      progress_text = g_strdup_printf ("%d%%  %.1lf/%.1lfKb",
                                       (gint)(progress * 100.0),
                                       priv->progress / (gdouble)1024,
                                       priv->max_progress / (gdouble)1024);
    }
  else
    {
      progress_text = g_strdup_printf ("%d%%  %.2lf/%.2lfMb",
                                       (gint)(progress * 100.0),
                                       priv->progress / (gdouble)(1024 * 1024),
                                       priv->max_progress /
                                         (gdouble)(1024 * 1024));
    }

  nbtk_label_set_text (NBTK_LABEL (priv->progress_label), progress_text);
  nbtk_progress_bar_set_progress (NBTK_PROGRESS_BAR (priv->progress_bar),
                                  progress);
  g_free (progress_text);
}

static void
mwb_download_finish (MwbDownload *download, const gchar *text, gboolean disable)
{
  NbtkWidget *label;
  MwbDownloadPrivate *priv = download->priv;

  /* Update icon on completion in case the file was shorter than 4KB */
  mwb_download_update_icon (download);

  /* Rearrange widgets (hide everything, add a label) */
  clutter_actor_hide (CLUTTER_ACTOR (priv->progress_bar));
  clutter_actor_hide (CLUTTER_ACTOR (priv->progress_label));
  clutter_actor_hide (CLUTTER_ACTOR (priv->cancel_button));

  if (!disable)
    {
      clutter_actor_show (CLUTTER_ACTOR (priv->open_button));
      clutter_actor_queue_relayout (CLUTTER_ACTOR (priv->open_button));
    }

  label = nbtk_label_new (text);
  clutter_actor_set_name (CLUTTER_ACTOR (label), "subtitle");
  nbtk_table_add_actor_with_properties (NBTK_TABLE (download),
                                        CLUTTER_ACTOR (label),
                                        1, 1,
                                        "col-span", 2,
                                        "x-expand", TRUE,
                                        "x-fill", TRUE,
                                        "y-expand", FALSE,
                                        "y-fill", TRUE,
                                        NULL);
}

static void
mwb_download_cancel (MwbDownload *download)
{
  MwbDownloadPrivate *priv = download->priv;

  if (priv->cancelled)
    return;

  priv->cancelled = TRUE;

  mwb_download_finish (download, _("Download cancelled"), TRUE);
}

static void
mwb_download_complete (MwbDownload *download)
{
  MwbDownloadPrivate *priv = download->priv;

  if (priv->completed)
    return;

  priv->completed = TRUE;

  mwb_download_finish (download, _("Download completed"), FALSE);
}

static void
mwb_download_set_property (GObject *object, guint property_id,
                           const GValue *value, GParamSpec *pspec)
{
  MwbDownload *download = MWB_DOWNLOAD (object);
  MwbDownloadPrivate *priv = download->priv;

  switch (property_id)
    {
    case PROP_DOWNLOAD :
      priv->download = g_value_get_object (value);
      break;

    case PROP_SRC :
      break;

    case PROP_DEST :
      {
        const gchar *uri;

        uri = g_value_get_string (value);

        /* Don't do anything if the URI is the same */
        if (priv->file)
          {
            if (uri)
              {
                char *current_uri = g_file_get_uri (priv->file);
                gboolean no_change;

                no_change = !strcmp (current_uri, uri);
                g_free (current_uri);

                if (no_change)
                  break;
              }

            g_object_unref (priv->file);
            g_object_unref (priv->file_part);

            priv->file = NULL;
            priv->file_part = NULL;
          }

        if (uri)
          {
            gchar *file = g_filename_from_uri (uri, NULL, NULL);

            if (file)
              {
                gchar *base = g_filename_display_basename (file);
                gchar *uri_part = g_strconcat (uri, ".part", NULL);

                priv->file = g_file_new_for_uri (uri);
                priv->file_part = g_file_new_for_uri (uri_part);

                nbtk_label_set_text (NBTK_LABEL (priv->file_label), base);

                g_free (uri_part);
                g_free (base);
                g_free (file);
              }
          }

        mwb_download_update_icon (download);

        break;
      }

    case PROP_PROGRESS :
      priv->progress = g_value_get_int64 (value);
      mwb_download_update_progress (download);
      break;

    case PROP_MAX_PROGRESS :
      priv->max_progress = g_value_get_int64 (value);
      mwb_download_update_progress (download);
      break;

    case PROP_COMPLETED :
      if (g_value_get_boolean (value))
        mwb_download_complete (download);
      break;

    case PROP_CANCELLED :
      if (g_value_get_boolean (value))
        mwb_download_cancel (download);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_download_dispose (GObject *object)
{
  MwbDownload *download = MWB_DOWNLOAD (object);
  MwbDownloadPrivate *priv = download->priv;

  if (priv->file)
    {
      g_object_unref (priv->file);
      priv->file = NULL;
    }

  if (priv->file_part)
    {
      g_object_unref (priv->file_part);
      priv->file_part = NULL;
    }

  G_OBJECT_CLASS (mwb_download_parent_class)->dispose (object);
}

static void
mwb_download_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_download_parent_class)->finalize (object);
}

static void
mwb_download_class_init (MwbDownloadClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbDownloadPrivate));

  object_class->get_property = mwb_download_get_property;
  object_class->set_property = mwb_download_set_property;
  object_class->dispose = mwb_download_dispose;
  object_class->finalize = mwb_download_finalize;

  g_object_class_install_property (object_class,
                                   PROP_DOWNLOAD,
                                   g_param_spec_object ("download",
                                                        "Download",
                                                        "Download object.",
                                                        CLUTTER_TYPE_MOZEMBED_DOWNLOAD,
                                                        G_PARAM_WRITABLE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_SRC,
                                   g_param_spec_string ("source",
                                                        "Source URI",
                                                        "Download source URI.",
                                                        NULL,
                                                        G_PARAM_WRITABLE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_DEST,
                                   g_param_spec_string ("destination",
                                                        "Destination URI",
                                                        "Download destination "
                                                        "URI.",
                                                        NULL,
                                                        G_PARAM_WRITABLE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_PROGRESS,
                                   g_param_spec_int64 ("progress",
                                                       "Progress",
                                                       "Download progress.",
                                                       -1, G_MAXINT64, 0,
                                                       G_PARAM_WRITABLE |
                                                       G_PARAM_STATIC_NAME |
                                                       G_PARAM_STATIC_NICK |
                                                       G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_MAX_PROGRESS,
                                   g_param_spec_int64 ("max-progress",
                                                       "Max progress",
                                                       "Maximum download "
                                                       "progress.",
                                                       -1, G_MAXINT64, 0,
                                                       G_PARAM_WRITABLE |
                                                       G_PARAM_STATIC_NAME |
                                                       G_PARAM_STATIC_NICK |
                                                       G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_COMPLETED,
                                   g_param_spec_boolean ("completed",
                                                         "Completed",
                                                         "Download completed.",
                                                         FALSE,
                                                         G_PARAM_WRITABLE |
                                                         G_PARAM_STATIC_NAME |
                                                         G_PARAM_STATIC_NICK |
                                                         G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_CANCELLED,
                                   g_param_spec_boolean ("cancelled",
                                                         "Cancelled",
                                                         "Download cancelled.",
                                                         FALSE,
                                                         G_PARAM_WRITABLE |
                                                         G_PARAM_STATIC_NAME |
                                                         G_PARAM_STATIC_NICK |
                                                         G_PARAM_STATIC_BLURB));
}

static void
mwb_download_open_cb (NbtkWidget *button, MwbDownload *self)
{
  MwbDownloadPrivate *priv = self->priv;

  if (priv->file)
    {
      gchar *uri;
      GError *error = NULL;
      GdkAppLaunchContext *context = gdk_app_launch_context_new ();

      /* TODO: Set icon, screen, etc. */

      /* Special timestamp to override focus-stealing prevention code */
      gdk_app_launch_context_set_timestamp (context, 0x7fffffff);

      uri = g_file_get_uri (priv->file);
      if (!g_app_info_launch_default_for_uri (uri,
                                              G_APP_LAUNCH_CONTEXT (context),
                                              &error))
        {
          /* Try to open the folder if we can't open the file */
          gchar *path_end = g_strrstr (uri, G_DIR_SEPARATOR_S);
          if (path_end)
            path_end[1] = '\0';

          g_clear_error (&error);

          if (!g_app_info_launch_default_for_uri (path_end ? uri : "/",
                                                  G_APP_LAUNCH_CONTEXT (context),
                                                  &error))
            {
              g_warning ("Error launching URI: %s", error->message);
              g_error_free (error);
            }
        }
      g_free (uri);
      g_object_unref (context);
    }
}

static void
mwb_download_cancel_cb (NbtkWidget *button, MwbDownload *self)
{
  MwbDownloadPrivate *priv = self->priv;

  if (priv->download)
    clutter_mozembed_download_cancel (priv->download);
}

static void
mwb_download_style_changed_cb (NbtkStylable *stylable, MwbDownload *download)
{
  ClutterColor *color;
  MwbDownloadPrivate *priv = download->priv;

  nbtk_stylable_get (stylable,
                     "color", &color,
                     NULL);

  if (color)
    {
      clutter_rectangle_set_color (CLUTTER_RECTANGLE (priv->separator), color);
      clutter_color_free (color);
    }
}

static void
mwb_download_label_style_changed_cb (NbtkStylable *stylable,
                                     MwbDownload  *download)
{
  gint font_size;
  gchar *font_family;
  MwbDownloadPrivate *priv = download->priv;

  nbtk_stylable_get (stylable,
                     "font-family", &font_family,
                     "font-size", &font_size,
                     NULL);

  if (font_family && font_size)
    {
      ClutterUnits units = { 0, };
      gchar *font_name = g_strdup_printf ("%s %d", font_family, font_size);

      clutter_units_em_for_font (&units, font_name, 20);
      clutter_actor_set_width (CLUTTER_ACTOR (priv->progress_label),
                               clutter_units_to_pixels (&units));
      g_free (font_name);
    }

  g_free (font_family);
}

static void
mwb_download_init (MwbDownload *self)
{
  MwbDownloadPrivate *priv = self->priv = DOWNLOAD_PRIVATE (self);

  nbtk_table_set_row_spacing (NBTK_TABLE (self), 4);
  nbtk_table_set_col_spacing (NBTK_TABLE (self), 4);

  priv->icon = clutter_texture_new ();
  priv->file_label = nbtk_label_new ("");
  priv->open_button = nbtk_button_new_with_label (_("Open file"));
  priv->progress_label = nbtk_label_new ("");
  g_object_set (G_OBJECT (nbtk_label_get_clutter_text (
                          NBTK_LABEL (priv->progress_label))),
                "ellipsize", PANGO_ELLIPSIZE_END,
                NULL);
  priv->progress_bar = nbtk_progress_bar_new ();
  priv->cancel_button = nbtk_button_new ();
  clutter_actor_set_name (CLUTTER_ACTOR (priv->cancel_button), "cancel");
  priv->separator = clutter_rectangle_new ();
  clutter_actor_set_height (priv->separator, 1);

  g_signal_connect (priv->open_button, "clicked",
                    G_CALLBACK (mwb_download_open_cb), self);
  g_signal_connect (priv->cancel_button, "clicked",
                    G_CALLBACK (mwb_download_cancel_cb), self);
  g_signal_connect (self, "style-changed",
                    G_CALLBACK (mwb_download_style_changed_cb), self);
  g_signal_connect (priv->progress_label, "style-changed",
                    G_CALLBACK (mwb_download_label_style_changed_cb), self);

  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        priv->icon,
                                        0, 0,
                                        "row-span", 2,
                                        "x-fill", TRUE,
                                        "y-fill", TRUE,
                                        "x-expand", FALSE,
                                        "y-expand", FALSE,
                                        "x-align", 0.5,
                                        "y-align", 0.5,
                                        NULL);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        CLUTTER_ACTOR (priv->file_label),
                                        0, 1,
                                        "x-align", 0.0,
                                        "y-align", 0.5,
                                        "y-fill", FALSE,
                                        NULL);
  mwb_utils_table_add (NBTK_TABLE (self),
                       CLUTTER_ACTOR (priv->open_button),
                       0, 2, FALSE, FALSE, FALSE, FALSE);
  clutter_actor_hide (CLUTTER_ACTOR (priv->open_button));
  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        CLUTTER_ACTOR (priv->progress_bar),
                                        1, 1,
                                        "col-span", 2,
                                        "x-expand", FALSE,
                                        "x-fill", TRUE,
                                        "y-expand", FALSE,
                                        "y-fill", TRUE,
                                        NULL);
  mwb_utils_table_add (NBTK_TABLE (self),
                       CLUTTER_ACTOR (priv->progress_label),
                       1, 3, FALSE, FALSE, FALSE, FALSE);
  mwb_utils_table_add (NBTK_TABLE (self),
                       CLUTTER_ACTOR (priv->cancel_button),
                       1, 4, FALSE, FALSE, FALSE, FALSE);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        priv->separator,
                                        2, 0,
                                        "col-span", 5,
                                        "x-expand", FALSE,
                                        "x-fill", TRUE,
                                        "y-expand", FALSE,
                                        "y-fill", FALSE,
                                        NULL);
}

