/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: stream.c,v 1.67 2004/06/03 08:00:58 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/stream.h>
#include <zorp/log.h>
#include <zorp/error.h>

#include <string.h>
#include <sys/types.h>

/* source functions delegating prepare/check/dispatch/finalize to the stream
 * methods. */

static gboolean
z_stream_source_prepare(GSource *s, gint *timeout)
{
  ZStreamSource *self = (ZStreamSource *) s;
  gboolean ret = FALSE;
  
  z_enter();
  ret = z_stream_watch_prepare(self->stream, s, timeout);
  z_leave();
  return ret;
}

static gboolean
z_stream_source_check(GSource *s)
{
  ZStreamSource *self = (ZStreamSource *) s;
  gboolean ret = FALSE;

  z_enter();
  ret = z_stream_watch_check(self->stream, s);
  z_leave();
  return ret;
}

static gboolean
z_stream_source_dispatch(GSource     *s,
                         GSourceFunc callback G_GNUC_UNUSED,
                         gpointer    user_data G_GNUC_UNUSED)
{
  ZStreamSource *self = (ZStreamSource *) s;
  gboolean ret = FALSE;
      
  z_enter();
  ret = z_stream_watch_dispatch(self->stream, s);
  z_leave();
  return ret;
}

static void
z_stream_source_finalize(GSource *s)
{
  ZStreamSource *self = (ZStreamSource *) s;
  ZStream *stream;
  
  z_enter();

  z_stream_watch_finalize(self->stream, s);

  if (self->stream->user_data_read && self->stream->user_data_read_notify)
    self->stream->user_data_read_notify(self->stream->user_data_read);
  if (self->stream->user_data_write && self->stream->user_data_write_notify)
    self->stream->user_data_write_notify(self->stream->user_data_write);
  if (self->stream->user_data_pri && self->stream->user_data_pri_notify)
    self->stream->user_data_pri_notify(self->stream->user_data_pri);

  stream = self->stream;
  self->stream = NULL;
  z_stream_unref(stream);
  
  z_leave();
}

static GSourceFuncs 
z_stream_source_funcs = 
{
  z_stream_source_prepare,
  z_stream_source_check,   
  z_stream_source_dispatch,
  z_stream_source_finalize,
  NULL,
  NULL
};

GSource *
z_stream_source_new(ZStream *stream)
{
  ZStreamSource *self = (ZStreamSource *) g_source_new(&z_stream_source_funcs, sizeof(ZStreamSource));
  
  z_enter();
  z_stream_ref(stream);
  self->stream = stream;
  
  z_leave();
  return &self->super;
}

/* ZStreamContext */

void
z_stream_context_destroy(ZStreamContext *self)
{
  z_enter();
  if (!self->restored)
    {
      if (self->user_data_read && self->user_data_read_notify)
        self->user_data_read_notify(self->user_data_read);
      if (self->user_data_write && self->user_data_write_notify)
        self->user_data_write_notify(self->user_data_write);
      if (self->user_data_pri && self->user_data_pri_notify)
        self->user_data_pri_notify(self->user_data_pri);
      g_free(self->stream_extra);
      self->stream_extra = NULL;
      self->restored = TRUE;
    }
  z_leave();
}

/* general functions to be used as ZStream virtual functions */

void
z_stream_free_method(ZObject *s)
{
  ZStream *self = Z_CAST(s, ZStream);
  
  z_enter();
  z_stream_unref(self->child);
  self->child = NULL;
  z_object_free_method(s);
  z_leave();
}

gboolean
z_stream_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen)
{
  gboolean res = FALSE;
  
  z_enter();
  switch (ZST_CTRL_MSG(function))
    {
    case ZST_CTRL_GET_COND_READ:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = s->want_read;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_COND_READ:
      if (vlen == sizeof(gboolean))
        {
          s->want_read =  *((gboolean *)value);
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_COND_WRITE:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = s->want_write;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_COND_WRITE:
      if (vlen == sizeof(gboolean))
        {
          s->want_write = *((gboolean *)value);
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_COND_PRI:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = s->want_pri;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_COND_PRI:
      if (vlen == sizeof(gboolean))
        {
          s->want_pri = *((gboolean *)value);
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_CALLBACK_READ:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          cbv->cb = s->read_cb;
          cbv->cb_data = s->user_data_read;
          cbv->cb_notify = s->user_data_read_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_READ:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          s->read_cb = cbv->cb;
          s->user_data_read = cbv->cb_data;
          s->user_data_read_notify = cbv->cb_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_CALLBACK_WRITE:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          cbv->cb = s->write_cb;
          cbv->cb_data = s->user_data_write;
          cbv->cb_notify = s->user_data_write_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_WRITE:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          s->write_cb = cbv->cb;
          s->user_data_write = cbv->cb_data;
          s->user_data_write_notify = cbv->cb_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_GET_CALLBACK_PRI:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          cbv->cb = s->pri_cb;
          cbv->cb_data = s->user_data_pri;
          cbv->cb_notify = s->user_data_pri_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_PRI:
      if (vlen == sizeof(ZStreamSetCb))
        {
          ZStreamSetCb *cbv = (ZStreamSetCb *)value;
          s->pri_cb = cbv->cb;
          s->user_data_pri = cbv->cb_data;
          s->user_data_pri_notify = cbv->cb_notify;
          res = TRUE;
        }
      break;
    case ZST_CTRL_SET_TIMEOUT_BLOCK:
      if (vlen == sizeof(gint))
        {
          s->timeout = *((gint *)value);
          res = TRUE;
        }
      break;
    default:
      if (s->child)
        {
          res = z_stream_ctrl(s->child, function, value, vlen);
          z_leave();
          return res;
        }
      break;
    }

  if (res && (function & ZST_CTRL_MSG_FORWARD) && s->child)
    res = z_stream_ctrl(s->child, function, value, vlen);
  
  z_leave();
  return res;
}

static gsize
z_stream_extra_get_size_method(ZStream *s)
{
  if (s->child)
    return z_stream_extra_get_size(s->child);
  return 0;
}

static gsize
z_stream_extra_save_method(ZStream *s, gpointer extra)
{
  if (s->child)
    return z_stream_extra_save(s->child, extra);
  return 0;
}

static gsize
z_stream_extra_restore_method(ZStream *s, gpointer extra)
{
  if (s->child)
    return z_stream_extra_restore(s->child, extra);
  return 0;
}

/**
 * z_stream_save_context:
 * @self: ZStream instance
 * @context: save stream context here
 * 
 * This function can be used to save the complete ZStream callback state. It
 * is usually needed when this stream is to be used in a completely
 * different context (e.g. ZTransfer vs. ZProxy). The function saves the
 * references to user_data associated with different callbacks, e.g. 
 * GDestroyNotify callbacks are called when the context is freed without
 * restoring it. The function also NULLs all fields in ZStream to make it
 * sure the ZStream will not do the same.
 *
 * Returns: always returns TRUE
 */
gboolean
z_stream_save_context(ZStream *self, ZStreamContext *context)
{
  gsize extra_size;
  
  z_enter();
  context->restored = FALSE;
  context->want_read = self->want_read;
  context->user_data_read = self->user_data_read;
  context->user_data_read_notify = self->user_data_read_notify;
  context->read_cb = self->read_cb;
         
  context->want_pri = self->want_pri;
  context->user_data_pri = self->user_data_pri;
  context->user_data_pri_notify = self->user_data_pri_notify;
  context->pri_cb = self->pri_cb;
  
  context->want_write = self->want_write;
  context->user_data_write = self->user_data_write;
  context->user_data_write_notify = self->user_data_write_notify;
  context->write_cb = self->write_cb;
  
  context->timeout = self->timeout;

  self->want_read = FALSE;
  self->user_data_read = NULL;
  self->user_data_read_notify = NULL;
  self->read_cb = NULL;
         
  self->want_pri = FALSE;
  self->user_data_pri = NULL;
  self->user_data_pri_notify = NULL;
  self->pri_cb = NULL;
  
  self->want_write = FALSE;
  self->user_data_write = NULL;
  self->user_data_write_notify = NULL;
  self->write_cb = NULL;
  
  extra_size = z_stream_extra_get_size(self);
  context->stream_extra = g_malloc0(extra_size);
  z_stream_extra_save(self, context->stream_extra);

  z_leave();
  return TRUE;
}

/**
 * z_stream_restore_context:
 * @self: ZStream instance
 * @context: stream context previously stored by z_stream_save_context
 *
 * This function restores the stream callback context previously saved by
 * z_stream_save_context.
 *
 * Returns: always returns TRUE
 */
gboolean
z_stream_restore_context(ZStream *self, ZStreamContext *context)
{
  z_enter();
  
  g_return_val_if_fail(!context->restored, FALSE);
  
  self->want_read = context->want_read;
  self->user_data_read = context->user_data_read;
  self->user_data_read_notify = context->user_data_read_notify;
  self->read_cb = context->read_cb;
         
  self->want_pri = context->want_pri;
  self->user_data_pri = context->user_data_pri;
  self->user_data_pri_notify = context->user_data_pri_notify;
  self->pri_cb = context->pri_cb;
  
  self->want_write = context->want_write;
  self->user_data_write = context->user_data_write;
  self->user_data_write_notify = context->user_data_write_notify;
  self->write_cb = context->write_cb;
  
  self->timeout = context->timeout;
  
  if (context->stream_extra)
    {
      z_stream_extra_restore(self, context->stream_extra);
      g_free(context->stream_extra);
      context->stream_extra = NULL;
    }

  context->restored = TRUE;
  z_leave();
  return TRUE;
}

ZStreamFuncs z_stream_funcs =
{
  {
    Z_FUNCS_COUNT(ZStream),
    z_stream_free_method,
  },
  NULL,   		/* read */
  NULL,   		/* write */
  NULL,   		/* read_pri */
  NULL,   		/* write_pri */
  NULL,   		/* shutdown */
  NULL,   		/* close */
  z_stream_ctrl_method, /* ctrl */
  NULL,                 /* attach_source */
  NULL,                 /* detach_source */
  NULL,			/* watch_prepare */
  NULL,			/* watch_check */
  NULL,			/* watch_dispatch */
  NULL,			/* watch_finalize */
  z_stream_extra_get_size_method,
  z_stream_extra_save_method,
  z_stream_extra_restore_method
};

#ifdef G_OS_WIN32
  LIBZORPLL_DLLIMPEX
#endif
ZClass ZStream__class = 
{
  Z_CLASS_HEADER,
  &ZObject__class,
  "ZStream",
  sizeof(ZStream),
  &z_stream_funcs.super,
};

ZStream *
z_stream_new(ZClass *class, gchar *name, ZStream *child, gint umbrella_flags)
{
  ZStream *self;

  z_enter();
  self = Z_NEW_COMPAT(class, ZStream);
  g_strlcpy(self->name, name, sizeof(self->name));
  self->timeout = -2;
  self->time_open = time(NULL);
  self->child = z_stream_ref(child);
  self->umbrella_flags = umbrella_flags;
  z_leave();
  return self;
}

ZStream *
z_stream_search_stack(ZStream *top, gint direction, ZClass *class)
{
  ZStream *p;

  z_enter();
  for (p = top; p; p = p->child)
    {
      if (z_object_is_instance(&p->super, class))
        return p;
        
      if ((p->umbrella_flags & direction) == direction)
        {
          /* this direction is shadowed by the current entry */
          break;
        }
    }
  z_leave();
  return NULL;
}
