/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <gio/gdesktopappinfo.h>

#include "zeitgeist-event.h"
#include "zeitgeist-ontology-interpretations.h"
#include "zeitgeist-ontology-manifestations.h"


typedef struct
{
  
} Fixture;

static void setup    (Fixture *fix, gconstpointer data);
static void teardown (Fixture *fix, gconstpointer data);

static const gchar *old_xdg_data_dirs = NULL;

static void
setup (Fixture *fix, gconstpointer data)
{
  if (old_xdg_data_dirs != NULL)
    old_xdg_data_dirs = g_getenv ("XDG_DATA_DIRS");
  g_setenv ("XDG_DATA_DIRS", TEST_DIR, TRUE);
}

static void
teardown (Fixture *fix, gconstpointer data)
{
  g_setenv ("XDG_DATA_DIRS", old_xdg_data_dirs, TRUE);  
}

static void
test_create_empty (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent *ev;

  ev = zeitgeist_event_new ();

  g_assert_cmpint (0, ==, zeitgeist_event_get_id (ev));
  g_assert_cmpint (0, ==, zeitgeist_event_get_timestamp (ev));
  g_assert_cmpstr (NULL, ==, zeitgeist_event_get_interpretation (ev));
  g_assert_cmpstr (NULL, ==, zeitgeist_event_get_manifestation (ev));
  g_assert_cmpstr (NULL, ==, zeitgeist_event_get_actor (ev));
  g_assert_cmpint (0, ==, zeitgeist_event_num_subjects (ev));
  g_assert (zeitgeist_event_get_payload (ev) == NULL);

  g_object_unref (ev);
}

static void
test_create_full (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent *ev;
  ZeitgeistSubject *su;

  ev = zeitgeist_event_new_full (
                   ZEITGEIST_ZG_ACCESS_EVENT,
                   ZEITGEIST_ZG_USER_ACTIVITY,
                   "app://firefox.desktop",
                   zeitgeist_subject_new_full ("http://example.com",
                                               ZEITGEIST_NFO_WEBSITE,
                                               ZEITGEIST_NFO_REMOTE_DATA_OBJECT,
                                               "text/html",
                                               "http://example.com",
                                               "example.com",
                                               "net"),
                   NULL);

  g_assert_cmpint (0, ==, zeitgeist_event_get_id (ev));
  g_assert_cmpint (0, ==, zeitgeist_event_get_timestamp (ev));
  g_assert_cmpstr (ZEITGEIST_ZG_ACCESS_EVENT,==, zeitgeist_event_get_interpretation (ev));
  g_assert_cmpstr (ZEITGEIST_ZG_USER_ACTIVITY, ==, zeitgeist_event_get_manifestation (ev));
  g_assert_cmpstr ("app://firefox.desktop", ==, zeitgeist_event_get_actor (ev));
  g_assert_cmpint (1, ==, zeitgeist_event_num_subjects (ev));
  g_assert (zeitgeist_event_get_payload (ev) == NULL);

  su = zeitgeist_event_get_subject (ev, 0);
  g_assert_cmpstr ("http://example.com", ==, zeitgeist_subject_get_uri(su));
  g_assert_cmpstr (ZEITGEIST_NFO_WEBSITE, ==, zeitgeist_subject_get_interpretation (su));
  g_assert_cmpstr (ZEITGEIST_NFO_REMOTE_DATA_OBJECT, ==, zeitgeist_subject_get_manifestation (su));
  g_assert_cmpstr ("text/html", ==, zeitgeist_subject_get_mimetype (su));
  g_assert_cmpstr ("http://example.com", ==, zeitgeist_subject_get_origin (su));
  g_assert_cmpstr ("example.com", ==, zeitgeist_subject_get_text (su));
  g_assert_cmpstr ("net", ==, zeitgeist_subject_get_storage (su));
                   
  g_object_unref (ev);
}

static void
test_actor_from_app_info (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent *ev;
  GAppInfo       *appinfo;

  appinfo = G_APP_INFO (g_desktop_app_info_new_from_filename (TEST_DIR"/test.desktop"));
  g_assert (G_IS_APP_INFO (appinfo));
  
  ev = zeitgeist_event_new ();
  zeitgeist_event_set_actor_from_app_info (ev, appinfo);

  g_assert_cmpstr ("application://test.desktop", ==, zeitgeist_event_get_actor (ev));
}

int
main (int   argc,
      char *argv[])
{
  g_type_init ();
  g_test_init (&argc, &argv, NULL);
  
  g_test_add ("/Zeitgeist/Event/CreateEmpty", Fixture, NULL,
              setup, test_create_empty, teardown);
  g_test_add ("/Zeitgeist/Event/CreateFull", Fixture, NULL,
              setup, test_create_full, teardown);
  g_test_add ("/Zeitgeist/Event/ActorFromAppInfo", Fixture, NULL,
              setup, test_actor_from_app_info, teardown);              
  
  return g_test_run();
}
