/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by: Michal Hruby <michal.mhr@gmail.com>
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "zeitgeist-data-source-registry.h"
#include "zeitgeist-eggdbusconversions.h"
#include "zeitgeist-marshal.h"
#include "eggzeitgeistbindings.h"

/**
 * SECTION:zeitgeist-data-source-registry
 * @short_description: Query the Zeitgeist Data Source Registry extension
 * @include: zeitgeist.h
 *
 * The Zeitgeist engine maintains a publicly available list of recognized
 * data-sources (components inserting information into Zeitgeist).
 * #ZeitgeistDataSourceRegistry is used to register new data sources,
 * get information about them and gives the ability to enable or disable
 * the data sources.
 */

G_DEFINE_TYPE (ZeitgeistDataSourceRegistry,
               zeitgeist_data_source_registry,
               G_TYPE_OBJECT);
#define ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, ZEITGEIST_TYPE_DATA_SOURCE_REGISTRY, \
                               ZeitgeistDataSourceRegistryPrivate))

typedef struct
{
  /* The connection to the ZG daemon
   * Note: The EggZeitgeistDataSourceRegistry is owned by 
   * the EggDBusObjectProxy!
   */
  EggDBusObjectProxy *registry_proxy;
  EggZeitgeistDataSourceRegistry *registry;

  gulong src_registered_id;
  gulong src_disconnected_id;
  gulong src_enabled_id;

} ZeitgeistDataSourceRegistryPrivate;

/* Property ids */
enum
{
	PROP_0,

	LAST_PROPERTY
};

enum
{
  SOURCE_REGISTERED,
  SOURCE_ENABLED,
  SOURCE_DISCONNECTED,

  LAST_SIGNAL
};
static guint _registry_signals[LAST_SIGNAL] = { 0 };

static void
zeitgeist_data_source_registry_init (ZeitgeistDataSourceRegistry *self)
{
  ZeitgeistDataSourceRegistryPrivate *priv;
  EggDBusConnection   *conn;

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (self);
  
  /* Set up the connection to the ZG daemon */  
  conn = egg_dbus_connection_get_for_bus (EGG_DBUS_BUS_TYPE_SESSION);
  priv->registry_proxy = 
    egg_dbus_connection_get_object_proxy (conn,
                                          "org.gnome.zeitgeist.Engine",
                                          "/org/gnome/zeitgeist/data_source_registry");
  
  priv->registry =
    EGG_ZEITGEIST_QUERY_INTERFACE_DATA_SOURCE_REGISTRY (priv->registry_proxy);
  g_object_unref (conn);
}

static void
zeitgeist_data_source_registry_finalize (GObject *object)
{
  ZeitgeistDataSourceRegistry *registry;
  ZeitgeistDataSourceRegistryPrivate *priv;
  
  registry = ZEITGEIST_DATA_SOURCE_REGISTRY (object);
  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (registry);

  if (priv->src_registered_id)
    {
      g_signal_handler_disconnect (priv->registry, priv->src_registered_id);
    }

  if (priv->src_disconnected_id)
    {
      g_signal_handler_disconnect (priv->registry, priv->src_disconnected_id);
    }

  if (priv->src_enabled_id)
    {
      g_signal_handler_disconnect (priv->registry, priv->src_enabled_id);
    }

  /* Note: priv->registry is owned by priv->registry_proxy */
  if (priv->registry_proxy)
    {
      g_object_unref (priv->registry_proxy);
    }
  
  G_OBJECT_CLASS (zeitgeist_data_source_registry_parent_class)->finalize (object);
}

static void
zeitgeist_data_source_registry_get_property (GObject    *object,
                                             guint       prop_id,
                                             GValue     *value,
                                             GParamSpec *pspec)
{
  ZeitgeistDataSourceRegistryPrivate *priv;

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (object);

  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        return;
        break;
    }
}

static void
zeitgeist_data_source_registry_set_property (GObject      *object,
                                             guint         prop_id,
                                             const GValue *value,
                                             GParamSpec   *pspec)
{
  ZeitgeistDataSourceRegistryPrivate *priv;

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (object);

  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        return;
        break;
    }
}

static void
zeitgeist_data_source_registry_src_registered (
    ZeitgeistDataSourceRegistry *registry,
    EggZeitgeistDataSource *source)
{
  ZeitgeistDataSource *data_source;

  data_source = _egg_zeitgeist_data_source_to_zeitgeist_data_source (source);

  g_signal_emit (registry, _registry_signals[SOURCE_REGISTERED],
                 0, data_source);

  g_object_unref (data_source);
}

static void
zeitgeist_data_source_registry_src_disconnected (
    ZeitgeistDataSourceRegistry *registry,
    EggZeitgeistDataSource *source)
{
  ZeitgeistDataSource *data_source;

  data_source = _egg_zeitgeist_data_source_to_zeitgeist_data_source (source);

  g_signal_emit (registry, _registry_signals[SOURCE_DISCONNECTED],
                 0, data_source);

  g_object_unref (data_source);
}

static void
zeitgeist_data_source_registry_src_enabled (
    ZeitgeistDataSourceRegistry *registry,
    gchar *unique_id,
    gboolean enabled)
{
  g_signal_emit (registry, _registry_signals[SOURCE_ENABLED],
                 0, unique_id, enabled);
}

static void
zeitgeist_data_source_registry_constructed (GObject *object)
{
  ZeitgeistDataSourceRegistryPrivate *priv;

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (object);

  priv->src_registered_id = g_signal_connect_swapped (
      priv->registry, "data-source-registered",
      G_CALLBACK (zeitgeist_data_source_registry_src_registered),
      object);
  priv->src_disconnected_id = g_signal_connect_swapped (
      priv->registry, "data-source-disconnected",
      G_CALLBACK (zeitgeist_data_source_registry_src_disconnected),
      object);
  priv->src_enabled_id = g_signal_connect_swapped (
      priv->registry, "data-source-enabled",
      G_CALLBACK (zeitgeist_data_source_registry_src_enabled),
      object);
}

static void
zeitgeist_data_source_registry_class_init (ZeitgeistDataSourceRegistryClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;
  
  object_class->finalize     = zeitgeist_data_source_registry_finalize;
  object_class->get_property = zeitgeist_data_source_registry_get_property;
  object_class->set_property = zeitgeist_data_source_registry_set_property;
  object_class->constructed  = zeitgeist_data_source_registry_constructed;

  _registry_signals[SOURCE_REGISTERED] =
    g_signal_new ("source-registered",
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (ZeitgeistDataSourceRegistryClass, source_registered),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__OBJECT,
                  G_TYPE_NONE, 1, ZEITGEIST_TYPE_DATA_SOURCE);

  _registry_signals[SOURCE_DISCONNECTED] =
    g_signal_new ("source-disconnected",
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (ZeitgeistDataSourceRegistryClass, source_disconnected),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__OBJECT,
                  G_TYPE_NONE, 1, ZEITGEIST_TYPE_DATA_SOURCE);

  _registry_signals[SOURCE_ENABLED] =
    g_signal_new ("source-enabled",
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (ZeitgeistDataSourceRegistryClass, source_enabled),
                  NULL, NULL,
                  _zeitgeist_cclosure_marshal_VOID__STRING_BOOLEAN,
                  G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_BOOLEAN);

  g_type_class_add_private (object_class,
                            sizeof (ZeitgeistDataSourceRegistryPrivate));
}

/* Used to marshal the async callbacks from EggDBus into ones
 * coming from this DataSourceRegistry instance */
static void
dispatch_async_callback (GObject      *source_object,
                         GAsyncResult *res,
                         gpointer      user_data)
{ 
  gpointer                    *data = (gpointer*) user_data;
  ZeitgeistDataSourceRegistry *self = ZEITGEIST_DATA_SOURCE_REGISTRY (data[0]);
  GAsyncReadyCallback          callback = (GAsyncReadyCallback) data[1];
  gpointer                     _user_data = data[2];

  if (callback != NULL)
    {
      callback (G_OBJECT (self), res, _user_data);
    }
  
  g_object_unref (self);
  g_free (user_data);
}

/*
 * API BELOW HERE
 */

/**
 * zeitgeist_data_source_registry_new:
 *
 * Create a new data source registry instance.
 *
 * DataSourceRegistry instances are not overly expensive for neither 
 * client or the Zeitgeist daemon so there's no need to go to lengths
 * to keep singleton instances around.
 *
 * Returns: A reference to a newly allocated registry.
 */
ZeitgeistDataSourceRegistry*
zeitgeist_data_source_registry_new (void)
{
  ZeitgeistDataSourceRegistry *registry;

  registry = (ZeitgeistDataSourceRegistry*)
    g_object_new (ZEITGEIST_TYPE_DATA_SOURCE_REGISTRY, NULL);

  return registry;
}

void
zeitgeist_data_source_registry_get_data_sources (
                                     ZeitgeistDataSourceRegistry *self,
                                     GCancellable                *cancellable,
                                     GAsyncReadyCallback          callback,
                                     gpointer                     user_data)
{
  ZeitgeistDataSourceRegistryPrivate *priv;
  gpointer                           *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE_REGISTRY (self));
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE (cancellable));

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (self);

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  egg_zeitgeist_data_source_registry_get_data_sources (
      priv->registry,
      EGG_DBUS_CALL_FLAGS_NONE,
      cancellable,
      dispatch_async_callback,
      dispatch_data);
}

/**
 * zeitgeist_data_source_registry_get_data_sources_finish:
 * @self: Instance of #ZeitgeistDataSourceRegistry.
 * @res: a #GAsyncResult.
 * @error: a #GError or #NULL.
 *
 * Returns: Newly created #GPtrArray containing #ZeitgeistDataSource<!-- -->(s)
 *          registered in Zeitgeist. Free using g_ptr_array_unref() once
 *          you're done using it.
 */
GPtrArray*
zeitgeist_data_source_registry_get_data_sources_finish (
                                     ZeitgeistDataSourceRegistry *self,
                                     GAsyncResult                *res,
                                     GError                      **error)
{
  ZeitgeistDataSourceRegistryPrivate *priv;
  EggDBusArraySeq                    *array_seq;
  gboolean                            call_result;
  GPtrArray                          *data_sources = NULL;

  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE_REGISTRY (self), NULL);

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (self);

  call_result = egg_zeitgeist_data_source_registry_get_data_sources_finish (
      priv->registry,
      &array_seq,
      res,
      error);

  data_sources =
    _egg_zeitgeist_data_sources_to_zeitgeist_data_sources (array_seq);
  g_ptr_array_set_free_func (data_sources, g_object_unref);

  g_object_unref (array_seq);

  return data_sources;
}

/**
 * zeitgeist_data_source_registry_register_data_source:
 * @self: Instance of #ZeitgeistDataSourceRegistry.
 * @source: Data source to register.
 * @cancellable: a #GCancellable or #NULL.
 * @callback: a GAsyncReadyCallback to call when the request is finished.
 * @user_data: the data to pass to callback function.
 *
 * Registers new data source in the registry, the @source parameter needs to
 * have unique-id, name, description and optionally event_templates set,
 * therefore it is useful to pass #ZeitgeistDataSource instance created using
 * zeitgeist_data_source_new_full(). The registry will assume its ownership.
 */
void
zeitgeist_data_source_registry_register_data_source (
    ZeitgeistDataSourceRegistry *self,
    ZeitgeistDataSource         *source,
    GCancellable                *cancellable,
    GAsyncReadyCallback          callback,
    gpointer                     user_data)
{
  ZeitgeistDataSourceRegistryPrivate *priv;
  gpointer                           *dispatch_data;
  EggDBusArraySeq                    *templates;

  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE_REGISTRY (self));
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (source));
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE (cancellable));

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (self);

  g_object_ref_sink (G_OBJECT (source));
  
  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  templates = _zeitgeist_events_to_egg_zeitgeist_events (
      zeitgeist_data_source_get_event_templates (source)
  );

  egg_zeitgeist_data_source_registry_register_data_source (
      priv->registry,
      EGG_DBUS_CALL_FLAGS_NONE,
      zeitgeist_data_source_get_unique_id (source),
      zeitgeist_data_source_get_name (source),
      zeitgeist_data_source_get_description (source),
      templates,
      cancellable,
      dispatch_async_callback,
      dispatch_data);

  g_object_unref (templates);
  g_object_unref (source);
}

/**
 * zeitgeist_data_source_registry_register_data_source_finish:
 * @self: Instance of #ZeitgeistDataSourceRegistry.
 * @res: Result of the asynchronous operation.
 * @error: a #GError or NULL.
 *
 * Returns: If error is unset, returns whether this data source is enabled.
 */
gboolean
zeitgeist_data_source_registry_register_data_source_finish (
    ZeitgeistDataSourceRegistry *self,
    GAsyncResult                *res,
    GError                     **error)
{
  ZeitgeistDataSourceRegistryPrivate *priv;
  gboolean                            result = FALSE;

  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE_REGISTRY (self), FALSE);

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (self);

  egg_zeitgeist_data_source_registry_register_data_source_finish (
      priv->registry,
      &result,
      res,
      error);

  return result;
}

void
zeitgeist_data_source_registry_set_data_source_enabled (
    ZeitgeistDataSourceRegistry *self,
    const gchar                 *unique_id,
    gboolean                     enabled,
    GCancellable                *cancellable,
    GAsyncReadyCallback          callback,
    gpointer                     user_data)
{
  ZeitgeistDataSourceRegistryPrivate *priv;
  gpointer                           *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE_REGISTRY (self));
  g_return_if_fail (unique_id != NULL);
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE (cancellable));

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (self);

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  egg_zeitgeist_data_source_registry_set_data_source_enabled (
      priv->registry,
      EGG_DBUS_CALL_FLAGS_NONE,
      unique_id,
      enabled,
      cancellable,
      dispatch_async_callback,
      dispatch_data);
}

void
zeitgeist_data_source_registry_set_data_source_enabled_finish (
    ZeitgeistDataSourceRegistry *self,
    GAsyncResult                *res,
    GError                     **error)
{
  ZeitgeistDataSourceRegistryPrivate *priv;

  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE_REGISTRY (self));

  priv = ZEITGEIST_DATA_SOURCE_REGISTRY_GET_PRIVATE (self);

  egg_zeitgeist_data_source_registry_set_data_source_enabled_finish (
      priv->registry,
      res,
      error);
}

