/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * Authors:
 *  Florian Boucault <florian.boucault@canonical.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QHash>
#include <QByteArray>

#include <dee.h>
#include <glib-object.h>

#include "deelistmodel.h"


static QVariant
QVariantFromGValue(GValue* gvalue)
{
    switch(G_TYPE_FUNDAMENTAL(G_VALUE_TYPE(gvalue))) {
        case G_TYPE_CHAR:
            return QVariant::fromValue<char>(g_value_get_char(gvalue));
        case G_TYPE_UCHAR:
            return QVariant::fromValue<uchar>(g_value_get_uchar(gvalue));
        case G_TYPE_BOOLEAN:
            return QVariant::fromValue<bool>(g_value_get_boolean(gvalue));
        case G_TYPE_INT:
            return QVariant::fromValue<int>(g_value_get_int(gvalue));
        case G_TYPE_UINT:
            return QVariant::fromValue<uint>(g_value_get_uint(gvalue));
        case G_TYPE_LONG:
            return QVariant::fromValue<long>(g_value_get_long(gvalue));
        case G_TYPE_ULONG:
            return QVariant::fromValue<unsigned long>(g_value_get_ulong(gvalue));
        case G_TYPE_INT64:
            return QVariant::fromValue<qint64>(g_value_get_int64(gvalue));
        case G_TYPE_UINT64:
            return QVariant::fromValue<quint64>(g_value_get_uint64(gvalue));
        case G_TYPE_ENUM:
            return QVariant::fromValue<int>(g_value_get_enum(gvalue));
        case G_TYPE_FLAGS:
            return QVariant::fromValue<uint>(g_value_get_flags(gvalue));
        case G_TYPE_FLOAT:
            return QVariant::fromValue<float>(g_value_get_float(gvalue));
        case G_TYPE_DOUBLE:
            return QVariant::fromValue<double>(g_value_get_double(gvalue));
        case G_TYPE_STRING:
            return QVariant(QString::fromUtf8(g_value_get_string(gvalue)));
        case G_TYPE_POINTER:
            return QVariant::fromValue<gpointer>(g_value_get_pointer(gvalue));
        case G_TYPE_INVALID:
        case G_TYPE_NONE:
        default:
            /* Fallback on an empty QVariant.
               Missing conversion of following GTypes:
                - G_TYPE_INTERFACE
                - G_TYPE_OBJECT
                - G_TYPE_GTYPE
                - G_TYPE_VARIANT
                - G_TYPE_BOXED
                - G_TYPE_PARAM
            */
            return QVariant();
    }
}

class DeeListModelPrivate {
public:
    DeeListModelPrivate(DeeListModel* parent);
    ~DeeListModelPrivate();

    void connectToDeeModel();
    void disconnectFromDeeModel();
    void createRoles();

    /* GObject signal handlers for m_deeModel */
    static void onReady(GObject* emitter, DeeListModel* model);
    static void onRowAdded(GObject* emitter, DeeModelIter* iter, DeeListModel* model);
    static void onRowRemoved(GObject* emitter, DeeModelIter* iter, DeeListModel* model);
    static void onRowChanged(GObject* emitter, DeeModelIter* iter, DeeListModel* model);

    DeeListModel* m_parent;
    DeeModel* m_deeModel;
    QString m_name;
    bool m_ready;
};

DeeListModelPrivate::DeeListModelPrivate(DeeListModel* parent) : m_deeModel(NULL), m_ready(false), m_parent(parent)
{
}

DeeListModelPrivate::~DeeListModelPrivate()
{
    disconnectFromDeeModel();
}

void
DeeListModelPrivate::disconnectFromDeeModel()
{
    if (m_deeModel != NULL) {
        /* Disconnect from all GObject properties */
        g_object_disconnect(m_deeModel, "any_signal", G_CALLBACK(onReady), m_parent, NULL);
        g_object_disconnect(m_deeModel, "any_signal", G_CALLBACK(onRowAdded), m_parent, NULL);
        g_object_disconnect(m_deeModel, "any_signal", G_CALLBACK(onRowRemoved), m_parent, NULL);
        g_object_disconnect(m_deeModel, "any_signal", G_CALLBACK(onRowChanged), m_parent, NULL);

        g_object_unref(m_deeModel);
        m_deeModel = NULL;
        m_parent->setReady(false);
    }
}

void
DeeListModelPrivate::connectToDeeModel()
{
    disconnectFromDeeModel();

    if (!m_name.isEmpty())
    {
        m_deeModel = dee_shared_model_new_with_name(m_name.toUtf8().data());
        g_signal_connect(m_deeModel, "ready", G_CALLBACK(onReady), m_parent);
        g_signal_connect(m_deeModel, "row-added", G_CALLBACK(onRowAdded), m_parent);
        g_signal_connect(m_deeModel, "row-removed", G_CALLBACK(onRowRemoved), m_parent);
        g_signal_connect(m_deeModel, "row-changed", G_CALLBACK(onRowChanged), m_parent);
        dee_shared_model_connect(DEE_SHARED_MODEL(m_deeModel));
    }
}

void
DeeListModelPrivate::createRoles()
{
    if (m_deeModel == NULL) {
        return;
    }

    QHash<int, QByteArray> roles;
    QString column;
    guint n_columns = dee_model_get_n_columns(m_deeModel);

    for (int index=0; index<n_columns; index++)
    {
        column = QString("column_%1").arg(index);
        roles[index] = column.toAscii();
    }

    m_parent->setRoleNames(roles);
    m_parent->reset();
}

void
DeeListModelPrivate::onReady(GObject* emitter __attribute__ ((unused)),
                             DeeListModel *model)
{
    model->d->createRoles();
    model->setReady(true);
}

void
DeeListModelPrivate::onRowAdded(GObject *emitter __attribute__ ((unused)),
                                DeeModelIter* iter,
                                DeeListModel* model)
{
    gint position = dee_model_get_position(model->d->m_deeModel, iter);
    /* Force emission of QAbstractItemModel::rowsInserted by calling
       beginInsertRows and endInsertRows. Necessary because according to the
       documentation:
       "It can only be emitted by the QAbstractItemModel implementation, and
        cannot be explicitly emitted in subclass code."
    */
    model->beginInsertRows(QModelIndex(), position, position);
    model->endInsertRows();
}

void
DeeListModelPrivate::onRowRemoved(GObject *emitter __attribute__ ((unused)),
                                  DeeModelIter* iter,
                                  DeeListModel* model)
{
    gint position = dee_model_get_position(model->d->m_deeModel, iter);
    /* Force emission of QAbstractItemModel::rowsRemoved by calling
       beginRemoveRows and endRemoveRows. Necessary because according to the
       documentation:
       "It can only be emitted by the QAbstractItemModel implementation, and
        cannot be explicitly emitted in subclass code."
    */
    model->beginRemoveRows(QModelIndex(), position, position);
    model->endRemoveRows();
}

void
DeeListModelPrivate::onRowChanged(GObject *emitter __attribute__ ((unused)),
                                  DeeModelIter* iter,
                                  DeeListModel* model)
{
    gint position = dee_model_get_position(model->d->m_deeModel, iter);
    QModelIndex index = model->index(position);
    Q_EMIT model->dataChanged(index, index);
}



DeeListModel::DeeListModel(QObject *parent) :
    QAbstractListModel(parent), d(new DeeListModelPrivate(this))
{
    g_type_init();
}

DeeListModel::~DeeListModel()
{
    delete d;
}

QString
DeeListModel::name() const
{
    return d->m_name;
}

void
DeeListModel::setName(const QString& name)
{
    if (name != d->m_name) {
        d->m_name = name;
        Q_EMIT nameChanged(d->m_name);
        d->connectToDeeModel();
    }
}

bool
DeeListModel::ready() const
{
    return d->m_ready;
}

void
DeeListModel::setReady(bool ready)
{
    if (ready != d->m_ready) {
        d->m_ready = ready;
        Q_EMIT readyChanged(d->m_ready);
    }
}

int
DeeListModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)

    if (d->m_deeModel == NULL || !d->m_ready) {
        return 0;
    }

    return dee_model_get_n_rows(d->m_deeModel);
}

QVariant
DeeListModel::data(const QModelIndex &index, int role) const
{
    if (d->m_deeModel == NULL || !d->m_ready) {
        return QVariant();
    }

    if (!index.isValid())
        return QVariant();

    GValue gvalue = { 0 };
    DeeModelIter* iter;

    iter = dee_model_get_iter_at_row(d->m_deeModel, index.row());
    dee_model_get_value(d->m_deeModel, iter, role, &gvalue);
    QVariant qvariant = QVariantFromGValue(&gvalue);
    g_value_unset(&gvalue);

    return qvariant;
}
