/*
 * Copyright (C) 2005-2009 Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_v4lsink.c
 *
 * Description:    Implementation of V4L Sink Plugin for Gstreamer
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 */

/*=============================================================================
                            INCLUDE FILES
=============================================================================*/


#include <gst/gst.h>
#include <gst/video/gstvideosink.h>
#include <linux/videodev.h>
#ifdef USE_X11
#include <linux/mxcfb.h>
//#include "mxcfb.h"

#endif
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <string.h>
#include "mfw_gst_utils.h"
#include "mfw_gst_v4lsink.h"

#if defined(ENABLE_TVOUT) && defined (_MX27)
/*For TV-Out & change para on-the-fly*/
#include <errno.h>
#include <sys/time.h>
struct v4l2_output_dev
{
    __u32 disp_num;             /* output device index, for TV is 2, for LCD is 3 */
    __u32 id_len;               /* string id length */
    __u8 id[16];                /* string id of deivce, e.g. TV "DISP3 TV" */
};
#define VIDIOC_PUT_OUTPUT       _IOW  ('V', 90, struct v4l2_output_dev)
#define VIDIOC_GET_OUTPUT       _IOW  ('V', 91, struct v4l2_output_dev)
#endif


#ifdef _MX233
#define V4L2_BUF_TYPE_VIDEO_OUTPUT_OVERLAY  8
#define V4L2_FBUF_FLAG_GLOBAL_ALPHA	0x0010
#define V4L2_CAP_VIDEO_OUTPUT_OVERLAY   0x00000200  /* Can do video output overlay */
#endif

//#define GST_DEBUG g_print
/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/
/* None */

/*=============================================================================
                LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/

enum {
    PROP_0,
    //PROP_FULLSCREEN,		/* enable full screen image display */
    DISP_WIDTH,			/* display image width */
    DISP_HEIGHT,		/* display image height */
    AXIS_TOP,			/* image top axis offset */
    AXIS_LEFT,			/* image left axis offset */
    ROTATE,			/* image rotation value (0 - 7) */
    CROP_LEFT,			/* input image cropping in the left */
    CROP_RIGHT,			/* input image cropping in the right */
    CROP_TOP,			/* input image cropping in the top */
    CROP_BOTTOM,		/* input image cropping in the top */
    FULL_SCREEN_WIDTH,		/* full screen width of the display */
    FULL_SCREEN_HEIGHT,		/* full screen height of the display */
    //BASE_OFFSET,
#ifdef ENABLE_TVOUT
    TV_OUT,
    TV_MODE,
#endif 
#ifdef ENABLE_DUMP
    DUMP_LOCATION,
#endif
    ADDITIONAL_BUFFER_DEPTH,
    SETPARA,
    IMAGE_WIDTH, 
    IMAGE_HEIGHT,
    PROP_STRETCH,
#ifdef USE_X11
    PROP_X11ENABLED,
#endif     
};

#define HW_DEINTERLACE
/*=============================================================================
                              LOCAL MACROS
=============================================================================*/
/* used for debugging */


#define GST_CAT_DEFAULT mfw_gst_v4lsink_debug

#if defined(_MX51)
#define MIN_BUFFER_NUM              2   /* minimal 2 is default for compability of non-updated codec like wmv */
#define MAX_BUFFER_NUM              10  /* this number is related to driver */
#define BUFFER_RESERVED_NUM         0   /* 0 addtional buffer need reserved for v4l queue in vpu based decoder */
#define MAX_V4L_ALLOW_SIZE_IN_MB    48  /* 48MB limitation */

#elif defined (_MX37)
#define MIN_BUFFER_NUM              2   /* minimal 2 is default for compability of non-updated codec like wmv */
#define MAX_BUFFER_NUM              10  /* this number is related to driver */
#define BUFFER_RESERVED_NUM         0   /* 0 addtional buffer need reserved for v4l queue in vpu based decoder */
#define MAX_V4L_ALLOW_SIZE_IN_MB    15  /* 15MB limitation */

#elif defined(_MX27)
#define MIN_BUFFER_NUM              2   /* minimal 2, 5 is default for compability of non-updated codec like wmv */
#define MAX_BUFFER_NUM              10  /* this number is related to driver */
#define BUFFER_RESERVED_NUM         0   /* 0 addtional buffer need reserved for v4l queue in vpu based decoder */
#define MAX_V4L_ALLOW_SIZE_IN_MB    10  /* 10MB limitation */

#else
#define MIN_BUFFER_NUM              5   /* minimal 2, 5 is default for compability of non-updated codec like wmv */
#define MAX_BUFFER_NUM              10  /* this number is related to driver */
#define BUFFER_RESERVED_NUM         2   /* 2 addtional buffer need reserved for v4l queue */
#define MAX_V4L_ALLOW_SIZE_IN_MB    7   /* 7MB limitation */
#endif

#define BUFFER_NEW_RETRY_MAX        500
#define WAIT_ON_DQUEUE_FAIL_IN_MS   30000

#define MAX_V4L_ALLOW_SIZE_IN_BYTE  (MAX_V4L_ALLOW_SIZE_IN_MB*1024*1024)


#define RESERVEDHWBUFFER_DEPTH    2  /* must less than MIN_BUFFER_NUM */

#define IS_RESERVEDHWBUFFER_FULL(v4linfo) \
    (((v4linfo)->swbuffer_max==0) \
    || (g_slist_length((v4linfo)->reservedhwbuffer_list)>=RESERVEDHWBUFFER_DEPTH))

#define PUSHRESERVEDHWBUFFER(v4linfo, buffer) \
    if (buffer){\
        ((v4linfo)->reservedhwbuffer_list)=g_slist_append(((v4linfo)->reservedhwbuffer_list),\
                                                   (buffer));\
    }

#define POPRESERVEDHWBUFFER(v4linfo, buffer) \
    if (((v4linfo)->reservedhwbuffer_list)==NULL){\
        (buffer) = NULL;\
    }else{\
        GSList * tmp;\
        tmp = ((v4linfo)->reservedhwbuffer_list);\
        buffer = tmp->data;\
        ((v4linfo)->reservedhwbuffer_list) = \
            g_slist_delete_link(((v4linfo)->reservedhwbuffer_list), tmp);\
    }





#define DQUEUE_MAX_LOOP		200
#define NEXTDQ_WAIT_MSEC	30

#ifdef ENABLE_TVOUT
/*For TV-Out & change para on-the-fly*/
#include <errno.h>
#include <sys/time.h>

#define NTSC    0
#define PAL     1
#define NV_MODE 2
#endif


#define COLORKEY_RED 32
#define COLORKEY_GREEN   4
#define COLORKEY_BLUE   48

#define RGB888(r,g,b)\
    ((((guint32)(r))<<16)|(((guint32)(g))<<8)|(((guint32)(b))))
#define RGB888TORGB565(rgb)\
    ((((rgb)<<8)>>27<<11)|(((rgb)<<16)>>26<<5)|(((rgb)<<24)>>27))

#define RGB565TOCOLORKEY(rgb)                              \
      ( ((rgb & 0xf800)<<8)  |  ((rgb & 0xe000)<<3)  |     \
        ((rgb & 0x07e0)<<5)  |  ((rgb & 0x0600)>>1)  |     \
        ((rgb & 0x001f)<<3)  |  ((rgb & 0x001c)>>2)  )

#ifndef IPU_ROTATE_NONE
#define IPU_ROTATE_NONE     0
#define IPU_ROTATE_180      3
#define IPU_ROTATE_90_LEFT  7
#define IPU_ROTATE_90_RIGHT 4
#endif

/*=============================================================================
                             STATIC VARIABLES
=============================================================================*/

static GstElementDetails mfw_gst_v4lsink_details =
GST_ELEMENT_DETAILS("Freescale: v4l_sink",
		    "Sink/Video",
		    "Video rendering device plugin used to display"
		    "YUV/RGB data with the support to input cropping",
		    "Multimedia Team <mmsw@freescale.com>");

/*=============================================================================
                             GLOBAL VARIABLES
=============================================================================*/
/* None */

/*=============================================================================
                        LOCAL FUNCTION PROTOTYPES
=============================================================================*/

GST_DEBUG_CATEGORY_STATIC(mfw_gst_v4lsink_debug);
static void mfw_gst_v4lsink_base_init(gpointer);
static void mfw_gst_v4lsink_class_init(MFW_GST_V4LSINK_INFO_CLASS_T *);
static void mfw_gst_v4lsink_init(MFW_GST_V4LSINK_INFO_T *,
                                 MFW_GST_V4LSINK_INFO_CLASS_T *);

static void mfw_gst_v4lsink_get_property(GObject *,
                                         guint, GValue *,
                                         GParamSpec *);
static void mfw_gst_v4lsink_set_property(GObject *,
                                         guint, const GValue *,
                                         GParamSpec *);

static GstStateChangeReturn mfw_gst_v4lsink_change_state
    (GstElement *, GstStateChange);

static gboolean mfw_gst_v4lsink_setcaps(GstBaseSink *, GstCaps *);

static GstFlowReturn mfw_gst_v4lsink_show_frame
                            (GstBaseSink *, GstBuffer *);

static gboolean mfw_gst_v4lsink_output_init(MFW_GST_V4LSINK_INFO_T *,
                                            guint, guint, guint);
static gboolean mfw_gst_v4lsink_output_setup(struct v4l2_format *,
                                             MFW_GST_V4LSINK_INFO_T *);
static MFWGstV4LSinkBuffer* mfw_gst_v4lsink_hwbuffer_new(MFW_GST_V4LSINK_INFO_T *);

static GstFlowReturn mfw_gst_v4lsink_buffer_alloc(GstBaseSink * bsink,
                                                  guint64 offset,
                                                  guint size, GstCaps * caps,
                                                  GstBuffer ** buf);

#ifdef USE_X11
static GstXContext *
gst_v4lsink_xcontext_get (MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink);
static PARAM_SET
mfw_gst_v4lsink_get_geometry(MFW_GST_V4LSINK_INFO_T * mfw_gst_v4lsink);
static gint
mfw_gst_v4lsink_set_colorkey(MFW_GST_V4LSINK_INFO_T * mfw_gst_v4lsink);

#endif


/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/
#ifdef SUSPEND_SUPPORT
typedef struct v4l_run_info {
    guint8 v4l_id;  /* v4l handle id */
    guint8 pm_mode; /* 0:Normal, 1:suspend, 2: resume */
}V4L_RUN_INFO;

void mfw_gst_v4lsink_suspend(MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink)
{
    gint type;
    int i;
    /* swicth off the video stream */
    mfw_gst_v4lsink->suspend = TRUE;
    g_print(RED_STR("SUSPENDDDDDDD\n", 0));
    type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
    if (mfw_gst_v4lsink->stream_on){
 
        if (ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_STREAMOFF, &type)<0)
            g_print("Set VIDIOC_STREAMOFF failed.\n");
        mfw_gst_v4lsink->stream_on = FALSE;
		
        g_mutex_lock(mfw_gst_v4lsink->pool_lock);           
        {
            MFWGstV4LSinkBuffer * v4lsinkbuffer;
            for (i=0;i<mfw_gst_v4lsink->buffers_required;i++){
                v4lsinkbuffer = mfw_gst_v4lsink->all_buffer_pool[i];
                if (v4lsinkbuffer){
                    if (v4lsinkbuffer->bufstate == BUF_STATE_SHOWING){
                        GST_WARNING("stream off, buffer %p state changed from SHOWING\n", v4lsinkbuffer);
                        v4lsinkbuffer->bufstate = BUF_STATE_SHOWED;
                        gst_buffer_unref(v4lsinkbuffer);
                        mfw_gst_v4lsink->v4lqueued --;
                    }
                }
            }
        }
	 g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
        mfw_gst_v4lsink->qbuff_count = 0;
    }
}

void mfw_gst_v4lsink_resume(MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink)
{
    int i;
    g_print(RED_STR("RESUMMMMM\n", 0));
    if (mfw_gst_v4lsink->stream_on){
 
        mfw_gst_v4lsink->stream_on = FALSE;
		
        g_mutex_lock(mfw_gst_v4lsink->pool_lock);           
        {
            MFWGstV4LSinkBuffer * v4lsinkbuffer;
            for (i=0;i<mfw_gst_v4lsink->buffers_required;i++){
                v4lsinkbuffer = mfw_gst_v4lsink->all_buffer_pool[i];
                if (v4lsinkbuffer){
                    if (v4lsinkbuffer->bufstate == BUF_STATE_SHOWING){
                        GST_WARNING("stream off, buffer %p state changed from SHOWING\n", v4lsinkbuffer);
                        v4lsinkbuffer->bufstate = BUF_STATE_SHOWED;
                        gst_buffer_unref(v4lsinkbuffer);
                        mfw_gst_v4lsink->v4lqueued --;
                    }
                }
            }
        }
	 g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
        mfw_gst_v4lsink->qbuff_count = 0;
    }
    mfw_gst_v4lsink->suspend = FALSE;
    //mfw_gst_v4lsink->qbuff_count = 0;
}


static gboolean mfw_gst_v4lsink_clear_runinfo(MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
    FILE *fp;
    char command[32]; /* v4l_id:powerstate */
    strcpy(command, "echo \"00\" > /tmp/v4l.pid");
    system(command);
    return TRUE;
}

static gboolean mfw_gst_v4lsink_get_runinfo(MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
    FILE *fp;
    V4L_RUN_INFO v4l_info; 
    if (v4l_sink_info->is_paused == TRUE)
        return FALSE;
    /* FIXME: Not support for multi-instance */
    memset(&v4l_info,0, sizeof(V4L_RUN_INFO));
    fp = fopen("/tmp/v4l.pid","r");
    if (fp == NULL)
        return FALSE;

    fread((char *)&v4l_info,sizeof(V4L_RUN_INFO),1,fp);
    GST_DEBUG("%c,%c\n",v4l_info.v4l_id,v4l_info.pm_mode);
    fclose(fp);
    if (v4l_info.pm_mode == '1') { // suspend
       mfw_gst_v4lsink_suspend(v4l_sink_info);
	   v4l_sink_info->suspend = TRUE;
    }
    else if (v4l_info.pm_mode == '2') { // resume
       
       mfw_gst_v4lsink_resume(v4l_sink_info);
       mfw_gst_v4lsink_clear_runinfo(v4l_sink_info);

    }
    return TRUE;
}
#endif

#ifdef ENABLE_DUMP
static gboolean dumpfile_open (MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
  /* open the file */
  if (v4l_sink_info->dump_location == NULL || v4l_sink_info->dump_location[0] == '\0')
    goto no_dumpfilename;

  v4l_sink_info->dumpfile = fopen (v4l_sink_info->dump_location, "wb");
  if (v4l_sink_info->dumpfile == NULL)
    goto open_failed;

  v4l_sink_info->dump_length = 0;

  GST_DEBUG_OBJECT (v4l_sink_info, "opened file %s", v4l_sink_info->dump_location);

  return TRUE;

  /* ERRORS */
no_dumpfilename:
  {
    GST_ERROR (">>V4L_SINK: No file name specified for dumping.");
    return FALSE;
  }
open_failed:
  {
    GST_ERROR (">>V4L_SINK: Could not open file \"%s\" for writing.", v4l_sink_info->dump_location);
    return FALSE;
  }
}

static void dumpfile_close (MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
  if (v4l_sink_info->dumpfile) {
    if (fclose (v4l_sink_info->dumpfile) != 0)
      goto close_failed;

    GST_DEBUG_OBJECT (v4l_sink_info, "closed file");
    v4l_sink_info->dumpfile = NULL;
  }
  return;

  /* ERRORS */
close_failed:
  {
    GST_ERROR(">>V4L_SINK: Error closing file:%s", v4l_sink_info->dump_location);
    return;
  }
}

static gboolean dumpfile_set_location (MFW_GST_V4LSINK_INFO_T * v4l_sink_info, const gchar * location)
{
  if (v4l_sink_info->dumpfile)
    goto was_open;

  g_free (v4l_sink_info->dump_location);
  if (location != NULL) {
    v4l_sink_info->enable_dump = TRUE;
    v4l_sink_info->dump_location = g_strdup (location);
  } else {
    v4l_sink_info->enable_dump = FALSE;
    v4l_sink_info->dump_location = NULL;
  }

  return TRUE;

  /* ERRORS */
was_open:
  {
    g_warning ("Changing the `dump_location' property on v4lsink when "
        "a file is open not supported.");
    return FALSE;
  }
}

static gboolean dumpfile_write (MFW_GST_V4LSINK_INFO_T * v4l_sink_info, GstBuffer * buffer)
{
  guint64 cur_pos;
  guint size;

  size = GST_BUFFER_SIZE (buffer);

  cur_pos = v4l_sink_info->dump_length;

  GST_DEBUG_OBJECT (v4l_sink_info, "writing %u bytes at %" G_GUINT64_FORMAT,
      size, cur_pos);

  if (size > 0 && GST_BUFFER_DATA (buffer) != NULL)
  {
      if (v4l_sink_info->cr_left_bypixel != 0 || v4l_sink_info->cr_right_bypixel != 0
          || v4l_sink_info->cr_top_bypixel != 0 || v4l_sink_info->cr_bottom_bypixel != 0)
      {
          /* remove black edge */
          gint y;
          char *p;
          gint cr_left = v4l_sink_info->cr_left_bypixel_orig;
          gint cr_right = v4l_sink_info->cr_right_bypixel_orig;
          gint cr_top = v4l_sink_info->cr_top_bypixel_orig;
          gint cr_bottom = v4l_sink_info->cr_bottom_bypixel_orig;
          gint stride = v4l_sink_info->width + cr_left + cr_right;

          /* Y */
          for (y = cr_top; y < v4l_sink_info->height + cr_top; y++)
          {
              p = (char *) (GST_BUFFER_DATA(buffer)) +
                  y * stride + cr_left;
              fwrite (p, 1, v4l_sink_info->width, v4l_sink_info->dumpfile);
              v4l_sink_info->dump_length += v4l_sink_info->width;
          }

          /* U */
          for (y = cr_top / 2; y < (v4l_sink_info->height + cr_top) / 2; y++)
          {
              p = (char *) (GST_BUFFER_DATA(buffer)) +
                  stride * (v4l_sink_info->height + cr_top + cr_bottom) +
                  (y * stride + cr_left) / 2;
              fwrite (p, 1, v4l_sink_info->width / 2, v4l_sink_info->dumpfile);
              v4l_sink_info->dump_length += (v4l_sink_info->width / 2);
          }

          /* V */
          for (y = cr_top / 2; y < (v4l_sink_info->height + cr_top) / 2; y++)
          {
              p = (char *) (GST_BUFFER_DATA(buffer)) +
                  stride * (v4l_sink_info->height + cr_top + cr_bottom) * 5 / 4 +
                  (y * stride + cr_left) / 2;
              fwrite (p, 1, v4l_sink_info->width / 2, v4l_sink_info->dumpfile);
              v4l_sink_info->dump_length += (v4l_sink_info->width / 2);
          }
      }
      else
      {
          if (fwrite (GST_BUFFER_DATA (buffer), size, 1, v4l_sink_info->dumpfile) != 1)
              goto handle_error;

          v4l_sink_info->dump_length += size;
      }
  }

  return TRUE;

handle_error:
  {
    switch (errno) {
      case ENOSPC:{
        GST_ELEMENT_ERROR (v4l_sink_info, RESOURCE, NO_SPACE_LEFT, (NULL), (NULL));
        break;
      }
      default:{
        GST_ELEMENT_ERROR (v4l_sink_info, RESOURCE, WRITE,
            (("Error while writing to file \"%s\"."), v4l_sink_info->dump_location),
            ("%s", g_strerror (errno)));
      }
    }
    return FALSE;
  }
}

#endif
#ifdef USE_X11


#ifdef XWIN_EVENTS

/* This function handles XEvents that might be in the queue. It generates
   GstEvent that will be sent upstream in the pipeline to handle interactivity
   and navigation. It will also listen for configure events on the window to
   trigger caps renegotiation so on the fly software scaling can work. */
static void
mfw_gst_v4lsink_handle_xevents (MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink)
{
  XEvent e;
  guint pointer_x = 0, pointer_y = 0;
  gboolean pointer_moved = FALSE;

  /* Handle Expose */
  {
    gboolean exposed = FALSE, configured = FALSE;

    g_mutex_lock (mfw_gst_v4lsink->x_lock);
    while (XCheckWindowEvent (mfw_gst_v4lsink->xcontext->disp,
            mfw_gst_v4lsink->xwin, ExposureMask | StructureNotifyMask,
            &e)) {
      g_mutex_unlock (mfw_gst_v4lsink->x_lock);

      switch (e.type) {
        case Expose:
          exposed = TRUE;
          g_print("Exposured to true.\n");
          break;
        case ConfigureNotify:
          configured = TRUE;
        default:
          break;
      }
      g_mutex_lock (mfw_gst_v4lsink->x_lock);
    }
    g_mutex_unlock (mfw_gst_v4lsink->x_lock);

    if (configured) {
        PARAM_SET param;
        GST_WARNING("Get configure events(%dx%d).\n",e.xconfigure.width,e.xconfigure.height);
        param = mfw_gst_v4lsink_get_geometry(mfw_gst_v4lsink);
        g_mutex_lock(mfw_gst_v4lsink->flow_lock);
        mfw_gst_v4lsink->setpara |= param;
        g_mutex_unlock(mfw_gst_v4lsink->flow_lock);

    }
    if (exposed) {
        g_print("Get the expose event.\n");
    }
  }
}

static gpointer
gst_v4lsink_event_thread (MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink)
{
    while (mfw_gst_v4lsink->running) {
        if (mfw_gst_v4lsink->xwin != 0) {
            gst_v4lsink_handle_xevents (mfw_gst_v4lsink);
        }
        g_usleep (100000);
    }
    return NULL;
}

#endif

static gboolean
mfw_gst_v4lsink_set_black(MFW_GST_V4LSINK_INFO_T * mfw_gst_v4lsink,
                          struct v4l2_crop * crop)
{
    gint lw, lh, rw, rh;

    lw = crop->c.left - mfw_gst_v4lsink->axis_left;
    rw = mfw_gst_v4lsink->disp_width-lw-crop->c.width;

    lw = (lw < 0) ? 0: lw;
    rw = (rw < 0) ? 0: rw;

    lh = crop->c.top - mfw_gst_v4lsink->axis_top;
    rh = mfw_gst_v4lsink->disp_height-lh - crop->c.height;

    lh = (lh < 0) ? 0: lh;
    rh = (rh < 0) ? 0: rh;

    if ((lh == 1) && (rh == 0))
    {
        rh = 7;
    }

   if ((lw == 1) && (rw == 0))
    {
        rw = 7;
    }


    GST_DEBUG("Borders (lw,lh,rw,rh) = (%d, %d, %d, %d).\n",lw, lh, rw , rh);


    XSetForeground (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->gc,
      mfw_gst_v4lsink->xcontext->black);


    /* Left border */
    if  (lw > 0) {
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0, 0,
            lw, mfw_gst_v4lsink->disp_height);

    }
    /* Right border */
    if (rw > 0) {
        if (crop->c.width == mfw_gst_v4lsink->xcontext->width) {
        GST_DEBUG("Walkaround for IPU.\n");
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, crop->c.width - 7, 0,
           rw, mfw_gst_v4lsink->disp_height);

        }
        else
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, lw + crop->c.width, 0,
           rw, mfw_gst_v4lsink->disp_height);
    }

    /* Top border */
    if  (lh > 0) {
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0, 0,
            mfw_gst_v4lsink->disp_width, lh);
    }
    /* Bottom border */
    if (rh > 0) {
        if (crop->c.height == mfw_gst_v4lsink->xcontext->height) {
        g_print("Walkaround for IPU.\n");
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0,crop->c.height - 7 ,
            mfw_gst_v4lsink->disp_width, rh);
        }
        else {
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0,lh + crop->c.height,
            mfw_gst_v4lsink->disp_width, rh);

        }
    }

    XSync(mfw_gst_v4lsink->xcontext->disp, FALSE);

    return TRUE;
}

static gboolean
mfw_gst_v4lsink_fill_color(MFW_GST_V4LSINK_INFO_T * mfw_gst_v4lsink,gulong color)
{
    gint width,height;
    gint lw, lh, rw, rh;

    struct v4l2_crop * crop = &mfw_gst_v4lsink->crop;

    if (mfw_gst_v4lsink->stream_on == FALSE)
        return FALSE;

    lw = crop->c.left - mfw_gst_v4lsink->axis_left;
    rw = mfw_gst_v4lsink->disp_width-lw-crop->c.width;

    lw = (lw < 0) ? 0: lw;
    rw = (rw < 0) ? 0: rw;

    lh = crop->c.top - mfw_gst_v4lsink->axis_top;
    rh = mfw_gst_v4lsink->disp_height-lh - crop->c.height;

    lh = (lh < 0) ? 0: lh;
    rh = (rh < 0) ? 0: rh;

    if ((lh == 1) && (rh == 0))
    {
        rh = 7;
    }

   if ((lw == 1) && (rw == 0))
    {
        rw = 7;
    }


    width = crop->c.width;
    height = crop->c.height;

    g_mutex_lock (mfw_gst_v4lsink->x_lock);

    GST_DEBUG("[%s]:set xwindow color :0x%08x in (%dx%d)\n",__FUNCTION__,
        color,
        width, height);

    XSetForeground (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->gc,
                    color);


    XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
        mfw_gst_v4lsink->gc, lw, lh,
        width, height);

    XSync(mfw_gst_v4lsink->xcontext->disp, FALSE);


    GST_DEBUG("Borders (lw,lh,rw,rh) = (%d, %d, %d, %d).\n",lw, lh, rw , rh);


    XSetForeground (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->gc,
      mfw_gst_v4lsink->xcontext->black);


    /* Left border */
    if  (lw > 0) {
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0, 0,
            lw, mfw_gst_v4lsink->disp_height);

    }
    /* Right border */
    if (rw > 0) {
        if (crop->c.width == mfw_gst_v4lsink->xcontext->width) {
        GST_DEBUG("Walkaround for IPU.\n");
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, crop->c.width - 7, 0,
           rw, mfw_gst_v4lsink->disp_height);

        }
        else
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, lw + crop->c.width, 0,
           rw, mfw_gst_v4lsink->disp_height);
    }

    /* Top border */
    if  (lh > 0) {
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0, 0,
            mfw_gst_v4lsink->disp_width, lh);
    }
    /* Bottom border */
    if (rh > 0) {
        if (crop->c.height == mfw_gst_v4lsink->xcontext->height) {
        GST_DEBUG("Walkaround for IPU.\n");
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0,crop->c.height - 7 ,
            mfw_gst_v4lsink->disp_width, rh);
        }
        else {
        XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            mfw_gst_v4lsink->gc, 0,lh + crop->c.height,
            mfw_gst_v4lsink->disp_width, rh);

        }
    }

    XSync(mfw_gst_v4lsink->xcontext->disp, FALSE);

    g_mutex_unlock (mfw_gst_v4lsink->x_lock);


    return TRUE;
}

static gboolean
mfw_gst_v4lsink_clear_color(MFW_GST_V4LSINK_INFO_T * mfw_gst_v4lsink)
{
    gint width,height;
    gint lw, lh, rw, rh;

    struct v4l2_crop * crop = &mfw_gst_v4lsink->crop;

    if (mfw_gst_v4lsink->stream_on == TRUE)
        return FALSE;

    lw = crop->c.left - mfw_gst_v4lsink->axis_left;
    rw = mfw_gst_v4lsink->disp_width-lw-crop->c.width;

    lw = (lw < 0) ? 0: lw;
    rw = (rw < 0) ? 0: rw;

    lh = crop->c.top - mfw_gst_v4lsink->axis_top;
    rh = mfw_gst_v4lsink->disp_height-lh - crop->c.height;

    lh = (lh < 0) ? 0: lh;
    rh = (rh < 0) ? 0: rh;

    if ((lh == 1) && (rh == 0))
    {
        rh = 7;
    }

   if ((lw == 1) && (rw == 0))
    {
        rw = 7;
    }


    width = crop->c.width;
    height = crop->c.height;

    g_mutex_lock (mfw_gst_v4lsink->x_lock);

    GST_DEBUG("[%s]:Clear xwindow by set black in (%dx%d)\n",__FUNCTION__,
        width, height);

    XSetForeground (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->gc,
                    mfw_gst_v4lsink->xcontext->black);


    XFillRectangle (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
        mfw_gst_v4lsink->gc, lw, lh,
        width, height);

    XSync(mfw_gst_v4lsink->xcontext->disp, FALSE);


    g_mutex_unlock (mfw_gst_v4lsink->x_lock);


    return TRUE;
}


static gpointer
mfw_gst_v4lsink_event_thread (MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink)
{
    while (mfw_gst_v4lsink->running) {
        if (mfw_gst_v4lsink->xwin != 0) {
            mfw_gst_v4lsink_handle_xevents (mfw_gst_v4lsink);
        }
        g_usleep (100000);
    }
    return NULL;
}

static void
gst_xv4lsink_set_xwindow_id (GstXOverlay * overlay, XID xwindow_id)
{
    MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink = MFW_GST_V4LSINK(overlay);
    XWindowAttributes attr;
    XGCValues values;

    GST_WARNING("[%s] xwindow_id: %d\n", __FUNCTION__, xwindow_id);

    /* If we already use that window return */
    if (xwindow_id == 0) {
        g_print("invalid window id.\n");
        return;
    }

    if(mfw_gst_v4lsink->xwin == xwindow_id) {
        PARAM_SET param;
        gulong color;
        param = mfw_gst_v4lsink_get_geometry(mfw_gst_v4lsink);

        mfw_gst_v4lsink_clear_color(mfw_gst_v4lsink);

        g_mutex_lock(mfw_gst_v4lsink->flow_lock);
        mfw_gst_v4lsink->setpara |= param;
        g_mutex_unlock(mfw_gst_v4lsink->flow_lock);

        color = RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE));
        mfw_gst_v4lsink_fill_color(mfw_gst_v4lsink,color);

        GST_DEBUG("set param to %x.\n",mfw_gst_v4lsink->setpara);

        return;
    }

    /* If a window is there already we destroy it */
    if (mfw_gst_v4lsink->xcontext == NULL) {
        mfw_gst_v4lsink->xcontext = gst_v4lsink_xcontext_get (mfw_gst_v4lsink);
    }
    mfw_gst_v4lsink->xwin = xwindow_id;

    mfw_gst_v4lsink->gc = XCreateGC (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
            0, &values);

    XMapRaised (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin);

#ifdef XWIN_EVENTS

    mfw_gst_v4lsink->running = TRUE;
    mfw_gst_v4lsink->event_thread = g_thread_create (
            (GThreadFunc) mfw_gst_v4lsink_event_thread, mfw_gst_v4lsink, TRUE, NULL);

    XSelectInput(mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin,
    StructureNotifyMask);

#endif

}

static void
gst_ximagesink_xoverlay_init (GstXOverlayClass * iface)
{
  iface->set_xwindow_id = gst_xv4lsink_set_xwindow_id;
}

#endif

#if defined(ENABLE_TVOUT) && defined (_MX27)
void tv_out_open(MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
    struct v4l2_output_dev odev = {
        .disp_num = 2,
        .id_len = 11,
        .id = "DISP3 TVOUT"
    };

    v4l_sink_info->fd_tvout = open("/dev/fb/1", O_RDWR);
    if (v4l_sink_info->fd_tvout < 0) {
        GST_ERROR(">>V4L_SINK: Unable to open /dev/fb/1\n");
    }

    if (ioctl(v4l_sink_info->v4l_id, VIDIOC_PUT_OUTPUT, &odev) < 0)
        GST_ERROR(">>V4L_SINK: TV-OUT ioctl VIDIOC_PUT_OUTPUT failed!\n");

}

void tv_out_close(MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
    if (v4l_sink_info->fd_tvout > 0) {
        struct v4l2_output_dev odev = {
            .disp_num = 2,
            .id_len = 11,
            .id = "DISP3 TVOUT"
        };

        if (ioctl(v4l_sink_info->v4l_id, VIDIOC_GET_OUTPUT, &odev) < 0)
            GST_ERROR(">>V4L_SINK: TV-OUT ioctl VIDIOC_GET_OUTPUT failed!\n");

        close(v4l_sink_info->fd_tvout);
        v4l_sink_info->fd_tvout = 0;
    }
}
#endif

static void
mfw_gst_v4lsink_buffer_finalize(MFWGstV4LSinkBuffer *v4lsink_buffer_released)
{
    MFW_GST_V4LSINK_INFO_T * v4lsink_info;

    /*use buf state to control drop frame */
    g_return_if_fail(v4lsink_buffer_released != NULL);
    v4lsink_info = v4lsink_buffer_released->v4lsinkcontext;

    switch (v4lsink_buffer_released->bufstate) {
    case BUF_STATE_ALLOCATED:
        GST_DEBUG(">>V4L_SINK: Buffer %d maybe dropped.\n", v4lsink_buffer_released->v4l_buf.index);
        v4lsink_info->frame_dropped++;
        if (!(v4lsink_info->frame_dropped & 0x3f)){
            GST_ERROR(">>V4L_SINK: %d dropped while %d showed!\n", v4lsink_info->frame_dropped, v4lsink_info->qbuff_count);
        }
    case BUF_STATE_SHOWED:
        g_mutex_lock(v4lsink_info->pool_lock);

        v4lsink_buffer_released->bufstate = BUF_STATE_IDLE;
	    if (GST_BUFFER_FLAG_IS_SET(v4lsink_buffer_released, GST_BUFFER_FLAG_LAST)){
            /* hwbuffer, put on the head of free pool. */
            if (!IS_RESERVEDHWBUFFER_FULL(v4lsink_info)){
                PUSHRESERVEDHWBUFFER(v4lsink_info, v4lsink_buffer_released);
            }else{
                v4lsink_info->free_pool = g_slist_prepend(v4lsink_info->free_pool, v4lsink_buffer_released);
            }
        }else{
            /* swbuffer, put on the tail of free pool. */
            v4lsink_info->free_pool = g_slist_append(v4lsink_info->free_pool, v4lsink_buffer_released);
        }
        g_mutex_unlock(v4lsink_info->pool_lock);
	    gst_buffer_ref(GST_BUFFER_CAST(v4lsink_buffer_released));
	break;

    case BUF_STATE_FREE:
        /*free it,do not need ref. */
        GST_DEBUG(">>V4L_SINK: Buffer %d is freed.\n", v4lsink_buffer_released->v4l_buf.index);

        if (GST_BUFFER_DATA(v4lsink_buffer_released)){
            if (GST_BUFFER_FLAG_IS_SET(v4lsink_buffer_released, GST_BUFFER_FLAG_LAST)){
                munmap(GST_BUFFER_DATA(v4lsink_buffer_released),
	                v4lsink_buffer_released->v4l_buf.length);
            }else{
                g_free(GST_BUFFER_DATA(v4lsink_buffer_released));
            }
        }
        v4lsink_info->all_buffer_pool[v4lsink_buffer_released->v4l_buf.index] = NULL;

        GST_BUFFER_DATA(v4lsink_buffer_released) = NULL;

        v4lsink_info->querybuf_index--;
        if (v4lsink_info->querybuf_index==0){
            /* close the v4l driver */
            g_free(v4lsink_info->all_buffer_pool);
            v4lsink_info->all_buffer_pool = NULL;
            close(v4lsink_info->v4l_id);

#ifdef USE_X11
        if (v4lsink_info->fd_fb){
            close(v4lsink_info->fd_fb);
            //g_print("close device %d\n", v4lsink_info->fd_fb);
            v4lsink_info->fd_fb = 0;
        }
#endif
            g_print(">>V4L_SINK: All buffer freed, close device.\n");
        }
	break;

    default:
        gst_buffer_ref(GST_BUFFER_CAST(v4lsink_buffer_released));
        GST_ERROR(">>V4L_SINK: Buffer %d:%p is unref with error state %d!\n",  v4lsink_buffer_released->v4l_buf.index, v4lsink_buffer_released, v4lsink_buffer_released->bufstate);
    }

    return;

}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_buffer_init

DESCRIPTION:        This funtion initialises the buffer class of the V4lsink
                    plug-in

ARGUMENTS PASSED:
        v4lsink_buffer -   pointer to V4Lsink buffer class
        g_class        -   global pointer


RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/


static void
mfw_gst_v4lsink_buffer_init(MFWGstV4LSinkBuffer * v4lsink_buffer,
			    gpointer g_class)
{

    memset(&v4lsink_buffer->v4l_buf, 0, sizeof(struct v4l2_buffer));
    return;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_buffer_class_init

DESCRIPTION:        This funtion registers the  funtions used by the
                    buffer class of the V4lsink plug-in

ARGUMENTS PASSED:
        g_class        -   class from which the mini objext is derived
        class_data     -   global class data

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static void
mfw_gst_v4lsink_buffer_class_init(gpointer g_class, gpointer class_data)
{
    GstMiniObjectClass *mini_object_class = GST_MINI_OBJECT_CLASS(g_class);
    mini_object_class->finalize = (GstMiniObjectFinalizeFunction)mfw_gst_v4lsink_buffer_finalize;
    return;

}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_buffer_get_type

DESCRIPTION:        This funtion registers the  buffer class
                    on to the V4L sink plugin

ARGUMENTS PASSED:   None

RETURN VALUE:       return the registered buffer class

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

GType mfw_gst_v4lsink_buffer_get_type(void)
{

    static GType _mfw_gst_v4lsink_buffer_type;

    if (G_UNLIKELY(_mfw_gst_v4lsink_buffer_type == 0)) {
	static const GTypeInfo v4lsink_buffer_info = {
	    sizeof(GstBufferClass),
	    NULL,
	    NULL,
	    mfw_gst_v4lsink_buffer_class_init,
	    NULL,
	    NULL,
	    sizeof(MFWGstV4LSinkBuffer),
	    0,
	    (GInstanceInitFunc) mfw_gst_v4lsink_buffer_init,
	    NULL
	};
	_mfw_gst_v4lsink_buffer_type =
	    g_type_register_static(GST_TYPE_BUFFER, "MFWGstV4LSinkBuffer",
				   &v4lsink_buffer_info, 0);
    }

    return _mfw_gst_v4lsink_buffer_type;
}

/*=============================================================================
FUNCTION:          mfw_gst_v4lsink_close

DESCRIPTION:       This funtion clears the list of all the buffers maintained
                   in the buffer pool. swirches of the video stream and closes
                   the V4L device driver.

ARGUMENTS PASSED:   v4l_sink_info  - V4lsink plug-in context

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_v4lsink_close(MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
    MFWGstV4LSinkBuffer *v4lsink_buffer = NULL;
    GstBuffer ** pbuffer;
    gint type;
    gint totalbuffernum = (v4l_sink_info->buffers_required+v4l_sink_info->swbuffer_max);
    gint i;

    // Exit if we have already closed before to avoid hangs
    if (v4l_sink_info->pool_lock == NULL)
    {
        return;
    }

    g_mutex_lock(v4l_sink_info->pool_lock);

    if (v4l_sink_info->init)
    {
        /* switch off the video stream */
        type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
        ioctl(v4l_sink_info->v4l_id, VIDIOC_STREAMOFF, &type);
    }

    if (v4l_sink_info->all_buffer_pool)
    {
        /* try to unref all buffer in pool */
        for (i=0;i<totalbuffernum;i++)
        {
            v4lsink_buffer = (MFWGstV4LSinkBuffer *)(v4l_sink_info->all_buffer_pool[i]);

            /* for buffers in IDLE and SHOWING state, no unref outside, explicit unref it */
            if (v4lsink_buffer)
            {
                GST_WARNING(">>V4L_SINK: try to free buffer %d at state %d\n",
                v4lsink_buffer->v4l_buf.index, v4lsink_buffer->bufstate);

                if ((v4lsink_buffer->bufstate == BUF_STATE_IDLE) ||
                    (v4lsink_buffer->bufstate == BUF_STATE_SHOWING))
                {
                    if (v4lsink_buffer->bufstate == BUF_STATE_IDLE)
                    {
                        v4l_sink_info->free_pool = g_slist_remove(v4l_sink_info->free_pool, v4lsink_buffer);
                        v4l_sink_info->reservedhwbuffer_list =
                            g_slist_remove(v4l_sink_info->reservedhwbuffer_list, v4lsink_buffer);
                    }
                    v4lsink_buffer->bufstate = BUF_STATE_FREE;
                    gst_buffer_unref(v4lsink_buffer);
                } else {
                    v4lsink_buffer->bufstate = BUF_STATE_FREE;
                }
            }
        }
    }


    g_mutex_unlock(v4l_sink_info->pool_lock);

    GST_DEBUG(">>V4L SINK: Close the v4l device.\n");


#ifdef ENABLE_DUMP
    if (v4l_sink_info->enable_dump)
        dumpfile_close(v4l_sink_info);
#endif
    g_mutex_free(v4l_sink_info->pool_lock);
    v4l_sink_info->pool_lock = NULL;

#ifdef ENABLE_TVOUT
    if(v4l_sink_info->tv_out == TRUE)
#if defined(_MX31) || defined(_MX35)
    {
        /* switch to LCD mode when playing over*/
        FILE *pfb0_mode;
        gchar *mode = "U:480x640p-67\n";

        pfb0_mode = fopen("/sys/class/graphics/fb0/mode", "w");
        fwrite(mode, 1, strlen(mode), pfb0_mode);
        fflush(pfb0_mode);
        fclose(pfb0_mode);

    }
#endif
#if defined(_MX37) || defined(_MX51)
    {
        /* blank fb1 for tv*/
        FILE *pfb1_blank;
        gchar *blank = "4\n";

        pfb1_blank = fopen("/sys/class/graphics/fb1/blank", "w");
        fwrite(blank, 1, strlen(blank), pfb1_blank);
        fflush(pfb1_blank);
        fclose(pfb1_blank);
    }
#endif

#if defined (_MX27)
    {
        tv_out_close(v4l_sink_info);
    }
#endif

#endif
    v4l_sink_info->init=FALSE;

    return;
}



/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_new_swbuffer

DESCRIPTION:        This function allocate a new software display buffer.

ARGUMENTS PASSED:
        v4l_sink_info -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       returns the pointer to the software display buffer

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
MFWGstV4LSinkBuffer *mfw_gst_v4lsink_new_swbuffer(MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
    MFWGstV4LSinkBuffer *v4lsink_buffer;
    struct v4l2_buffer *v4lbuf;
    void * pdata;
    gint buf_size;

    g_return_val_if_fail(MFW_GST_IS_V4LSINK(v4l_sink_info), NULL);

    v4lsink_buffer = (MFWGstV4LSinkBuffer *)gst_mini_object_new(MFW_GST_TYPE_V4LSINK_BUFFER);

    v4lsink_buffer->bufstate = BUF_STATE_FREE;

    /* try to allocate data buffer for swbuffer */
    if (v4l_sink_info->outformat != V4L2_PIX_FMT_RGB565)
        buf_size = (v4l_sink_info->width+v4l_sink_info->cr_left_bypixel+v4l_sink_info->cr_right_bypixel)*
                        (v4l_sink_info->height+v4l_sink_info->cr_left_bypixel+v4l_sink_info->cr_right_bypixel)*3/2;
    else
        buf_size = (v4l_sink_info->width+v4l_sink_info->cr_left_bypixel+v4l_sink_info->cr_right_bypixel)*
                        (v4l_sink_info->height+v4l_sink_info->cr_left_bypixel+v4l_sink_info->cr_right_bypixel)*2;

    pdata = g_malloc(buf_size);

    if (pdata==NULL){
        GST_ERROR(">>V4L_SINK: Can not allocate data buffer for swbuffer!\n");
        gst_buffer_unref(v4lsink_buffer);
        return NULL;
    }

    GST_BUFFER_DATA(v4lsink_buffer) = pdata;
    GST_BUFFER_OFFSET(v4lsink_buffer) = 0;

    v4lbuf = &v4lsink_buffer->v4l_buf;

    memset(v4lbuf, 0, sizeof(struct v4l2_buffer));
    v4lbuf->index = v4l_sink_info->querybuf_index;

    v4lsink_buffer->v4lsinkcontext = v4l_sink_info;

    v4lsink_buffer->bufstate = BUF_STATE_IDLE;
    /* register swbuffer to buffer pool */
    v4l_sink_info->all_buffer_pool[v4l_sink_info->querybuf_index++] = v4lsink_buffer;

    return v4lsink_buffer;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_new_hwbuffer

DESCRIPTION:        This function allocated a new hardware V4L  buffer.

ARGUMENTS PASSED:
        v4l_sink_info -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       returns the pointer to the hardware V4L buffer

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
MFWGstV4LSinkBuffer *mfw_gst_v4lsink_new_hwbuffer(MFW_GST_V4LSINK_INFO_T* v4l_sink_info)
{
    MFWGstV4LSinkBuffer *v4lsink_buffer = NULL;
    guint image_width = 0;
    gint extra_pixel = 0;
    struct v4l2_buffer *v4lbuf;
    gint cr_left = 0, cr_right = 0, cr_top = 0;

    g_return_val_if_fail(MFW_GST_IS_V4LSINK(v4l_sink_info), NULL);

    v4lsink_buffer = (MFWGstV4LSinkBuffer *)gst_mini_object_new(MFW_GST_TYPE_V4LSINK_BUFFER);
    memset(&v4lsink_buffer->v4l_buf, 0, sizeof(struct v4l2_buffer));

    v4lbuf = &v4lsink_buffer->v4l_buf;
    v4lbuf->index = v4l_sink_info->querybuf_index;
    v4lbuf->type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
    v4lbuf->memory = V4L2_MEMORY_MMAP;

    /* Buffer queried from the /V4L driver */

    if (ioctl(v4l_sink_info->v4l_id, VIDIOC_QUERYBUF, v4lbuf) < 0) {
    	GST_ERROR(">>V4L_SINK: VIDIOC_QUERYBUF failed %d\n", v4l_sink_info->querybuf_index);
    	v4lsink_buffer = NULL;
    	goto queryret;

    }

    /* Buffer queried for is mapped from the /V4L driver space */
    GST_BUFFER_OFFSET(v4lsink_buffer) = (size_t) v4lbuf->m.offset;
    GST_BUFFER_DATA(v4lsink_buffer) =
	mmap(NULL, v4lbuf->length,
	     PROT_READ | PROT_WRITE, MAP_SHARED,
	     v4l_sink_info->v4l_id, v4lbuf->m.offset);

    if (GST_BUFFER_DATA(v4lsink_buffer) == NULL) {
	    GST_ERROR(">>V4L_SINK: v4l2_out test: mmap failed\n");
	    v4lsink_buffer = NULL;
	    goto queryret;
    }


    cr_left = v4l_sink_info->cr_left_bypixel;
    cr_right = v4l_sink_info->cr_right_bypixel;
    cr_top = v4l_sink_info->cr_top_bypixel;

    /* The input cropping is set here */
    if ((cr_left != 0) || (cr_right != 0) || (cr_top != 0)) {
	image_width = v4l_sink_info->width;
	v4lbuf->m.offset = v4lbuf->m.offset
	    + (cr_top * (image_width + cr_left + cr_right))
	    + cr_left;
    }
    v4lsink_buffer->bufstate = BUF_STATE_IDLE;
    GST_BUFFER_FLAG_SET(v4lsink_buffer, GST_BUFFER_FLAG_LAST);
    v4lsink_buffer->v4lsinkcontext = v4l_sink_info;
    v4l_sink_info->all_buffer_pool[v4l_sink_info->querybuf_index++] = v4lsink_buffer;
queryret:
    return v4lsink_buffer;
}



/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_new_buffer

DESCRIPTION:        This function gets a  v4l buffer, hardware or software.

ARGUMENTS PASSED:
        v4l_sink_info -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       returns the pointer to the v4l buffer

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

MFWGstV4LSinkBuffer *mfw_gst_v4lsink_new_buffer(MFW_GST_V4LSINK_INFO_T* v4l_sink_info)
{
    gint type = 0;
    MFWGstV4LSinkBuffer *v4lsink_buffer = NULL;
    struct v4l2_buffer *v4lbuf = NULL;
    {
        int loopcount = 0;
        GSList * tmp;
        int ret;
        struct v4l2_buffer v4l2buf;

        g_mutex_lock(v4l_sink_info->pool_lock);

        if ((IS_RESERVEDHWBUFFER_FULL(v4l_sink_info)) && (tmp = v4l_sink_info->free_pool)){
                v4lsink_buffer = (MFWGstV4LSinkBuffer *)tmp->data;
                    v4lsink_buffer->bufstate = BUF_STATE_ALLOCATED;
                v4l_sink_info->free_pool = g_slist_delete_link(v4l_sink_info->free_pool, tmp);
                g_mutex_unlock(v4l_sink_info->pool_lock);
                return v4lsink_buffer;
        }

        while ((loopcount++)<BUFFER_NEW_RETRY_MAX){
            ret = 0;
            memset(&v4l2buf, 0, sizeof(struct v4l2_buffer));
		    v4l2buf.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
		    v4l2buf.memory = V4L2_MEMORY_MMAP;

    		if ((v4l_sink_info->v4lqueued>1)
                &&(!((ret = ioctl(v4l_sink_info->v4l_id, VIDIOC_DQBUF, &v4l2buf))< 0))) {
                    MFWGstV4LSinkBuffer  * v4lsinkbuffer;
                    v4lsinkbuffer = (MFWGstV4LSinkBuffer  *)(v4l_sink_info->all_buffer_pool[v4l2buf.index]);
                    if ((v4lsinkbuffer) && (v4lsinkbuffer->bufstate == BUF_STATE_SHOWING)){
                    v4l_sink_info->v4lqueued --;
                    v4lsinkbuffer->bufstate = BUF_STATE_SHOWED;
                    g_mutex_unlock(v4l_sink_info->pool_lock);
                    gst_buffer_unref(GST_BUFFER_CAST(v4lsinkbuffer));
                    g_mutex_lock(v4l_sink_info->pool_lock);
                    }
            }

            if (tmp = v4l_sink_info->free_pool){
                v4lsink_buffer = (MFWGstV4LSinkBuffer *)tmp->data;
                v4lsink_buffer->bufstate = BUF_STATE_ALLOCATED;
                v4l_sink_info->free_pool = g_slist_delete_link(v4l_sink_info->free_pool, tmp);
                g_mutex_unlock(v4l_sink_info->pool_lock);
                return v4lsink_buffer;
            }
            if (v4l_sink_info->v4lqueued<2){//no hardware buffer
                if (v4l_sink_info->swbuffer_count<v4l_sink_info->swbuffer_max){
                    v4lsink_buffer = mfw_gst_v4lsink_new_swbuffer(v4l_sink_info);
                    v4lsink_buffer->bufstate = BUF_STATE_ALLOCATED;
                    v4l_sink_info->swbuffer_count++;
                    g_mutex_unlock(v4l_sink_info->pool_lock);

                    return v4lsink_buffer;
                }

            }
            if (ret<0){
                g_mutex_unlock(v4l_sink_info->pool_lock);
                usleep(WAIT_ON_DQUEUE_FAIL_IN_MS);
                g_mutex_lock(v4l_sink_info->pool_lock);
            }

        }
        GST_ERROR(">>V4L_SINK: Try new buffer failed, ret %d %s queued %d\n", errno, strerror(errno), v4l_sink_info->v4lqueued);

        g_mutex_unlock(v4l_sink_info->pool_lock);
        return NULL;
    }
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_output_setup

DESCRIPTION:        This function set up the display device format

ARGUMENTS PASSED:
        fmt            -   pointer to format for the display device
        v4l_sink_info  -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE( sucess/failure)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean
mfw_gst_v4lsink_output_setup(struct v4l2_format *fmt,
			                 MFW_GST_V4LSINK_INFO_T * v4l_sink_info)
{
    struct v4l2_requestbuffers buf_req;

	int ret;

    if (ret=ioctl(v4l_sink_info->v4l_id, VIDIOC_S_FMT, fmt) < 0) {
	    GST_ERROR(">>V4L_SINK: set format failed %d\n",ret);
	    return FALSE;
    }

    if (ioctl(v4l_sink_info->v4l_id, VIDIOC_G_FMT, fmt) < 0) {
	    GST_ERROR(">>V4L_SINK: get format failed\n");
	    return FALSE;
    }

#if 0//test code for sw copy render, also need set MIN_BUFFER_NUM 2
    v4l_sink_info->swbuffer_max = v4l_sink_info->buffers_required-2;
    v4l_sink_info->buffers_required = 2;

#endif

    while(v4l_sink_info->buffers_required>=MIN_BUFFER_NUM){

        memset(&buf_req, 0, sizeof(buf_req));
        buf_req.count = v4l_sink_info->buffers_required;
        buf_req.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
        buf_req.memory = V4L2_MEMORY_MMAP;


        if (ioctl(v4l_sink_info->v4l_id, VIDIOC_REQBUFS, &buf_req) >= 0) {
    	    GST_DEBUG(">>V4L_SINK: %d hwbuffers sucessfully allocated.\n", v4l_sink_info->buffers_required);
    	    return TRUE;
        }


#ifdef NO_SWBUFFER
        g_print(RED_STR("Can not allocate enough v4lbuffer\n", 0));

        return FALSE;
#endif
        g_print("allocated software buffer\n");
        if (v4l_sink_info->swbuffer_max == 0)
            v4l_sink_info->swbuffer_max = 2;
        v4l_sink_info->swbuffer_max++;
        v4l_sink_info->buffers_required--;
    }

    v4l_sink_info->buffers_required =
        v4l_sink_info->swbuffer_max = 0;

    return FALSE;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_output_init

DESCRIPTION:        This function initialise the display device with the specified parameters.

ARGUMENTS PASSED:
        v4l_sink_info  -   pointer to MFW_GST_V4LSINK_INFO_T
        inp_format     -   the display foramt
        disp_width     -   width to be displayed
        disp_height    -   height to be displayed

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

gboolean mfw_gst_v4lsink_output_init(MFW_GST_V4LSINK_INFO_T *v4l_sink_info,
                                     guint inp_format,
                                     guint disp_width, guint disp_height)
{
    struct v4l2_control ctrl;
    struct v4l2_framebuffer fb;
    struct v4l2_cropcap cropcap;
    struct v4l2_crop *crop;
#ifdef _MX233
    struct v4l2_capability cap;
    struct v4l2_output output;
#endif
    struct v4l2_format fmt;
    gboolean retval = TRUE;
    struct v4l2_buffer buf;
    gint i;
#ifdef _MX233
    gchar v4l_device1[100] = "/dev/video0";
#else
    gchar v4l_device1[100] = "/dev/video16";
#endif
#ifdef USE_X11
    gchar fb_device[100] = "/dev/fb0";
#endif
    guint in_fmt = V4L2_PIX_FMT_YUV420;
    guint in_width = 0, display_width = 0;
    guint in_height = 0, display_height = 0;
    guint video_width = 0, video_height = 0;
    gint cr_left = 0;
    gint cr_top = 0;
    gint cr_right = 0;
    gint cr_bottom = 0;

    guint in_width_chroma = 0, in_height_chroma = 0;
    gint crop_left_chroma = 0;
    gint crop_right_chroma = 0;
    gint crop_top_chroma = 0;
    gint crop_bottom_chroma = 0;
#ifdef _MX233
    gint out_idx;
#endif
    struct v4l2_mxc_offset off;

    crop = &v4l_sink_info->crop;
    if (v4l_sink_info->init == FALSE)
	v4l_sink_info->querybuf_index = 0;

    v4l_sink_info->qbuff_count = 0;
    v4l_sink_info->frame_dropped = 0;

    /* Read the variables passed by the user */
    display_width = disp_width;
    display_height = disp_height;
    in_fmt = inp_format;

    /*No need to open v4l device when it has opened--change para on-the-fly */
    if (v4l_sink_info->init == FALSE) {
        /* open the V4l device */
        if ((v4l_sink_info->v4l_id =
            open(v4l_device1, O_RDWR | O_NONBLOCK, 0)) < 0) {
            GST_ERROR(">>V4L_SINK: Unable to open %s\n", v4l_device1);
            retval = FALSE;
            goto err0;
        }

        #ifdef USE_X11
	if ((v4l_sink_info->fd_fb =
	     open(fb_device, O_RDWR, 0)) < 0) {
	    g_print("Unable to open %s %d\n", fb_device, v4l_sink_info->fd_fb);
        v4l_sink_info->fd_fb = 0;
	    retval = FALSE;
	    goto err0;
	}
#endif
    }


#ifdef ENABLE_TVOUT
#if defined(_MX31) || defined(_MX35) 

    if (TRUE == v4l_sink_info->tv_out){
        gint out;
        gchar *mode;
        FILE *pfb0_mode;

        pfb0_mode = fopen("/sys/class/graphics/fb0/mode", "w");
        GST_DEBUG(">>V4L_SINK: pfb0_mode : %x\n", pfb0_mode );
        if(v4l_sink_info->tv_mode == PAL) {
            mode = "U:640x480p-50\n";
            fwrite(mode, 1, strlen(mode), pfb0_mode);
        }
        else if(v4l_sink_info->tv_mode == NTSC) {
            mode = "U:640x480p-60\n";
            fwrite(mode, 1, strlen(mode), pfb0_mode);
        }
        else {
            GST_ERROR(">>V4L_SINK: Wrong TV mode.\n");
            fclose(pfb0_mode);
            goto err0;
        }
        fflush(pfb0_mode);
        close(pfb0_mode);

        out = 3;
        ioctl(v4l_sink_info->v4l_id, VIDIOC_S_OUTPUT, &out);
    }
    else {
        gint out = 3;
        gchar *mode;
        FILE *pfb0_mode;

        if((v4l_sink_info->tv_mode == PAL) || (v4l_sink_info->tv_mode == NTSC)) {
            v4l_sink_info->tv_mode = NV_MODE;
            pfb0_mode = fopen("/sys/class/graphics/fb0/mode", "w");
            mode = "U:480x640p-67\n";
            fwrite(mode, 1, strlen(mode), pfb0_mode);
            fflush(pfb0_mode);
            fclose(pfb0_mode);
        }
        ioctl(v4l_sink_info->v4l_id, VIDIOC_S_OUTPUT, &out);
    }
 #endif

#if defined(_MX37) || defined(_MX51)
    if (TRUE == v4l_sink_info->tv_out){
        gint out;
        gchar *mode;
        FILE *pfb1_mode;

        pfb1_mode = fopen("/sys/class/graphics/fb1/mode", "w");
        if(v4l_sink_info->tv_mode == PAL) {
            mode = "U:720x576i-50\n";
            fwrite(mode, 1, strlen(mode), pfb1_mode);
        }
        else if(v4l_sink_info->tv_mode == NTSC) {
            mode = "U:720x480i-60\n";
            fwrite(mode, 1, strlen(mode), pfb1_mode);
        }
        else {
            GST_ERROR(">>V4L_SINK: Wrong TV mode.\n");
            fclose(pfb1_mode);
            goto err0;
        }
        fflush(pfb1_mode);
        fclose(pfb1_mode);

        out = 5;
        ioctl(v4l_sink_info->v4l_id, VIDIOC_S_OUTPUT, &out);
    }
    else {
        gint out = 3;
        gchar *blank = "4\n";
#ifndef USE_X11
        FILE *pfb1_blank;

        pfb1_blank = fopen("/sys/class/graphics/fb1/blank", "w");
        fwrite(blank, 1, strlen(blank), pfb1_blank);
        fflush(pfb1_blank);
        fclose(pfb1_blank);
#endif

        v4l_sink_info->tv_mode = NV_MODE;
        ioctl(v4l_sink_info->v4l_id, VIDIOC_S_OUTPUT, &out);
    }
#endif

#if  defined(_MX27)
    if (TRUE == v4l_sink_info->tv_out)
                tv_out_open(v4l_sink_info);
    else
                tv_out_close(v4l_sink_info);
#endif
#endif

#ifdef ENABLE_DUMP
    if (TRUE == v4l_sink_info->enable_dump)
        dumpfile_open(v4l_sink_info);
#endif
    memset(&cropcap, 0, sizeof(cropcap));
    cropcap.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
#ifndef _MX233
    if (ioctl(v4l_sink_info->v4l_id, VIDIOC_CROPCAP, &cropcap) < 0) {
        GST_ERROR(">>V4L_SINK: get crop capability failed\n");
        retval = FALSE;
        goto err0;
    }
#endif

#ifdef _MX233
        if (ioctl(v4l_sink_info->v4l_id, VIDIOC_QUERYCAP, &cap) < 0) {
             GST_ERROR(">>V4L_SINK:query cap failed\n");
             retval = FALSE;
             goto err0;;
        }

	if (!(cap.capabilities &
		(V4L2_CAP_VIDEO_OUTPUT | V4L2_CAP_VIDEO_OUTPUT_OVERLAY))) {
		GST_ERROR("video output overlay not detected\n");
		retval = FALSE;
		goto err0;

	}


	out_idx = 1;

	if (ioctl(v4l_sink_info->v4l_id, VIDIOC_S_OUTPUT, &out_idx) < 0) {
		GST_ERROR("failed to set output\n");
		retval = FALSE;
		goto err0;

	}

	output.index = out_idx;

	if (ioctl(v4l_sink_info->v4l_id, VIDIOC_ENUMOUTPUT, &output) < 0)
	{
		GST_ERROR("failed to VIDIOC_ENUMOUTPUT\n");
		retval = FALSE;
		goto err0;
	}


	fb.flags = V4L2_FBUF_FLAG_OVERLAY;
	fb.flags |= V4L2_FBUF_FLAG_GLOBAL_ALPHA;
	if (ioctl(v4l_sink_info->v4l_id, VIDIOC_S_FBUF, &fb) < 0) {
             GST_ERROR(">>V4L_SINK: set fbuf failed\n");
             retval = FALSE;
             goto err0;
	}
#endif
    //GST_DEBUG(">>V4L_SINK: cropcap bound left=%d, top=%d, width=%d height=%d\n",cropcap.bounds.left,cropcap.bounds.top, cropcap.bounds.width, cropcap.bounds.height);
    //GST_DEBUG(">>V4L_SINK: cropcap defrect left=%d, top=%d, width=%d height=%d\n",cropcap.defrect.left,cropcap.defrect.top, cropcap.defrect.width, cropcap.defrect.height);
    //GST_DEBUG(">>V4L_SINK: cropcap pixelaspect num=%d, den=%d\n",cropcap.pixelaspect.numerator,cropcap.pixelaspect.denominator);

    /* set the image rectangle of the display by
       setting the appropriate parameters */
#ifndef _MX233
    crop->type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
#else
    crop->type = V4L2_BUF_TYPE_VIDEO_OUTPUT_OVERLAY;
#endif

#if 1 //

    v4l_sink_info->fullscreen_width = cropcap.defrect.width;
    v4l_sink_info->fullscreen_height = cropcap.defrect.height;
#if 0
    if ((cropcap.defrect.width == display_width)
        && (cropcap.defrect.height == display_height ))
        v4l_sink_info->full_screen = TRUE;
    else
        v4l_sink_info->full_screen = FALSE;
    /* set the image rectangle of the display by
       setting the appropriate parameters */
#endif
    display_width = (display_width>>3)<<3;
    display_height = (display_height>>3)<<3;




	crop->c.width = display_width;
	crop->c.height = display_height;
	crop->c.top = v4l_sink_info->axis_top;
	crop->c.left = v4l_sink_info->axis_left;

    if (!(v4l_sink_info->stretch)){
    
        if (v4l_sink_info->rotate == IPU_ROTATE_90_LEFT
                || v4l_sink_info->rotate == IPU_ROTATE_90_RIGHT)
        {
            video_width = v4l_sink_info->height;
            video_height = v4l_sink_info->width;
        }
        else
        {
            video_width = v4l_sink_info->width;
            video_height = v4l_sink_info->height;
        }
        if (crop->c.width * video_height > crop->c.height * video_width)
        {
            int width = video_width * crop->c.height / video_height;
            width = (width>>3)<<3;
            crop->c.left = crop->c.left + (crop->c.width - width) / 2;
            crop->c.width = width;
        }
        else if (crop->c.width * video_height < crop->c.height * video_width)
        {
            int height = video_height * crop->c.width / video_width;
            height = (height>>3)<<3;
            crop->c.top = crop->c.top + (crop->c.height - height) / 2;
            crop->c.height = height;
        }
        else
        {
            /* do nothing */
        }
    }

    //WORKAROUND FIX ME
    if  ((v4l_sink_info->rotate==0)
        &&((v4l_sink_info->cr_left_bypixel)
        ||(v4l_sink_info->cr_top_bypixel)
        ||(v4l_sink_info->cr_right_bypixel)
        ||(v4l_sink_info->cr_bottom_bypixel))){
        if ((v4l_sink_info->width==crop->c.width) 
            &&(v4l_sink_info->height==crop->c.height)){
            g_print("workaround in\n");
            if (v4l_sink_info->width>v4l_sink_info->height){
                crop->c.height-=8;
            }else{
                crop->c.width-=8;
            }
        }
    }
    //


    /* Set the xwindow background to black */

    g_print("[V4L Display]: left=%d, top=%d, width=%d, height=%d\n",
            crop->c.left, crop->c.top,
            crop->c.width, crop->c.height);

#endif


    if (ioctl(v4l_sink_info->v4l_id, VIDIOC_S_CROP, crop) < 0) {
        GST_ERROR(">>V4L_SINK: set crop failed\n");
        retval = FALSE;
        goto err0;
    }

    /* Set the rotation */
    /* Base offset removed as it is changed in the new BSP */
    ctrl.id = V4L2_CID_PRIVATE_BASE; //+ v4l_sink_info->base_offset;
    ctrl.value = v4l_sink_info->rotate;
    if (ioctl(v4l_sink_info->v4l_id, VIDIOC_S_CTRL, &ctrl) < 0) {
        GST_ERROR(">>V4L_SINK: set ctrl failed\n");
        retval = FALSE;
        goto err0;
    }

    /*No need to set input fmt and req buffer again when change para on-the-fly */
    if (v4l_sink_info->init == FALSE)
    {
        /* set the input cropping parameters */
        in_width = v4l_sink_info->width;
        in_height = v4l_sink_info->height;
        memset(&fmt, 0, sizeof(fmt));
        fmt.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;

        cr_left = v4l_sink_info->cr_left_bypixel;
        cr_top = v4l_sink_info->cr_top_bypixel;
        cr_right = v4l_sink_info->cr_right_bypixel;
        cr_bottom = v4l_sink_info->cr_bottom_bypixel;
#ifndef _MX233
        fmt.fmt.pix.width = in_width;
        fmt.fmt.pix.height = in_height;
        fmt.fmt.pix.pixelformat = in_fmt;
#else
        fmt.fmt.pix.width = in_width + cr_left + cr_right;
        fmt.fmt.pix.height = in_height + cr_top + cr_bottom;
        fmt.fmt.pix.pixelformat = in_fmt;
#endif

        in_width_chroma = in_width / 2;
        in_height_chroma = in_height / 2;

        crop_left_chroma = cr_left / 2;
        crop_right_chroma = cr_right / 2;
        crop_top_chroma = cr_top / 2;
        crop_bottom_chroma = cr_bottom / 2;

        if ((cr_left!=0) || (cr_top!=0) || (cr_right!=0) || (cr_bottom!=0))
        {

            if (in_fmt==V4L2_PIX_FMT_YUV420){
                off.u_offset =
                    ((cr_left + cr_right + in_width) * (in_height + cr_bottom)) - cr_left
                    + (crop_top_chroma * (in_width_chroma + crop_left_chroma + crop_right_chroma))
                    + crop_left_chroma;
                off.v_offset = off.u_offset +
                    (crop_left_chroma + crop_right_chroma + in_width_chroma)
                    * ( in_height_chroma  + crop_bottom_chroma + crop_top_chroma);

                fmt.fmt.pix.bytesperline = in_width + cr_left + cr_right;
                fmt.fmt.pix.priv = (guint32) & off;
                fmt.fmt.pix.sizeimage = (in_width + cr_left + cr_right)
                    * (in_height + cr_top + cr_bottom) * 3 / 2;
                
            }else if (in_fmt==V4L2_PIX_FMT_NV12){

                //g_print("%d %d %d %d\n", cr_left, cr_right, cr_top, cr_bottom);
                //g_print("%d %d\n", in_width, in_height);
                off.u_offset = off.v_offset = 
                    ((cr_left + cr_right + in_width) * (in_height + cr_bottom)) - cr_left 
                    + (crop_top_chroma* (cr_left + cr_right + in_width)) 
                    + cr_left;
                
                fmt.fmt.pix.bytesperline = in_width + cr_left + cr_right;
                fmt.fmt.pix.priv = (guint32) & off;
                fmt.fmt.pix.sizeimage = (in_width + cr_left + cr_right)
                    * (in_height + cr_top + cr_bottom) * 3 / 2;
                
            }else{
                g_print(RED_STR("unsupport input format 0x%x for non-zero crop\n", in_fmt));
            }
        } else {
            fmt.fmt.pix.bytesperline = in_width;
            fmt.fmt.pix.priv = 0;
            fmt.fmt.pix.sizeimage = 0;
        }

        retval = mfw_gst_v4lsink_output_setup(&fmt, v4l_sink_info);
        if (retval == FALSE)
        {
            GST_ERROR(">>V4L_SINK: Error in mfw_gst_v4lsink_output_setup\n");
        }
#ifdef _MX233
    fmt.type = V4L2_BUF_TYPE_VIDEO_OUTPUT_OVERLAY;
    fmt.fmt.win.w.left = cr_left;
    fmt.fmt.win.w.top = cr_top;
    fmt.fmt.win.w.width = in_width;
    fmt.fmt.win.w.height = in_height; 
    fmt.fmt.win.global_alpha = 0;
    fmt.fmt.win.chromakey = 0;
	if (ioctl(v4l_sink_info->v4l_id, VIDIOC_S_FMT, &fmt) < 0) {
		perror("VIDIOC_S_FMT output overlay");
		return 1;
	}
#endif
    }
 err0:
    return retval;

}

#ifdef USE_X11
static PARAM_SET
mfw_gst_v4lsink_get_geometry(MFW_GST_V4LSINK_INFO_T * mfw_gst_v4lsink)
{
    gboolean ret = 0;
    XWindowAttributes attr, root_attr;
    gint x=0, y=0;
    Window w;

    ret = PARAM_NULL; /* nothing to do */

    if(mfw_gst_v4lsink->xcontext == NULL
            || mfw_gst_v4lsink->xwin == 0
            || mfw_gst_v4lsink->gc == 0)
        return ret;


    g_mutex_lock (mfw_gst_v4lsink->x_lock);

    XGetWindowAttributes (mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin, &attr);
    XGetWindowAttributes (mfw_gst_v4lsink->xcontext->disp, attr.root, &root_attr);
    XTranslateCoordinates(mfw_gst_v4lsink->xcontext->disp, mfw_gst_v4lsink->xwin, attr.root,
            0, 0, &x, &y, &w);


    if(mfw_gst_v4lsink->axis_left != x
        || mfw_gst_v4lsink->axis_top != y) {

        mfw_gst_v4lsink->axis_left = x;
        mfw_gst_v4lsink->axis_top = y;

        GST_WARNING("set v4l param.\n");
        ret |= PARAM_SET_V4L;

    }
 
    if ( (attr.width<16) || (attr.height<16)) {
        GST_WARNING("Display window is :%d,%d,ignore it.\n",attr.width, attr.height);
        g_mutex_unlock(mfw_gst_v4lsink->x_lock);
        return ret;
    }
       
    if ( mfw_gst_v4lsink->disp_width != attr.width
    || mfw_gst_v4lsink->disp_height != attr.height) {
        GST_WARNING("set v4l param and color key.\n");

        ret |=  PARAM_SET_COLOR_KEY | PARAM_SET_V4L;
        mfw_gst_v4lsink->disp_width = attr.width;
        mfw_gst_v4lsink->disp_height = attr.height;

    }

    g_mutex_unlock (mfw_gst_v4lsink->x_lock);



    GST_DEBUG("%s:return %x.\n",__FUNCTION__,ret);
    return ret;
}

static gboolean
mfw_gst_v4lsink_set_colorkey(MFW_GST_V4LSINK_INFO_T * mfw_gst_v4lsink)
{
    gboolean ret = FALSE;

    struct mxcfb_color_key color_key;
    struct mxcfb_gbl_alpha alpha;
    gulong color;
    if(mfw_gst_v4lsink->xcontext == NULL
            || mfw_gst_v4lsink->xwin == 0
            || mfw_gst_v4lsink->gc == 0)
        return ret;
    if(mfw_gst_v4lsink->init) {

        color = RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE));

        mfw_gst_v4lsink_fill_color(mfw_gst_v4lsink,color);

        alpha.alpha = 255;
        alpha.enable = 1;
        if (ioctl(mfw_gst_v4lsink->fd_fb, MXCFB_SET_GBL_ALPHA, &alpha) < 0) {
            g_print("set color key failed.\n");
            ret = FALSE;
        }

        color_key.color_key = RGB565TOCOLORKEY(color);
        GST_DEBUG("set color key:0x%08x.\n",color_key.color_key);

        color_key.enable = 1;
        if (ioctl(mfw_gst_v4lsink->fd_fb, MXCFB_SET_CLR_KEY, &color_key) < 0) {
            g_print("set color key failed.\n");
            ret = FALSE;
        }
    }



    return ret;
}
#endif

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_buffer_alloc

DESCRIPTION:        This function initailise the v4l driver
                    and gets the new buffer for display

ARGUMENTS PASSED:
          bsink :   pointer to GstBaseSink
		  buf   :   pointer to new GstBuffer
		  size  :   size of the new buffer
          offset:   buffer offset
		  caps  :   pad capability

RETURN VALUE:       GST_FLOW_OK/GST_FLOW_ERROR

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn
mfw_gst_v4lsink_buffer_alloc(GstBaseSink * bsink, guint64 offset,
				     guint size, GstCaps * caps,
				     GstBuffer ** buf)
{
    GstBuffer *newbuf = NULL;
    MFW_GST_V4LSINK_INFO_T *v4l_sink_info = MFW_GST_V4LSINK(bsink);
    MFWGstV4LSinkBuffer *v4lsink_buffer = NULL;
    GstStructure *s = NULL;

    gint frame_buffer_size;
    gint max_frames;
    gint hwbuffernumforcodec;

    gboolean result = FALSE;

    gint aspectratio_n = 0, aspectratio_d = 0;
    gint bpp, depth;

    v4l_sink_info->buffer_alloc_called = TRUE;

    if (G_UNLIKELY(v4l_sink_info->init == FALSE))
    {
    
#ifdef USE_X11
        if (v4l_sink_info->x11enabled){
            gint timeout=40;//timeout 2s
            while ((v4l_sink_info->disp_height<16) && (timeout-->0)) {
                usleep(50000);
            }
        }
#endif
        // comment out because using videotestsrc as input it causes gst_caps asserts
        // not needed it seems
        //caps = gst_caps_make_writable(caps);
        s = gst_caps_get_structure(caps, 0);

        v4l_sink_info->buffers_required = 0;

        gst_structure_get_int(s, "width", &v4l_sink_info->width);
        gst_structure_get_int(s, "height", &v4l_sink_info->height);
        gst_structure_get_fraction(s, "pixel-aspect-ratio", &aspectratio_n,
				   &aspectratio_d);
        gst_structure_get_int(s, "crop-left-by-pixel",
			      &v4l_sink_info->cr_left_bypixel);
        gst_structure_get_int(s, "crop-top-by-pixel",
			      &v4l_sink_info->cr_top_bypixel);
        gst_structure_get_int(s, "crop-right-by-pixel",
			      &v4l_sink_info->cr_right_bypixel);
        gst_structure_get_int(s, "crop-bottom-by-pixel",
			      &v4l_sink_info->cr_bottom_bypixel);
        gst_structure_get_int(s, "num-buffers-required",
			      &v4l_sink_info->buffers_required);

    if (v4l_sink_info->buffers_required==0)
        v4l_sink_info->buffers_required = 6;

    gst_structure_get_int (s, "bpp", &bpp);
    gst_structure_get_int (s, "depth", &depth);

    GST_DEBUG("bpp:%x, depth:%x.\n",bpp,depth);
        gst_structure_get_fourcc(s, "format", &v4l_sink_info->fourcc);

    if (v4l_sink_info->fourcc == GST_STR_FOURCC("I420")) {
        GST_WARNING("Default mode: I420\n");
    }else if (v4l_sink_info->fourcc==GST_STR_FOURCC("NV12")){
        GST_WARNING("set to nv12 mode\n");
        v4l_sink_info->outformat = V4L2_PIX_FMT_NV12;
    }
    else if (bpp==32) {
        GST_WARNING("set to RGB mode.\n");
        v4l_sink_info->outformat = V4L2_PIX_FMT_RGB32;
    }
    else if (bpp==24) {
        GST_WARNING("Set the V4L display to RGB24 format.\n");
        v4l_sink_info->outformat = V4L2_PIX_FMT_RGB24;
    }
    else if (bpp=16) {
        GST_WARNING("Set the V4L display to RGB565 format.\n");
        v4l_sink_info->outformat = V4L2_PIX_FMT_RGB565;
    }
    else{
        GST_ERROR("\nWrong FOURCC Type for Display:%d.\n",v4l_sink_info->fourcc);
    }



#ifdef ENABLE_DUMP
        v4l_sink_info->cr_left_bypixel_orig = v4l_sink_info->cr_left_bypixel;
        v4l_sink_info->cr_right_bypixel_orig = v4l_sink_info->cr_right_bypixel;
        v4l_sink_info->cr_top_bypixel_orig = v4l_sink_info->cr_top_bypixel;
        v4l_sink_info->cr_bottom_bypixel_orig = v4l_sink_info->cr_bottom_bypixel;
#endif

        GST_DEBUG("crop_left_bypixel=%d\n",v4l_sink_info->cr_left_bypixel);
        GST_DEBUG("crop_top_by_pixel=%d\n",v4l_sink_info->cr_top_bypixel);
        GST_DEBUG("crop_right_bypixel=%d\n",v4l_sink_info->cr_right_bypixel);
        GST_DEBUG("crop_bottom_by_pixel=%d\n",v4l_sink_info->cr_bottom_bypixel);

        GST_DEBUG("aspectratio_n=%d\n", aspectratio_n);
        GST_DEBUG("aspectratio_d=%d\n", aspectratio_d);
        GST_DEBUG("\n Decoded Width = %d, Decoded Height = %d\n",
		      v4l_sink_info->width, v4l_sink_info->height);

        GST_DEBUG(">>V4L_SINK: Decoder maximal reserved %d buffers.\n",
                v4l_sink_info->buffers_required);

        v4l_sink_info->buffers_required += BUFFER_RESERVED_NUM;

        if (v4l_sink_info->buffers_required < MIN_BUFFER_NUM)
        {
	        v4l_sink_info->buffers_required = MIN_BUFFER_NUM;
        }

        frame_buffer_size = (v4l_sink_info->width* v4l_sink_info->height)*3/2;
        max_frames = MAX_V4L_ALLOW_SIZE_IN_BYTE/frame_buffer_size;

        if (v4l_sink_info->buffers_required > max_frames)
        {
            v4l_sink_info->swbuffer_max = 2;
            v4l_sink_info->swbuffer_max += (v4l_sink_info->buffers_required-max_frames);
            v4l_sink_info->buffers_required = max_frames;
        }

#if !defined(_Mx27)
        if ((v4l_sink_info->cr_left_bypixel == 0) &&
            (v4l_sink_info->crop_left != 0))
#endif
        {
            v4l_sink_info->cr_left_bypixel = v4l_sink_info->crop_left;
            v4l_sink_info->cr_top_bypixel = v4l_sink_info->crop_top;
            v4l_sink_info->cr_right_bypixel = v4l_sink_info->crop_right;
            v4l_sink_info->cr_bottom_bypixel = v4l_sink_info->crop_bottom;
        }

        v4l_sink_info->width = v4l_sink_info->width -
            v4l_sink_info->cr_left_bypixel -
            v4l_sink_info->cr_right_bypixel;

        v4l_sink_info->height = v4l_sink_info->height -
            v4l_sink_info->cr_top_bypixel -
            v4l_sink_info->cr_bottom_bypixel;

        /* if the aspect ratio is set by the decoder compute the
            display size and the pixet offsets for the display origin
            so that the display is resized to the given aspect ratio
            and it is displayed in the centre */
#if 0
        if (aspectratio_d != 0 && aspectratio_n != 0)
        {
            if (v4l_sink_info->disp_height == 0)
            {
                v4l_sink_info->disp_width =
                    ((gfloat) aspectratio_n / aspectratio_d)
		             * v4l_sink_info->width;
                v4l_sink_info->disp_height =
		            ((gfloat) aspectratio_n / aspectratio_d)
		            * v4l_sink_info->height;
            }

            if ((v4l_sink_info->axis_top == 0) && (v4l_sink_info->axis_left == 0))
            {
                if (v4l_sink_info->disp_height > v4l_sink_info->fullscreen_height)
                {
                    v4l_sink_info->axis_top = 0;
                } else {
                    v4l_sink_info->axis_top =
                        (v4l_sink_info->fullscreen_height -
                        v4l_sink_info->disp_height) / 2;
                }

                if (v4l_sink_info->disp_width > v4l_sink_info->fullscreen_width) {
		            v4l_sink_info->axis_left = 0;
                } else {
                    v4l_sink_info->axis_left =
                        (v4l_sink_info->fullscreen_width -
                        v4l_sink_info->disp_width) / 2;
                }
            }
        }
#endif        

        GST_DEBUG("\n display Width = %d, display Height = %d\n",
            v4l_sink_info->disp_width,
            v4l_sink_info->disp_height);

        GST_DEBUG("\n display origin offset top = %d, display origin offset left \
                = %d\n", v4l_sink_info->axis_top, v4l_sink_info->axis_left);
        g_mutex_lock(v4l_sink_info->pool_lock);

        /* initialize the V4l driver based on the parameters set by
	       the previous element */
        if (((v4l_sink_info->disp_width == 0) &&
             (v4l_sink_info->disp_height == 0)))
        {
            result = mfw_gst_v4lsink_output_init(v4l_sink_info,
                                    v4l_sink_info->outformat,
                                    v4l_sink_info->width,
                                    v4l_sink_info->height);
        } else {
            result = mfw_gst_v4lsink_output_init(v4l_sink_info,
                                    v4l_sink_info->outformat,
                                    v4l_sink_info->disp_width,
                                    v4l_sink_info->disp_height);
        }

        g_mutex_unlock(v4l_sink_info->pool_lock);

        if (result != TRUE) {
	        g_print("\n>>V4L_SINK: Failed to initalize the v4l driver\n");
	        mfw_gst_v4lsink_close(v4l_sink_info);
	        v4l_sink_info->init = FALSE;
	        return GST_FLOW_ERROR;
        }

        hwbuffernumforcodec = v4l_sink_info->buffers_required;
        if (v4l_sink_info->swbuffer_max == 0){
            hwbuffernumforcodec -= BUFFER_RESERVED_NUM;
        } else {
            hwbuffernumforcodec -= (BUFFER_RESERVED_NUM+RESERVEDHWBUFFER_DEPTH);
        }

        if (v4l_sink_info->buffers_required != hwbuffernumforcodec)
        {
            GValue value = {G_TYPE_INT, hwbuffernumforcodec};
            gst_structure_set_value(s,  "num-buffers-required", &value);
	    }

        g_print(">>V4L_SINK: Actually buffer status:\n\thardware buffer : %d\n\tsoftware buffer : %d\n",
            v4l_sink_info->buffers_required,
            v4l_sink_info->swbuffer_max);


        if (v4l_sink_info->all_buffer_pool){
            g_free(v4l_sink_info->all_buffer_pool);
            v4l_sink_info->all_buffer_pool = NULL;
        }

        v4l_sink_info->all_buffer_pool = g_malloc(sizeof(GstBuffer *)*
                            (v4l_sink_info->buffers_required+v4l_sink_info->swbuffer_max));

        if (v4l_sink_info->all_buffer_pool == NULL) {
	        GST_ERROR("\n>>V4L_SINK: Failed to allocate buffer pool container\n");
            return GST_FLOW_ERROR;
        }

        /* no software buffer at all, no reserved needed */

        v4l_sink_info->swbuffer_count = 0;

        memset(v4l_sink_info->all_buffer_pool, 0, (sizeof(GstBuffer *)*
                            (v4l_sink_info->buffers_required+v4l_sink_info->swbuffer_max)));
        v4l_sink_info->init = TRUE;
        {
            MFWGstV4LSinkBuffer * tmpbuffer;

            g_mutex_lock(v4l_sink_info->pool_lock);

            while(!IS_RESERVEDHWBUFFER_FULL(v4l_sink_info))
            {
                tmpbuffer = mfw_gst_v4lsink_new_hwbuffer(v4l_sink_info);
                if (tmpbuffer){
    	            PUSHRESERVEDHWBUFFER(v4l_sink_info, tmpbuffer);
                }
            }

            while (v4l_sink_info->querybuf_index < v4l_sink_info->buffers_required)
            {
                tmpbuffer = mfw_gst_v4lsink_new_hwbuffer(v4l_sink_info);
                if (tmpbuffer){
                    v4l_sink_info->free_pool = g_slist_prepend(v4l_sink_info->free_pool, tmpbuffer);
                }
            }

            g_mutex_unlock(v4l_sink_info->pool_lock);

        }
    }

    /* get the V4L hardware buffer */
    v4lsink_buffer = mfw_gst_v4lsink_new_buffer(v4l_sink_info);
    if (v4lsink_buffer == NULL) {
        GST_ERROR("\n!!>>V4L_SINK: Could not allocate buffer from V4L Driver\n");
        *buf = NULL;
        return GST_FLOW_ERROR;
    } else {
        GST_BUFFER_SIZE(v4lsink_buffer) = size;
    	newbuf = GST_BUFFER_CAST(v4lsink_buffer);
    	gst_buffer_set_caps(newbuf, caps);
    	*buf = newbuf;
    	return GST_FLOW_OK;
    }
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_set_property

DESCRIPTION:        This function is notified if application changes the
                    values of a property.

ARGUMENTS PASSED:
        object  -   pointer to GObject
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec

RETURN VALUE:       GST_FLOW_OK/GST_FLOW_ERROR
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static void
mfw_gst_v4lsink_set_property(GObject * object, guint prop_id,
			     const GValue * value, GParamSpec * pspec)
{

    MFW_GST_V4LSINK_INFO_T *v4l_sink_info = MFW_GST_V4LSINK(object);
    switch (prop_id) {
#if 0        
    case PROP_FULLSCREEN:
	v4l_sink_info->full_screen = g_value_get_boolean(value);
	GST_DEBUG("fullcreen = %d\n", v4l_sink_info->full_screen);
	break;
#endif    
    case DISP_WIDTH:
	v4l_sink_info->disp_width = g_value_get_int(value);
	GST_DEBUG("width = %d\n", v4l_sink_info->disp_width);
	break;
    case DISP_HEIGHT:
	v4l_sink_info->disp_height = g_value_get_int(value);
	GST_DEBUG("height = %d\n", v4l_sink_info->disp_height);
	break;
    case AXIS_TOP:
	v4l_sink_info->axis_top = g_value_get_int(value);
	GST_DEBUG("axis_top = %d\n", v4l_sink_info->axis_top);
	break;
    case AXIS_LEFT:
	v4l_sink_info->axis_left = g_value_get_int(value);
	GST_DEBUG("axis_left = %d\n", v4l_sink_info->axis_left);
	break;
    case ROTATE:
	v4l_sink_info->rotate = g_value_get_int(value);
	GST_DEBUG("rotate = %d\n", v4l_sink_info->rotate);
	break;
    case CROP_LEFT:
	v4l_sink_info->crop_left = g_value_get_int(value);
	GST_DEBUG("crop_left = %d\n", v4l_sink_info->crop_left);
	break;

    case CROP_RIGHT:
	v4l_sink_info->crop_right = g_value_get_int(value);
	GST_DEBUG("crop_right = %d\n", v4l_sink_info->crop_right);
	break;

    case CROP_TOP:
	v4l_sink_info->crop_top = g_value_get_int(value);
	GST_DEBUG("crop_top = %d\n", v4l_sink_info->crop_top);
	break;

    case CROP_BOTTOM:
	v4l_sink_info->crop_bottom = g_value_get_int(value);
	GST_DEBUG("crop_bottom = %d\n", v4l_sink_info->crop_bottom);
	break;

#if 0        
    case FULL_SCREEN_WIDTH:
	v4l_sink_info->fullscreen_width = g_value_get_int(value);
	GST_DEBUG("fullscreen_width = %d\n", v4l_sink_info->fullscreen_width);
	break;

    case FULL_SCREEN_HEIGHT:
	v4l_sink_info->fullscreen_height = g_value_get_int(value);
	GST_DEBUG("fullscreen_height = %d\n", v4l_sink_info->fullscreen_height);
	break;

    case BASE_OFFSET:
	v4l_sink_info->base_offset = g_value_get_int(value);
	GST_DEBUG("base_offset = %d\n", v4l_sink_info->base_offset);
	break;
#endif    
#ifdef ENABLE_TVOUT
    case TV_OUT:
	v4l_sink_info->tv_out = g_value_get_boolean(value);
	break;
    case TV_MODE:
	v4l_sink_info->tv_mode = g_value_get_int(value);
	break;
	/*It's an ugly code,consider how to realize it by event */
#endif
#ifdef ENABLE_DUMP
    case DUMP_LOCATION:
    dumpfile_set_location (v4l_sink_info, g_value_get_string (value));
    break;
#endif
    case SETPARA:
	v4l_sink_info->setpara |= g_value_get_int(value);
	break;
    
    case ADDITIONAL_BUFFER_DEPTH:
    v4l_sink_info->additional_buffer_depth = g_value_get_int(value);
    break;
    case PROP_STRETCH:
    v4l_sink_info->stretch = g_value_get_boolean(value);
    break;

#ifdef USE_X11
    case PROP_X11ENABLED:
    v4l_sink_info->x11enabled = g_value_get_boolean(value);
    break;
#endif

    default:
	G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
    }

    return;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_get_property

DESCRIPTION:        This function is notified if application requests the
                    values of a property.

ARGUMENTS PASSED:
        object  -   pointer to GObject
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static void
mfw_gst_v4lsink_get_property(GObject * object, guint prop_id,
			     GValue * value, GParamSpec * pspec)
{
    MFW_GST_V4LSINK_INFO_T *v4l_sink_info = MFW_GST_V4LSINK(object);
    switch (prop_id) {
#if 0        
        
    case PROP_FULLSCREEN:
	g_value_set_boolean(value, v4l_sink_info->full_screen);
	break;
#endif    
    case DISP_WIDTH:
	g_value_set_int(value, v4l_sink_info->disp_width);
	break;
    case DISP_HEIGHT:
	g_value_set_int(value, v4l_sink_info->disp_height);
	break;
    case AXIS_TOP:
	g_value_set_int(value, v4l_sink_info->axis_top);
	break;
    case AXIS_LEFT:
	g_value_set_int(value, v4l_sink_info->axis_left);
	break;
    case ROTATE:
	g_value_set_int(value, v4l_sink_info->rotate);
	break;
    case CROP_LEFT:
	g_value_set_int(value, v4l_sink_info->crop_left);
	break;
    case CROP_TOP:
	g_value_set_int(value, v4l_sink_info->crop_top);
	break;
    case CROP_RIGHT:
	g_value_set_int(value, v4l_sink_info->crop_right);
	break;
    case CROP_BOTTOM:
	g_value_set_int(value, v4l_sink_info->crop_bottom);
	break;
    
    case FULL_SCREEN_WIDTH:
	g_value_set_int(value, v4l_sink_info->fullscreen_width);
	break;
    case FULL_SCREEN_HEIGHT:
	g_value_set_int(value, v4l_sink_info->fullscreen_height);
	break;

    case IMAGE_WIDTH:
	g_value_set_int(value, v4l_sink_info->width);
	break;
    case IMAGE_HEIGHT:
	g_value_set_int(value, v4l_sink_info->height);
	break;
#if 0    
    case BASE_OFFSET:
	g_value_set_int(value, v4l_sink_info->base_offset);
	break;
#endif

#ifdef ENABLE_TVOUT
    case TV_OUT:
	g_value_set_boolean(value, v4l_sink_info->tv_out);
	break;
    case TV_MODE:
	g_value_set_int(value, v4l_sink_info->tv_mode);
	break;
#endif
#ifdef ENABLE_DUMP
    case DUMP_LOCATION:
	g_value_set_string(value, v4l_sink_info->dump_location);
    break;
#endif
    case SETPARA:
	g_value_set_int(value, v4l_sink_info->setpara);
	break;

    case ADDITIONAL_BUFFER_DEPTH:
	g_value_set_int(value, v4l_sink_info->additional_buffer_depth);
	break;

    case PROP_STRETCH:
    g_value_set_boolean(value, v4l_sink_info->stretch);
    break;
    
#ifdef USE_X11
    case PROP_X11ENABLED:
    g_value_set_boolean(value, v4l_sink_info->x11enabled);
    break;
#endif

    default:
	G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
	break;
    }

    return;
}

#define DEQUEUE_TIMES_IN_SHOW 16

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_show_frame

DESCRIPTION:        Process data to display

ARGUMENTS PASSED:
        pad -   pointer to GstPad;
        buf -   pointer to GstBuffer

RETURN VALUE:       GST_FLOW_OK/GST_FLOW_ERROR
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn mfw_gst_v4lsink_show_frame(GstBaseSink * basesink,
                                                GstBuffer * buf)
{
    MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink = MFW_GST_V4LSINK(basesink);
    struct v4l2_buffer *v4l_buf=NULL;
    GstBuffer *outbuffer=NULL;
    GSList *searchlist;
    gint type;

    guint8 i = 0;
    MFWGstV4LSinkBuffer *v4lsink_buffer = NULL;
    /* This is to enable the integration of the peer elements which do not 
    call the gst_pad_alloc_buffer() to allocate their output buffers */

#ifdef SUSPEND_SUPPORT
    // mfw_gst_v4lsink_get_runinfo(mfw_gst_v4lsink);

    if (mfw_gst_v4lsink->suspend){
        return GST_FLOW_OK;
    }
#endif
        
    if(G_UNLIKELY(mfw_gst_v4lsink->buffer_alloc_called == FALSE))
    {
        mfw_gst_v4lsink_buffer_alloc(basesink,0,GST_BUFFER_SIZE(buf),
            mfw_gst_v4lsink->store_caps,&outbuffer);
        memcpy(GST_BUFFER_DATA(outbuffer),GST_BUFFER_DATA(buf),GST_BUFFER_SIZE(buf));
        //gst_buffer_unref(outbuffer);
        mfw_gst_v4lsink->buffer_alloc_called = FALSE;
    }
    else
    {
        outbuffer=buf;
    }

#if 1

#ifdef USE_X11
    /* Set parameter if geometry is changed. */
    {
        PARAM_SET param;
        param = mfw_gst_v4lsink_get_geometry(mfw_gst_v4lsink);
        g_mutex_lock(mfw_gst_v4lsink->flow_lock);
        mfw_gst_v4lsink->setpara |= param;
        g_mutex_unlock(mfw_gst_v4lsink->flow_lock);
    }
#endif

    if (mfw_gst_v4lsink->stream_on == TRUE) {
        g_mutex_lock(mfw_gst_v4lsink->flow_lock);



        if (G_UNLIKELY((mfw_gst_v4lsink->setpara & PARAM_SET_V4L) == PARAM_SET_V4L)) {
            gint type;
            gboolean result = FALSE;
            mfw_gst_v4lsink->setpara &= (~PARAM_SET_V4L);
            g_mutex_unlock(mfw_gst_v4lsink->flow_lock);

            GST_DEBUG("start setpara begin\n");
            g_mutex_lock(mfw_gst_v4lsink->pool_lock);
            /* swicth off the video stream */
            type = V4L2_BUF_TYPE_VIDEO_OUTPUT;

            if (ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_STREAMOFF, &type)<0)
                g_print("Set VIDIOC_STREAMOFF failed.\n");
            mfw_gst_v4lsink->stream_on = FALSE;
#ifdef USE_X11
            mfw_gst_v4lsink_clear_color(mfw_gst_v4lsink);
#endif
            g_print("[display]: left=%d, top=%d, width=%d, height=%d\n",
                    mfw_gst_v4lsink->axis_left, mfw_gst_v4lsink->axis_top,
                    mfw_gst_v4lsink->disp_width, mfw_gst_v4lsink->disp_height);


            /* initialize the V4l driver based on the parameters set by
               the previous element */
            if (((mfw_gst_v4lsink->disp_width == 0) &&
                            (mfw_gst_v4lsink->disp_height == 0))) {
                    result =
        		mfw_gst_v4lsink_output_init(mfw_gst_v4lsink,
        					    mfw_gst_v4lsink->outformat,
        					    mfw_gst_v4lsink->width,
        					    mfw_gst_v4lsink->height);
        	} else {

        	    result =
        		mfw_gst_v4lsink_output_init(mfw_gst_v4lsink,
        					    mfw_gst_v4lsink->outformat,
        					    mfw_gst_v4lsink->disp_width,
        					    mfw_gst_v4lsink->disp_height);
        	}
            if (result != TRUE) {
        	    GST_ERROR("\nFailed to initalize the v4l driver\n");
                g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
        	    return GST_FLOW_ERROR;
        	}


            {
                MFWGstV4LSinkBuffer * v4lsinkbuffer;
                for (i=0;i<mfw_gst_v4lsink->buffers_required;i++){
                    v4lsinkbuffer = mfw_gst_v4lsink->all_buffer_pool[i];
                    if (v4lsinkbuffer){
                        if (v4lsinkbuffer->bufstate == BUF_STATE_SHOWING){
                            GST_WARNING("stream off, buffer %p state changed from SHOWING\n", v4lsinkbuffer);
                            v4lsinkbuffer->bufstate = BUF_STATE_SHOWED;
                            g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
                            gst_buffer_unref(v4lsinkbuffer);
                            g_mutex_lock(mfw_gst_v4lsink->pool_lock);
                            mfw_gst_v4lsink->v4lqueued --;
                        }
                    }
                }
            }
            g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
            GST_DEBUG("setpara end\n");

        }
        else {
            g_mutex_unlock(mfw_gst_v4lsink->flow_lock);
        }

        g_mutex_lock(mfw_gst_v4lsink->flow_lock);

        if (G_UNLIKELY((mfw_gst_v4lsink->setpara & PARAM_SET_COLOR_KEY) == PARAM_SET_COLOR_KEY)) {
            gint type;
            gboolean result = FALSE;
            mfw_gst_v4lsink->setpara &= (~PARAM_SET_COLOR_KEY);
            g_mutex_unlock(mfw_gst_v4lsink->flow_lock);
#ifdef USE_X11
            g_mutex_lock(mfw_gst_v4lsink->pool_lock);
            mfw_gst_v4lsink_set_colorkey(mfw_gst_v4lsink);
            g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
#endif

        }
        else {
            g_mutex_unlock(mfw_gst_v4lsink->flow_lock);

        }

        /* Show frame call set colorkey. */
        if (G_UNLIKELY(mfw_gst_v4lsink->qbuff_count == 2)) {
#ifdef USE_X11
            mfw_gst_v4lsink_set_colorkey(mfw_gst_v4lsink);
#endif
        }
    }

#else
    if (G_UNLIKELY(mfw_gst_v4lsink->setpara == TRUE)) {
    	gint type;
    	gboolean result = FALSE;
        GST_DEBUG(">>V4L_SINK: start setpara begin\n");
        g_mutex_lock(mfw_gst_v4lsink->pool_lock);
    	/* swicth off the video stream */
    	type = V4L2_BUF_TYPE_VIDEO_OUTPUT;

    	ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_STREAMOFF, &type);
    	/* initialize the V4l driver based on the parameters set by
    	   the previous element */
    	if (((mfw_gst_v4lsink->disp_width == 0) &&
    	     (mfw_gst_v4lsink->disp_height == 0))) {
    	    result =
    		mfw_gst_v4lsink_output_init(mfw_gst_v4lsink,
    					    mfw_gst_v4lsink->outformat,
    					    mfw_gst_v4lsink->width,
    					    mfw_gst_v4lsink->height);
    	} else {
    	    result =
    		mfw_gst_v4lsink_output_init(mfw_gst_v4lsink,
    					    mfw_gst_v4lsink->outformat,
    					    mfw_gst_v4lsink->disp_width,
    					    mfw_gst_v4lsink->disp_height);
    	}
        if (result != TRUE) {
    	    GST_ERROR("\n>>V4L_SINK: Failed to initalize the v4l driver\n");
            g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
    	    return GST_FLOW_ERROR;
    	}

        {
            MFWGstV4LSinkBuffer * v4lsinkbuffer;
            for (i=0;i<mfw_gst_v4lsink->buffers_required;i++){
                v4lsinkbuffer = mfw_gst_v4lsink->all_buffer_pool[i];
                if (v4lsinkbuffer){
                    if (v4lsinkbuffer->bufstate == BUF_STATE_SHOWING){
                        GST_DEBUG(">>V4L_SINK: stream off, buffer %p state changed from SHOWING\n", v4lsinkbuffer);
                        v4lsinkbuffer->bufstate = BUF_STATE_SHOWED;
                        g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
                        gst_buffer_unref(v4lsinkbuffer);
                        g_mutex_lock(mfw_gst_v4lsink->pool_lock);
                        mfw_gst_v4lsink->v4lqueued --;
                    }
                }
            }
        }
        g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
        GST_DEBUG(">>V4L_SINK: setpara end\n");

    	mfw_gst_v4lsink->setpara = FALSE;
    }
#endif
    g_mutex_lock(mfw_gst_v4lsink->pool_lock);

    if (G_UNLIKELY(!GST_BUFFER_FLAG_IS_SET(outbuffer, GST_BUFFER_FLAG_LAST))){//sw buffer

        POPRESERVEDHWBUFFER(mfw_gst_v4lsink, v4lsink_buffer);

        if (v4lsink_buffer){
            memcpy(GST_BUFFER_DATA(v4lsink_buffer), GST_BUFFER_DATA(outbuffer),
                GST_BUFFER_SIZE(outbuffer));
            ((MFWGstV4LSinkBuffer *)outbuffer)->bufstate = BUF_STATE_SHOWED;
        }else{
            //try to dq once only
            struct v4l2_buffer v4l2buf;

            memset(&v4l2buf, 0, sizeof(struct v4l2_buffer));
		    v4l2buf.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
		    v4l2buf.memory = V4L2_MEMORY_MMAP;


    		if (mfw_gst_v4lsink->v4lqueued>1){
                if(!((ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_DQBUF, &v4l2buf))< 0)) {

                    MFWGstV4LSinkBuffer  * v4lsinkbuffer;
                    v4lsinkbuffer = (MFWGstV4LSinkBuffer  *)(mfw_gst_v4lsink->all_buffer_pool[v4l2buf.index]);
                    if ((v4lsinkbuffer) && (v4lsinkbuffer->bufstate == BUF_STATE_SHOWING)){
                        mfw_gst_v4lsink->v4lqueued --;
                        v4lsinkbuffer->bufstate = BUF_STATE_SHOWED;
                        g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
                        gst_buffer_unref(GST_BUFFER_CAST(v4lsinkbuffer));
                        g_mutex_lock(mfw_gst_v4lsink->pool_lock);
                    }
                }
    		}

            POPRESERVEDHWBUFFER(mfw_gst_v4lsink, v4lsink_buffer);

            if (v4lsink_buffer){

                memcpy(GST_BUFFER_DATA(v4lsink_buffer), GST_BUFFER_DATA(outbuffer),
                    GST_BUFFER_SIZE(outbuffer));
                ((MFWGstV4LSinkBuffer *)outbuffer)->bufstate = BUF_STATE_SHOWED;
            }else{
                g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
                GST_WARNING(">>V4L_SINK: drop because no reserved hwbuffer%d\n", mfw_gst_v4lsink->v4lqueued);
                return GST_FLOW_OK;
            }
        }
    }else{
        v4lsink_buffer = (MFWGstV4LSinkBuffer *)outbuffer;
        if (mfw_gst_v4lsink->buffer_alloc_called==TRUE){
            gst_buffer_ref(GST_BUFFER_CAST(v4lsink_buffer));
        }
    }

    v4l_buf = &v4lsink_buffer->v4l_buf;
#ifdef ENABLE_DUMP
    if (mfw_gst_v4lsink->enable_dump){
        dumpfile_write (mfw_gst_v4lsink, v4lsink_buffer);
        v4lsink_buffer->bufstate = BUF_STATE_SHOWED;
        g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
        gst_buffer_unref(GST_BUFFER_CAST(v4lsink_buffer));
        return GST_FLOW_OK;

    }
#endif
    {
        /*display immediately */

	    struct timeval queuetime;
	    gettimeofday(&queuetime, NULL);
	    v4l_buf->timestamp = queuetime;
    }

    /* queue the buffer to be displayed into the V4L queue */

#ifdef HW_DEINTERLACE
        /* Set the field information to v4l buffer */
        v4l_buf->field = mfw_gst_v4lsink->field;
#endif
    if (G_UNLIKELY(ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_QBUF, v4l_buf) < 0)) {
    	GST_ERROR(">>V4L_SINK: VIDIOC_QBUF failed\n");
        g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
    	return GST_FLOW_ERROR;
    }

    mfw_gst_v4lsink->v4lqueued++;

    v4lsink_buffer->bufstate = BUF_STATE_SHOWING;

    /* Switch on the stream display as soon as there are more than 1 buffer
       in the V4L queue */
    if (G_UNLIKELY(mfw_gst_v4lsink->qbuff_count == 1)) {
#ifdef HW_DEINTERLACE

        gint field;
        struct v4l2_format fmt;
        gint err;
        GstCaps *caps;
        GstStructure *s;


        caps = GST_BUFFER_CAPS(buf);

        GST_INFO("****  caps:%s.\n",gst_caps_to_string(caps));
        s = gst_caps_get_structure(caps, 0);

        if (mfw_gst_v4lsink->width > 720)
            goto no_field;
        if (gst_structure_get_int (s, "field", &field) ) {
            switch (field) {
                case FIELD_TOP:
                    mfw_gst_v4lsink->field = V4L2_FIELD_TOP;
                    break;
                case FIELD_BOTTOM:
                    mfw_gst_v4lsink->field = V4L2_FIELD_BOTTOM;
                    break;
                case FIELD_INTERLACED_TB:
                    mfw_gst_v4lsink->field = V4L2_FIELD_INTERLACED_TB;
                    break;
                case FIELD_INTERLACED_BT:
                    mfw_gst_v4lsink->field = V4L2_FIELD_INTERLACED_BT;
                    break;
                default:
                    goto no_field;
                    break;
            }

    		if ((mfw_gst_v4lsink->field == V4L2_FIELD_TOP) ||
    		    (mfw_gst_v4lsink->field == V4L2_FIELD_BOTTOM) ||
    		    (mfw_gst_v4lsink->field == V4L2_FIELD_INTERLACED_TB) ||
    		    (mfw_gst_v4lsink->field == V4L2_FIELD_INTERLACED_BT)) {
    			/* For interlace feature */
    			fmt.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
    			err = ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_G_FMT, &fmt);
    			if (err < 0) {
    				g_print("VIDIOC_G_FMT failed\n");
    			}
    			if ((field == V4L2_FIELD_TOP) ||
    			    (field == V4L2_FIELD_BOTTOM))
    				fmt.fmt.pix.field = V4L2_FIELD_ALTERNATE;
    			else
    				fmt.fmt.pix.field = mfw_gst_v4lsink->field;
    			fmt.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
                GST_INFO("set v4l field:%d \n",mfw_gst_v4lsink->field);

    			err = ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_S_FMT, &fmt);
    			if (err < 0) {
    				g_print("VIDIOC_S_FMT failed\n");
    			}
    		}

        }
        
no_field:
    
#endif        
    	type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
    	if (ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_STREAMON, &type) < 0) {
    	    GST_ERROR(">>V4L_SINK: Could not stream on\n");            
        }
        mfw_gst_v4lsink->stream_on = TRUE;
    }
    
    mfw_gst_v4lsink->qbuff_count++;

    if (G_LIKELY(GST_BUFFER_FLAG_IS_SET(outbuffer, GST_BUFFER_FLAG_LAST))) {
        /* for direct rendering, try dequeue, just try twice */
        gint cnt;
        for (cnt = 0; cnt < DEQUEUE_TIMES_IN_SHOW; cnt++) {
            gint ret = 0;
            struct v4l2_buffer v4l2buf;

            memset(&v4l2buf, 0, sizeof(struct v4l2_buffer));
            v4l2buf.type = V4L2_BUF_TYPE_VIDEO_OUTPUT;
            v4l2buf.memory = V4L2_MEMORY_MMAP;

            if (mfw_gst_v4lsink->v4lqueued <= 2)
                break;
            if (!((ret = ioctl(mfw_gst_v4lsink->v4l_id, VIDIOC_DQBUF, &v4l2buf))< 0)) {
                MFWGstV4LSinkBuffer  * v4lsinkbuffer;
                v4lsinkbuffer = (MFWGstV4LSinkBuffer  *)(mfw_gst_v4lsink->all_buffer_pool[v4l2buf.index]);
                if ((v4lsinkbuffer) && (v4lsinkbuffer->bufstate == BUF_STATE_SHOWING)){
                    mfw_gst_v4lsink->v4lqueued --;
                    v4lsinkbuffer->bufstate = BUF_STATE_SHOWED;
                    g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
                    gst_buffer_unref(GST_BUFFER_CAST(v4lsinkbuffer));
                    g_mutex_lock(mfw_gst_v4lsink->pool_lock);
                }
                break;
            }

            if (cnt == (DEQUEUE_TIMES_IN_SHOW-1) ) {
                GST_WARNING("V4L_SINK: Can not dequeue buffer in show frame.\n");
                break;
            }

            if (ret<0){
                g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
                usleep(WAIT_ON_DQUEUE_FAIL_IN_MS);
                g_mutex_lock(mfw_gst_v4lsink->pool_lock);
            }
        }
    }

    g_mutex_unlock(mfw_gst_v4lsink->pool_lock);
    return GST_FLOW_OK;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_setcaps

DESCRIPTION:        This function does the capability negotiation between adjacent pad

ARGUMENTS PASSED:
        basesink    -   pointer to v4lsink
        vscapslist  -   pointer to GstCaps

RETURN VALUE:       TRUE or FALSE depending on capability is negotiated or not.

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None

=============================================================================*/

static gboolean mfw_gst_v4lsink_setcaps(GstBaseSink * basesink,
                                        GstCaps * vscapslist)
{
    MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink = MFW_GST_V4LSINK(basesink);
    guint32 format = 0;
    GstStructure *structure = NULL;
    mfw_gst_v4lsink->store_caps = vscapslist;
    gint new_width, new_height;

    gint bpp, depth;
    structure = gst_caps_get_structure(vscapslist, 0);
    gst_structure_get_fourcc(structure, "format",
			     &mfw_gst_v4lsink->fourcc);
    gst_structure_get_fraction(structure, "framerate",
			       &mfw_gst_v4lsink->framerate_n,
			       &mfw_gst_v4lsink->framerate_d);
    gst_structure_get_int (structure, "width", &new_width);
    gst_structure_get_int (structure, "height", &new_height);

    gst_structure_get_int (structure, "bpp", &bpp);
    gst_structure_get_int (structure, "depth", &depth);

    GST_DEBUG("bpp:%x, depth:%x.\n",bpp,depth);
    GST_DEBUG("setcaps: width: %d, height: %d\n", new_width, new_height);

    {
    gint sfd_val = 0;
    gboolean ret;

    ret = gst_structure_get_int(structure,"sfd",&sfd_val);
    if (ret == TRUE) {
        GST_DEBUG("sfd = %d.\n",sfd_val);
        if (sfd_val == 1)
            basesink->abidata.ABI.max_lateness = -1;

    }else {
        basesink->abidata.ABI.max_lateness = 60000000; /* 60ms */
        GST_DEBUG(">>V4L_SINK: no sfd field found in caps.\n");
    }

    }
    GST_DEBUG(">>V4L_SINK: Set max lateness = %lld.\n",basesink->abidata.ABI.max_lateness);


    if (mfw_gst_v4lsink->fourcc == GST_STR_FOURCC("I420")) {
        GST_WARNING("Default mode: I420\n");
    }else if (mfw_gst_v4lsink->fourcc==GST_STR_FOURCC("NV12")){
        GST_WARNING("set to nv12 mode\n");
        mfw_gst_v4lsink->outformat = V4L2_PIX_FMT_NV12;
    }
    else if (bpp==32) {
        GST_WARNING("set to RGB mode.\n");
        mfw_gst_v4lsink->outformat = V4L2_PIX_FMT_RGB32;
    }
    else if (bpp==24) {
        GST_WARNING("Set the V4L display to RGB24 format.\n");
        mfw_gst_v4lsink->outformat = V4L2_PIX_FMT_RGB24;
    }
    else if (bpp=16) {
        GST_WARNING("Set the V4L display to RGB565 format.\n");
        mfw_gst_v4lsink->outformat = V4L2_PIX_FMT_RGB565;
    }
    else{
        GST_ERROR("\n>>V4L_SINK: Wrong FOURCC Type for Display\n");
        return FALSE;
    }

    return TRUE;
}

#ifdef USE_X11
static GstXContext *
gst_v4lsink_xcontext_get (MFW_GST_V4LSINK_INFO_T *mfw_gst_v4lsink)
{
  GstXContext *xcontext = NULL;
    XPixmapFormatValues *px_formats = NULL;
    gint nb_formats = 0, i;

  xcontext = g_new0 (GstXContext, 1);
  xcontext->disp = XOpenDisplay (NULL);
  if (!xcontext->disp) {
    g_free (xcontext);
    return NULL;
  }

  xcontext->screen = DefaultScreenOfDisplay (xcontext->disp);
  xcontext->screen_num = DefaultScreen (xcontext->disp);
  xcontext->visual = DefaultVisual (xcontext->disp, xcontext->screen_num);
  xcontext->root = DefaultRootWindow (xcontext->disp);
  xcontext->white = XWhitePixel (xcontext->disp, xcontext->screen_num);
  xcontext->black = XBlackPixel (xcontext->disp, xcontext->screen_num);
  xcontext->depth = DefaultDepthOfScreen (xcontext->screen);

  xcontext->width = DisplayWidth (xcontext->disp, xcontext->screen_num);
  xcontext->height = DisplayHeight (xcontext->disp, xcontext->screen_num);
  xcontext->widthmm = DisplayWidthMM (xcontext->disp, xcontext->screen_num);
  xcontext->heightmm = DisplayHeightMM (xcontext->disp, xcontext->screen_num);

  g_print("xcontext: width=%d, height=%d\n", xcontext->width, xcontext->height);
  GST_DEBUG("black: %llx, white: %llx\n", xcontext->black, xcontext->white);

    /* We get supported pixmap formats at supported depth */
    px_formats = XListPixmapFormats (xcontext->disp, &nb_formats);

    if (!px_formats) {
        XCloseDisplay (xcontext->disp);
        g_free (xcontext);
        return NULL;
    }

    /* We get bpp value corresponding to our running depth */
    for (i = 0; i < nb_formats; i++) {
        if (px_formats[i].depth == xcontext->depth)
            xcontext->bpp = px_formats[i].bits_per_pixel;
    }
    XFree (px_formats);

    xcontext->endianness =
        (ImageByteOrder (xcontext->disp) ==
         LSBFirst) ? G_LITTLE_ENDIAN : G_BIG_ENDIAN;

    GST_DEBUG("depth: %d, bpp: %d, endianess: %d\n",
            xcontext->depth, xcontext->bpp, xcontext->endianness);

    return xcontext;
}
#endif
/*=============================================================================
FUNCTION:           mfw_gst_V4Lsink_change_state

DESCRIPTION:        This function keeps track of different states of pipeline.

ARGUMENTS PASSED:
        element     -   pointer to element
        transition  -   state of the pipeline

RETURN VALUE:
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstStateChangeReturn mfw_gst_v4lsink_change_state(GstElement * element,
                                                         GstStateChange transition)
{
    MFW_GST_V4LSINK_INFO_T *v4l_sink_info = MFW_GST_V4LSINK(element);

    GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
    guint8 index;
    switch (transition) {
        case GST_STATE_CHANGE_NULL_TO_READY:
            v4l_sink_info->width = -1;
            v4l_sink_info->height = -1;
            v4l_sink_info->framerate_n = 0;
            v4l_sink_info->framerate_d = 1;
            v4l_sink_info->init = FALSE;
            v4l_sink_info->buffer_alloc_called=FALSE;
            v4l_sink_info->pool_lock = g_mutex_new();
            v4l_sink_info->free_pool = NULL;
            v4l_sink_info->reservedhwbuffer_list = NULL;
            v4l_sink_info->v4lqueued = 0;
            v4l_sink_info->swbuffer_count = 0;
            v4l_sink_info->frame_dropped = 0;
            v4l_sink_info->swbuffer_max = 0;
            v4l_sink_info->cr_left_bypixel = 0;
            v4l_sink_info->cr_right_bypixel = 0;
            v4l_sink_info->cr_top_bypixel = 0;
            v4l_sink_info->cr_bottom_bypixel = 0;

#ifdef USE_X11
if (v4l_sink_info->xwin == 0) {
        gst_x_overlay_prepare_xwindow_id (GST_X_OVERLAY (v4l_sink_info));
    }
#endif
        break;
        case GST_STATE_CHANGE_READY_TO_PAUSED:
        break;
        case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
#ifdef SUSPEND_SUPPORT
        mfw_gst_v4lsink_get_runinfo(v4l_sink_info);
#endif

        break;
        default:
        break;
    }

    ret = GST_ELEMENT_CLASS(v4l_sink_info->parent_class)->
    change_state(element, transition);

    switch (transition) {
        case GST_STATE_CHANGE_PAUSED_TO_READY:
	    break;
        case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
        {    

#ifdef SUSPEND_SUPPORT
            mfw_gst_v4lsink_get_runinfo(v4l_sink_info);
#endif            
            
        }
        break;
        case GST_STATE_CHANGE_READY_TO_NULL:
        {
            mfw_gst_v4lsink_close(v4l_sink_info);
            v4l_sink_info->init = FALSE;
            g_mutex_free(v4l_sink_info->x_lock);
            g_mutex_free(v4l_sink_info->flow_lock);

#ifdef USE_X11

            if(v4l_sink_info->xcontext) {
                XCloseDisplay (v4l_sink_info->xcontext->disp);
                g_free (v4l_sink_info->xcontext);
            }
#endif
            break;
        }
        default:			/* do nothing */
        break;
    }
    return ret;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_init

DESCRIPTION:        Create the pad template that has been registered with the
                    element class in the _base_init and do library table
                    initialization

ARGUMENTS PASSED:
        v4l_sink_info  -    pointer to v4lsink element structure

RETURN VALUE:       NONE
PRE-CONDITIONS:     _base_init and _class_init are called
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static void mfw_gst_v4lsink_init(MFW_GST_V4LSINK_INFO_T * v4l_sink_info,
				                 MFW_GST_V4LSINK_INFO_CLASS_T * klass)
{
    v4l_sink_info->all_buffer_pool=NULL;
    v4l_sink_info->disp_height = 0;
    v4l_sink_info->disp_width = 0;
    v4l_sink_info->axis_top = 0;
    v4l_sink_info->axis_left = 0;
    v4l_sink_info->rotate = 0;
    v4l_sink_info->crop_left = 0;
    v4l_sink_info->crop_top = 0;
#if 0    
#ifndef _MX233
    v4l_sink_info->fullscreen_width = 240;
    v4l_sink_info->fullscreen_height = 320;
#else
    v4l_sink_info->fullscreen_width = 320;
    v4l_sink_info->fullscreen_height = 240;
#endif
#endif

    v4l_sink_info->full_screen = FALSE;
    v4l_sink_info->base_offset = 0;
    v4l_sink_info->parent_class = g_type_class_peek_parent(klass);
#ifdef ENABLE_TVOUT
/*For TV-Out & para change on-the-fly*/
    v4l_sink_info->tv_out = FALSE;
    v4l_sink_info->tv_mode = NV_MODE;
#endif
#ifdef ENABLE_DUMP
    v4l_sink_info->enable_dump = FALSE;
    v4l_sink_info->dump_location = NULL;
    v4l_sink_info->dumpfile = NULL;
    v4l_sink_info->dump_length = 0;
    v4l_sink_info->cr_left_bypixel_orig = 0;
    v4l_sink_info->cr_right_bypixel_orig = 0;
    v4l_sink_info->cr_top_bypixel_orig = 0;
    v4l_sink_info->cr_bottom_bypixel_orig = 0;
#endif
    v4l_sink_info->setpara = PARAM_NULL;
    v4l_sink_info->outformat = V4L2_PIX_FMT_YUV420;
#ifdef USE_X11
    v4l_sink_info->stream_on = FALSE;
    v4l_sink_info->changecnt = 0;
    v4l_sink_info->running = FALSE;
    v4l_sink_info->fd_fb = 0;
    v4l_sink_info->xwin = 0;
    v4l_sink_info->gc =0;
    v4l_sink_info->xcontext = NULL;
#endif

    v4l_sink_info->stretch = TRUE;
    v4l_sink_info->x_lock = g_mutex_new ();
    v4l_sink_info->flow_lock = g_mutex_new();
    v4l_sink_info->field = V4L2_FIELD_ANY;

#ifdef USE_X11
    v4l_sink_info->x11enabled = TRUE;
#endif

    
#define MFW_GST_V4LSINK_PLUGIN VERSION
    PRINT_PLUGIN_VERSION(MFW_GST_V4LSINK_PLUGIN);
    return;


}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_class_init

DESCRIPTION:        Initialise the class only once (specifying what signals,
                    arguments and virtual functions the class has and
                    setting up global state)

ARGUMENTS PASSED:
            klass   -   pointer to mp3decoder element class

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsink_class_init(MFW_GST_V4LSINK_INFO_CLASS_T * klass)
{

    GObjectClass *gobject_class;
    GstElementClass *gstelement_class;
    GstBaseSinkClass *gstvs_class;

    gobject_class = (GObjectClass *) klass;
    gstelement_class = (GstElementClass *) klass;
    gstvs_class = (GstBaseSinkClass *) klass;

    gstelement_class->change_state =
	GST_DEBUG_FUNCPTR(mfw_gst_v4lsink_change_state);

    gobject_class->set_property = mfw_gst_v4lsink_set_property;

    gobject_class->get_property = mfw_gst_v4lsink_get_property;

    gstvs_class->set_caps = GST_DEBUG_FUNCPTR(mfw_gst_v4lsink_setcaps);
    gstvs_class->render = GST_DEBUG_FUNCPTR(mfw_gst_v4lsink_show_frame);
    gstvs_class->buffer_alloc = GST_DEBUG_FUNCPTR(mfw_gst_v4lsink_buffer_alloc);
#if 0
    g_object_class_install_property(gobject_class, PROP_FULLSCREEN,
				    g_param_spec_boolean("fullscreen",
							 "Fullscreen",
							 "If true it will be Full screen",
							 FALSE,
							 G_PARAM_READWRITE));
#endif

    g_object_class_install_property(gobject_class, DISP_WIDTH,
				    g_param_spec_int("disp-width",
						     "Disp_Width",
						     "gets the width of the image to be displayed",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, DISP_HEIGHT,
				    g_param_spec_int("disp-height",
						     "Disp_Height",
						     "gets the height of the image to be displayed",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, AXIS_TOP,
				    g_param_spec_int("axis-top",
						     "axis-top",
						     "gets the top co-ordinate of the origin of display",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, AXIS_LEFT,
				    g_param_spec_int("axis-left",
						     "axis-left",
						     "gets the left co-ordinate of the origin of display",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));
#ifndef _MX233
    g_object_class_install_property(gobject_class, ROTATE,
				    g_param_spec_int("rotate", "Rotate",
						     "gets the angle at which display is to be rotated",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));
#endif
    g_object_class_install_property(gobject_class, CROP_LEFT,
				    g_param_spec_int("crop_left_by_pixel",
						     "crop-left-by-pixel",
						     "set the input image cropping in the left (width)",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, CROP_RIGHT,
				    g_param_spec_int("crop_right_by_pixel",
						     "crop-right-by-pixel",
						     "set the input image cropping in the right (width)",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));


    g_object_class_install_property(gobject_class, CROP_TOP,
				    g_param_spec_int("crop_top_by_pixel",
						     "crop-top-by-pixel",
						     "set the input image cropping in the top (height)",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, CROP_BOTTOM,
				    g_param_spec_int("crop_bottom_by_pixel",
				     "crop-bottom-by-pixel",
				     "set the input image cropping in the bottom (height)",
						     0, G_MAXINT, 0,
						     G_PARAM_READWRITE));

    g_object_class_install_property(gobject_class, FULL_SCREEN_WIDTH,
				    g_param_spec_int("fullscreen-width",
						     "fullscreen-width",
						     "gets the full screen width of the display",
						     0, G_MAXINT, 0,
						     G_PARAM_READABLE));

    g_object_class_install_property(gobject_class, FULL_SCREEN_HEIGHT,
				    g_param_spec_int("fullscreen-height",
						     "fullscreen-height",
						     "gets the full screen height of the display",
						     0, G_MAXINT, 0,
						     G_PARAM_READABLE));

    g_object_class_install_property(gobject_class, IMAGE_WIDTH,
                    g_param_spec_int("width",
                             "image-width",
                             "gets the source image width",
                             0, G_MAXINT, 0,
                             G_PARAM_READABLE));

    g_object_class_install_property(gobject_class, IMAGE_HEIGHT,
                    g_param_spec_int("height",
                             "image-height",
                             "gets the source image height",
                             0, G_MAXINT, 0,
                             G_PARAM_READABLE));
#if 0
    g_object_class_install_property(gobject_class, BASE_OFFSET,
				    g_param_spec_int("base-offset",
						     "base_offset",
						     "gets the base offet for the V4L Driver for a given BSP",
						     0, 1, 0,
						     G_PARAM_READWRITE));
#endif
/*For TV-Out & para change on-the-fly*/
    g_object_class_install_property(gobject_class, SETPARA,
				   g_param_spec_int ("setpara", "Setpara",
							 "set parameter of V4L2, 1: Set V4L 2: Set ColorKey",
							 0,3,0,
							 G_PARAM_READWRITE));
#ifdef ENABLE_TVOUT
    g_object_class_install_property(gobject_class, TV_OUT,
				   g_param_spec_boolean ("tv-out", "TV-OUT",
							 "set output to TV-OUT",
							 FALSE,
							 G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class, TV_MODE,
				   g_param_spec_int ("tv-mode", "TV-MODE",
							 "set mode to TV-OUT",
							 0, G_MAXINT, 0,
							 G_PARAM_READWRITE));
#endif
#ifdef ENABLE_DUMP
    g_object_class_install_property(gobject_class, DUMP_LOCATION,
				   g_param_spec_string ("dump_location", "Dump File Location",
							 "Location of the file to write cropped video YUV stream.",
							 NULL,
							 G_PARAM_READWRITE));
#endif


    g_object_class_install_property(gobject_class, ADDITIONAL_BUFFER_DEPTH,
    				    g_param_spec_int("add-buffer-len",
    				     "addtional buffer length",
    				     "set addtional buffer length",
    						     0, G_MAXINT, 0,
    						     G_PARAM_READWRITE));
    g_object_class_install_property(gobject_class, PROP_STRETCH,
				    g_param_spec_boolean("stretch",
							 "Stretch",
							 "If true it will be stretched video",
							 TRUE,
							 G_PARAM_READWRITE));

#ifdef USE_X11
    g_object_class_install_property(gobject_class, PROP_X11ENABLED,
				    g_param_spec_boolean("x11enable",
							 "X11Enable",
							 "Enabled x11 event handle",
							 TRUE,
							 G_PARAM_READWRITE));
#endif

    return;

}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_base_init

DESCRIPTION:       v4l Sink element details are registered with the plugin during
                   _base_init ,This function will initialise the class and child
                    class properties during each new child class creation

ARGUMENTS PASSED:
            Klass   -   void pointer

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void mfw_gst_v4lsink_base_init(gpointer g_class)
{
    GstElementClass *element_class = GST_ELEMENT_CLASS(g_class);
    GstCaps *capslist;
    GstPadTemplate *sink_template = NULL;
    gint i;
    guint32 formats[] = {
	GST_MAKE_FOURCC('I', '4', '2', '0'),
	GST_MAKE_FOURCC('Y', 'V', '1', '2'),
	GST_MAKE_FOURCC('Y', 'U', 'Y', '2'),
	GST_MAKE_FOURCC('N', 'V', '1', '2')

    };

    /* make a list of all available caps */
    capslist = gst_caps_new_empty();
    for (i = 0; i < G_N_ELEMENTS(formats); i++) {
	gst_caps_append_structure(capslist,
				  gst_structure_new("video/x-raw-yuv",
						    "format", GST_TYPE_FOURCC,formats[i],
                            "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
                            "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
                            "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, 100, 1,
                             NULL));
    }
	gst_caps_append_structure(capslist,
				  gst_structure_new("video/x-raw-rgb",
				  "bpp",GST_TYPE_INT_RANGE, 1,32,
				  "depth",GST_TYPE_INT_RANGE, 1,32,
						    NULL));

    sink_template = gst_pad_template_new("sink",
					 GST_PAD_SINK, GST_PAD_ALWAYS,
					 capslist);

    gst_element_class_add_pad_template(element_class, sink_template);
    gst_element_class_set_details(element_class, &mfw_gst_v4lsink_details);

    GST_DEBUG_CATEGORY_INIT(mfw_gst_v4lsink_debug, "mfw_v4lsink", 0,
			    "V4L video sink element");
    return;

}

#ifdef USE_X11
static gboolean
gst_v4lsink_interface_supported (GstImplementsInterface * iface, GType type)
{
  g_assert (type == GST_TYPE_X_OVERLAY);
  return TRUE;
}

static void
gst_v4lsink_interface_init (GstImplementsInterfaceClass * klass)
{
  klass->supported = gst_v4lsink_interface_supported;
}
#endif

/*=============================================================================
FUNCTION:           mfw_gst_v4lsink_get_type

DESCRIPTION:        Interfaces are initiated in this function.you can register one
                    or more interfaces  after having registered the type itself.

ARGUMENTS PASSED:   None

RETURN VALUE:       A numerical value ,which represents the unique identifier
                    of this element(v4lsink)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
GType mfw_gst_v4lsink_get_type(void)
{
    static GType mfwV4Lsink_type = 0;

    if (!mfwV4Lsink_type) {
	  static const GTypeInfo mfwV4Lsink_info = {
	    sizeof(MFW_GST_V4LSINK_INFO_CLASS_T),
	    mfw_gst_v4lsink_base_init,
	    NULL,
	    (GClassInitFunc) mfw_gst_v4lsink_class_init,
	    NULL,
	    NULL,
	    sizeof(MFW_GST_V4LSINK_INFO_T),
	    0,
	    (GInstanceInitFunc) mfw_gst_v4lsink_init,
	  };

#ifdef USE_X11
    static const GInterfaceInfo iface_info = {
      (GInterfaceInitFunc) gst_v4lsink_interface_init,
      NULL,
      NULL,
    };

    static const GInterfaceInfo overlay_info = {
      (GInterfaceInitFunc) gst_ximagesink_xoverlay_init,
      NULL,
      NULL,
    };

#endif

    mfwV4Lsink_type = g_type_register_static(GST_TYPE_VIDEO_SINK,
						 "MFW_GST_V4LSINK_INFO_T",
						 &mfwV4Lsink_info, 0);
#ifdef USE_X11
    g_type_add_interface_static (mfwV4Lsink_type, GST_TYPE_IMPLEMENTS_INTERFACE,
        &iface_info);
    g_type_add_interface_static (mfwV4Lsink_type, GST_TYPE_X_OVERLAY,
        &overlay_info);

#endif

    }

    GST_DEBUG_CATEGORY_INIT(mfw_gst_v4lsink_debug, "mfw_v4lsink",
			    0, "V4L sink");

    mfw_gst_v4lsink_buffer_get_type();

    return mfwV4Lsink_type;
}



/*=============================================================================
FUNCTION:           plugin_init

DESCRIPTION:        Special function , which is called as soon as the plugin or
                    element is loaded and information returned by this function
                    will be cached in central registry

ARGUMENTS PASSED:
        plugin     -    pointer to container that contains features loaded
                        from shared object module

RETURN VALUE:
        return TRUE or FALSE depending on whether it loaded initialized any
        dependency correctly

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean plugin_init(GstPlugin * plugin)
{
    if (!gst_element_register(plugin, "mfw_v4lsink", GST_RANK_PRIMARY,
			      MFW_GST_TYPE_V4LSINK))
        return FALSE;

    return TRUE;
}

/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/

GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/* major version of Gstreamer */
		  GST_VERSION_MINOR,	/* minor version of Gstreamer    */
		  "mfw_v4lsink",	/* name of the plugin            */
		  "Video display plug in based on V4L2",	/* what plugin actually does     */
		  plugin_init,	/* first function to be called   */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  "Freescale Semiconductor", "www.freescale.com ")
