/*
 * Copyright (C) 2008-2009 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_debug.h
 *
 * Description:    Head file of debug functions for Gstreamer plugins.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 *
 * Aug, 22 2008 Sario HU<b01138@freescale.com>
 * - Initial version
 */


#ifndef __MFW_GST_DEBUG_H__
#define __MFW_GST_DEBUG_H__

#ifndef MAX_SIZE_WARNING_THRESHOLD
#define MAX_SIZE_WARNING_THRESHOLD 1000000//1M
#endif


typedef struct _memdesc{
    struct _memdesc * next;
    guint size;
    void * mem;
    char * desstring;
    guint age;
}Mem_Desc;

typedef struct _mem_manager{
    Mem_Desc * allocatedbuffer;
    Mem_Desc * freelist;
    Mem_Desc * head;
    Mem_Desc * tail;
    guint age;
    guint size;
    guint maxsize;
    guint num;
    guint maxnum;
    guint allocatednum; 
    char * shortname;
}Mem_Mgr;


#define COPYMEMORYDESC(des, src) \
    do { \
        des->size = src->size; \
        des->mem = src->mem; \
        des->desstring = src->desstring;\
        des->age = src->age;\
    }while(0)

static Mem_Mgr mm_mgr = {0};



static Mem_Desc * new_mem_desc()
{
    Mem_Desc * newbuffer;
    if (mm_mgr.freelist){
        newbuffer = mm_mgr.freelist;
        mm_mgr.freelist = newbuffer->next;
        return newbuffer;
    }
    if (mm_mgr.allocatednum)
        mm_mgr.allocatednum <<=1;
    else
        mm_mgr.allocatednum = 4;
    if (newbuffer=g_malloc(sizeof(Mem_Desc)*mm_mgr.allocatednum)) {
        Mem_Desc *oldhead, *nb;
        int i = 0;
        
        oldhead = mm_mgr.head;
        nb = newbuffer;
        mm_mgr.freelist = mm_mgr.head = mm_mgr.tail = NULL;
        for (i=0;i<(mm_mgr.allocatednum-1);i++){
            if (oldhead){
                COPYMEMORYDESC(nb, oldhead);
                nb->next = NULL;
                if (mm_mgr.tail){
                    (mm_mgr.tail)->next = nb;
                    mm_mgr.tail = nb;
                }else{
                    mm_mgr.head = mm_mgr.tail = nb;
                }
                oldhead = oldhead->next;
            }else{
                nb->next = mm_mgr.freelist;
                mm_mgr.freelist = nb;
            }
            nb++;
        }
        if (mm_mgr.allocatedbuffer){
            g_free(mm_mgr.allocatedbuffer);
        }
        mm_mgr.allocatedbuffer = newbuffer;
        return nb;
    }else{
        return newbuffer;
    }
}



static void dbg_mem_init(char * shortname)
{
    memset(&mm_mgr, 0, sizeof(Mem_Mgr));
    if (shortname){
        mm_mgr.shortname = shortname;
    }else{
        mm_mgr.shortname = __FILE__;
    }
}

static void dbg_mem_print_nonfree()
{
    Mem_Desc * bt = mm_mgr.head;
    g_print(PURPLE_STR("%s:\t Non-freed memory list:\n", mm_mgr.shortname));
    int i = 0;
    while(bt){
        g_print("[%03d]mem 0x%p,\tsize = %ld,\tage = %ld.", i++, bt->mem, bt->size, bt->age);
        if (bt->desstring){
            g_print(" desc: %s\n", bt->desstring);
        }else{
            g_print("\n");
        }
        bt=bt->next;
    }
    g_print(PURPLE_STR("%s:\t End.\n", mm_mgr.shortname));
}

static void dbg_mem_clear_nonfree()
{
    Mem_Desc * bt = mm_mgr.head;
    while(bt){
        g_free(bt->mem);
        bt=bt->next;
    }
}

    
static void dbg_mem_deinit()
{

    dbg_mem_print_nonfree();
    dbg_mem_clear_nonfree();
    if (mm_mgr.allocatedbuffer){
        g_free(mm_mgr.allocatedbuffer);
    }
    memset(&mm_mgr, 0, sizeof(Mem_Mgr));
}


static void * dbg_mem_alloc(guint size, char * desc)
{
    Mem_Desc * bt;
    void * buf = NULL;
    if (size>MAX_SIZE_WARNING_THRESHOLD)
        g_print(RED_STR("%s: try to allocate large memory %ld bytes %s\n", mm_mgr.shortname, size, desc ? desc : "no description"));
    if ((buf = g_malloc(size)) && (bt = new_mem_desc())){
        mm_mgr.age++;
        mm_mgr.size+=size;
        mm_mgr.num++;
        if ((mm_mgr.num>mm_mgr.maxnum) || (mm_mgr.size>mm_mgr.maxsize)){
            if (mm_mgr.num>mm_mgr.maxnum){
                mm_mgr.maxnum = mm_mgr.num;
            }
            if (mm_mgr.size>mm_mgr.maxsize){
                mm_mgr.maxsize = mm_mgr.size;
            }
            g_print("%s:\t mem exceed %ld:%ld / %ld:%ld (pcs/bytes)\n", mm_mgr.shortname, 
                mm_mgr.num, mm_mgr.maxnum, mm_mgr.size, mm_mgr.maxsize);
        }
        bt->size = size;
        bt->age = mm_mgr.age;
        bt->mem = buf;
        bt->desstring = desc;
        bt->next = NULL;
       
        if (mm_mgr.tail){
            (mm_mgr.tail)->next = bt;
            mm_mgr.tail = bt;
        }else{
            mm_mgr.head = mm_mgr.tail = bt;
        }
    }else{
        if (buf){
            g_free(buf);
            buf = NULL;
        }else{
            g_print(RED_STR("%s:\t Can not allocate %ld bytes - %s\n", mm_mgr.shortname, size, desc));
        }
        g_print(PURPLE_STR("FATAL ERROR: Can not allocate memory for memmanager!!\n", 0));
    }
    return buf;
}

static void dbg_mem_free(void * mem, char * desc)
{
    Mem_Desc * bt = mm_mgr.head, *btpr = NULL;
    while(bt){
        if (bt->mem==mem){
            mm_mgr.size-=bt->size;
            mm_mgr.num--;
            g_free(mem);
            if (btpr){
                btpr->next = bt->next;
                if (mm_mgr.tail==bt){
                    mm_mgr.tail = btpr;
                }
            }else{//head
                mm_mgr.head = bt->next;
                if (mm_mgr.head==NULL){
                    mm_mgr.tail = NULL;
                }
            }
            bt->next = mm_mgr.freelist;
            mm_mgr.freelist = bt;
            return;
        }
        btpr = bt;
        bt=bt->next;
    }
    g_print(PURPLE_STR("%s Non-exist mem to free %p",mm_mgr.shortname, mem));
    
    if (desc){
        g_print(PURPLE_STR("- %s\n", desc));
    }else{
        g_print("\n");
    }
}

static void dbg_print_buffer(char * buf, int len)
{
    int i;
    for (i=0;i<len;i++){
        if ((i&0xf)==0){
            printf("\n%06x: ", i);
        }
        g_print("%02x ", buf[i]);
        
    }
    g_print("\n");
}

#endif //__MFW_GST_DEBUG_H__

