//
//  libavg - Media Playback Engine. 
//  Copyright (C) 2003-2008 Ulrich von Zadow
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//  Current versions can be found at www.libavg.de
//

#include "DisplayEngine.h"

#include "../base/Logger.h"
#include "../base/ScopeTimer.h"
#include "../base/Profiler.h"
#include "../base/Exception.h"
#include "../base/ObjectCounter.h"

using namespace std;

namespace avg {

DisplayEngine::DisplayEngine()
    : m_NumFrames(0),
      m_VBRate(0),
      m_Framerate(30),
      m_bInitialized(false),
      m_StartFramerateCalcTime(0),
      m_EffFramerate(0)
{
    ObjectCounter::get()->incRef(&typeid(*this));
}

DisplayEngine::~DisplayEngine()
{
    ObjectCounter::get()->decRef(&typeid(*this));
}


void DisplayEngine::initRender()
{
    bool bUseVBlank = false;
    if (m_VBRate != 0) {
        bUseVBlank = initVBlank(m_VBRate);
        m_Framerate = getRefreshRate()/m_VBRate;
        if (!bUseVBlank) {
            AVG_TRACE(Logger::WARNING, "Using framerate of " << m_Framerate << 
                    " instead of VBRate of " << m_VBRate);
        }
    }
    m_NumFrames = 0;
    m_FramesTooLate = 0;
    m_TimeSpentWaiting = 0;
    m_StartTime = TimeSource::get()->getCurrentMillisecs();
    m_LastFrameTime = m_StartTime*1000;
    m_LastVideoFrameTime = 0;
    m_bInitialized = true;
    if (!bUseVBlank) {
        m_VBRate = 0;
    }
}

void DisplayEngine::deinitRender()
{
    AVG_TRACE(Logger::PROFILE, "Framerate statistics: ");
    AVG_TRACE(Logger::PROFILE, "  Total frames: " << m_NumFrames);
    double TotalTime = double(TimeSource::get()->getCurrentMillisecs()
            -m_StartTime)/1000;
    AVG_TRACE(Logger::PROFILE, "  Total time: " << TotalTime << " seconds");
    AVG_TRACE(Logger::PROFILE, "  Framerate achieved: " 
            << (m_NumFrames+1)/TotalTime);
    AVG_TRACE(Logger::PROFILE, "  Frames too late: " << m_FramesTooLate);
    AVG_TRACE(Logger::PROFILE, "  Percent of time spent waiting: " 
            << double (m_TimeSpentWaiting)/(10*TotalTime));
    if (m_Framerate != 0) {
        AVG_TRACE(Logger::PROFILE, "  Framerate goal was: " << m_Framerate);
    }
    m_bInitialized = false;
}

void DisplayEngine::setFramerate(double rate)
{
    if (rate != 0 && m_bInitialized) {
        // TODO: Is this nessesary?
        initVBlank(0);
    }
    m_Framerate = rate;
    m_VBRate = 0;
}

double DisplayEngine::getFramerate()
{
    return m_Framerate;
}

double DisplayEngine::getEffectiveFramerate()
{
    return m_EffFramerate;
}

bool DisplayEngine::setVBlankRate(int rate) {
    m_VBRate = rate;
    if (m_bInitialized) {
        bool bOK = initVBlank(rate);
        if (bOK && rate != 0) { 
            m_Framerate = 0;
            return true;
        } else {
            return false;
        }
    } else {
        return true;
    }
}

bool DisplayEngine::wasFrameLate()
{
    return m_bFrameLate;
}

static ProfilingZone WaitProfilingZone("Render - wait");

void DisplayEngine::frameWait()
{
    ScopeTimer Timer(WaitProfilingZone);

    m_NumFrames++;
    calcEffFramerate();

    m_FrameWaitStartTime = TimeSource::get()->getCurrentMillisecs();
    m_TargetTime = m_LastFrameTime+(long long)(1000000/m_Framerate);
    if (m_VBRate != 0) {
        m_bFrameLate = !vbWait(m_VBRate);
        m_LastVideoFrameTime += (long long)(1000000/m_Framerate);
    } else {
        m_bFrameLate = false;
        if (m_FrameWaitStartTime <= m_TargetTime/1000) {
            long long WaitTime = m_TargetTime/1000-m_FrameWaitStartTime;
            if (WaitTime > 5000) {
                AVG_TRACE (Logger::WARNING, 
                        "DisplayEngine: waiting " << WaitTime << " ms.");
            }
            TimeSource::get()->sleepUntil(m_TargetTime/1000);
        }
    }
}

long long DisplayEngine::getDisplayTime() 
{
    if (m_VBRate == 0) {
        return m_LastFrameTime/1000-m_StartTime;
    } else {
        return m_LastVideoFrameTime/1000;
    }
}

void DisplayEngine::checkJitter()
{
    m_LastFrameTime = TimeSource::get()->getCurrentMillisecs()*1000;
    int maxDelay;
    if (m_VBRate == 0) {
        maxDelay = 2;
    } else {
        maxDelay = 6;
    }
    if ((m_LastFrameTime - m_TargetTime)/1000 > maxDelay || m_bFrameLate) {
        AVG_TRACE (Logger::PROFILE_LATEFRAMES, 
                "DisplayEngine: frame too late by " 
                << (m_LastFrameTime - m_TargetTime)/1000 << " ms.");
        m_bFrameLate = true;
        m_FramesTooLate++;
    }
    m_TimeSpentWaiting += m_LastFrameTime/1000-m_FrameWaitStartTime;
}
   
void DisplayEngine::calcEffFramerate()
{
    long long CurIntervalTime = TimeSource::get()->getCurrentMillisecs()
            -m_StartFramerateCalcTime;
    m_EffFramerate = 1000.0/CurIntervalTime;
    m_StartFramerateCalcTime = TimeSource::get()->getCurrentMillisecs();
}

}
