/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * dev.c --
 *
 * Device half of Linux kernel module implementation of driver for the
 * VMware Host/Guest filesystem.
 */

/* Must come before any kernel header file */
#include "driver-config.h"

/* for our /proc entry */
#include <linux/proc_fs.h>
#include <linux/poll.h>

#include "main.h"
#include "hgfsProto.h"
#include "hgfsDevLinux.h"       // for communication with pserver


/*
 * Our entries in the /proc filesystem
 */

static struct proc_dir_entry *hgfsProcFsEntry;


/*
 *----------------------------------------------------------------------
 *
 * HgfsClientWaitingOnRequest --
 *
 *    Determine if there is a process currently waiting for
 *    the reply to this request.
 *
 * Results:
 *    Returns NULL if there is no process waiting on this request,
 *    pointer to task_struct of waiting process if there is.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline struct task_struct*
HgfsClientWaitingOnRequest(HgfsReq *req) // IN: Request
{
   ASSERT(req);
   return req->client;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsWakeWaitingClient --
 *
 *    Wakes up the client process waiting for the reply to this
 *    request.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline void
HgfsWakeWaitingClient(HgfsReq *req)  // IN: Request
{
   ASSERT(req);
   wake_up_process(req->client);
}


/*
 * HGFS device operations. These functions get called when a userland
 * program does a syscall on our /proc device. The pointers to these
 * functions get set in HgfsSetupProcDevice, which is called when the
 * module gets loaded.
 */


/*
 *----------------------------------------------------------------------
 *
 * HgfsInitSuperInfo --
 *
 *    Allocate and initialize a new HgfsSuperInfo object
 *
 * Results:
 *    Returns a new HgfsSuperInfo object with all its fields initialized.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static HgfsSuperInfo *
HgfsInitSuperInfo(void)
{
   HgfsSuperInfo *si;

   si = kmalloc(sizeof *si, GFP_KERNEL);
   if (si == NULL) {
      return NULL;
   }

   /* Initialize fields */
   memset(si, 0, sizeof *si);
   init_waitqueue_head(&si->waitQ);
   INIT_LIST_HEAD(&si->outQ);
   si->nextIno = HGFS_START_INO;

   return si;
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsDevOpen --
 *
 *    The main entry point of the proc side of the driver. Called when a
 *    userland process does an open(2) on the proc device, in order to become
 *    an hgfs request server. This makes the whole driver transition to its
 *    initial state
 *
 * Results:
 *    0 on success
 *    error < 0 on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static int
HgfsDevOpen(struct inode *inode, // IN: Inode of the proc file
            struct file *file)   // IN: File for this process' proc file access
{
   HgfsSuperInfo *si;

   ASSERT(inode);
   ASSERT(file);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevOpen: was called\n"));

   if ((file->f_mode & (FMODE_READ | FMODE_WRITE))
          != (FMODE_READ | FMODE_WRITE)) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevOpen: wrong access mode\n"));
      return -EINVAL;
   }

   si = HgfsInitSuperInfo();
   if (!si) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevOpen: superinfo init failed\n"));
      return -ENOMEM;
   }

   ASSERT(si->userFile == NULL);
   /* Tell the filesystem side of the driver that the proc side is here. */
   si->userFile = file;

   HGFS_SET_DEV_TO_COMMON(file, si);

   //   atomic_inc(&userFile->f_count);  // XXX do we need this?

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsWaitRequest --
 *
 *       Wait for a filesystem request. This is called by HgfsDevRead in
 *       the context of the server process. Adds this process to the
 *       wait queue on the superblock associated with this server
 *       process and schedules until there is a request waiting on
 *       the out queue.
 *
 *       Called with the big lock held, releases it while calling
 *       schedule() and then grabs it again.
 *
 * Results: none
 *
 * Side effects: none
 *
 *----------------------------------------------------------------------
 */

static void
HgfsWaitRequest(HgfsSuperInfo *si)  // IN/OUT: Superblock info for this mount
{
   DECLARE_WAITQUEUE(wait, current);

   ASSERT(si);

   add_wait_queue_exclusive(&si->waitQ, &wait);

   while ((si->sb) && (list_empty(&si->outQ))) {
      set_current_state(TASK_INTERRUPTIBLE);
      if (signal_pending(current)) {
         /*
          * We caught a signal, return.
          */
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsWaitRequest: caught a "
                 "signal while waiting, exiting\n"));
	 break;
      }
      spin_unlock(&hgfsBigLock);
      schedule();
      spin_lock(&hgfsBigLock);
   }
   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsWaitRequest: got a request\n"));

   set_current_state(TASK_RUNNING);
   remove_wait_queue(&si->waitQ, &wait);
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsCopyOutToUser --
 *
 *    Copy bytes out to the user from a kernel buffer. Increments the
 *    user buffer pointer and decrements the buffer size count so that
 *    it can be called from a loop to send multiple packets if so
 *    desired. On return, *dstBufP points to the next free byte in the
 *    user buffer and *dstSizeP contains the remaining free space in
 *    the user buffer.
 *
 * Results:
 *    Returns zero on success and a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline int
HgfsCopyOutToUser(char **dstBufP,   // IN/OUT: Pointer to the destination (user) buffer
                  size_t *dstSizeP, // IN/OUT: Pointer to size of user buffer
                  const void *src,  // IN:     Source (kernel) buffer
                  size_t srcLength) // IN:     Length of the source buffer

{
   ASSERT(dstBufP);
   ASSERT(dstSizeP);
   ASSERT(src);

   if (*dstSizeP < srcLength) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsCopyOutToUser: user buffer too small\n"));
      return -EIO;
   }

   if (copy_to_user(*dstBufP, src, srcLength)) {
      return -EFAULT;
   }

   *dstBufP += srcLength;
   *dstSizeP -= srcLength;

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsCopyOutToUser: copied %Zu bytes to user\n",
           srcLength));

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsSendPacketToUser --
 *
 *    Send a HgfsPacket to the user. Pass it along to HgfsCopyOutToUser,
 *    which just does the raw copy to the user buffer.
 *
 * Results:
 *    Returns the number of bytes sent on success or a negative error
 *    on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline int
HgfsSendPacketToUser(char *packetOut,   // IN: Packet to send.
                     uint32 packetSize, // IN: Size of the packet.
                     char *buf,         // IN: User buffer to copy to
                     size_t count)      // IN: Size of the user buffer
{
   int error;
   size_t numBytesTried = count;

   ASSERT(packetOut);
   ASSERT(buf);
   ASSERT(packetSize <= HGFS_PACKET_MAX);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsSendPacketToUser: trying to write %u bytes\n",
           packetSize));

   error = HgfsCopyOutToUser(&buf, &count, packetOut, packetSize);
   if (error) {
      return error;
   }

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsSendPacketToUser: finished, "
           "wrote %Zu bytes, %Zu left\n", (numBytesTried - count), 
           count));

   return numBytesTried - count;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDevRead --
 *
 *    Called when the server process does a read on our /proc
 *    device. Sleep until a request shows up on the outQ, then copy
 *    the request out to the server process and remove the request
 *    from the outQ.
 *
 * Results:
 *    Returns the number of bytes copied into the user buffer on
 *    success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static ssize_t
HgfsDevRead(struct file *file, // IN:  File pointer of file user is reading
            char *buf,         // OUT: User buffer to copy to
            size_t count,      // IN:  Number of bytes to copy to user
            loff_t *offset)    // IN:  Offset to read from. ignored.
{
   ssize_t result;
   HgfsSuperInfo *si;
   HgfsReq *req;

   ASSERT(file);
   ASSERT(buf);
   ASSERT(offset);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevRead: called, to read %Zu bytes\n",
           count));

   si = HGFS_DEV_TO_COMMON(file);
   ASSERT(si);

   spin_lock(&hgfsBigLock);

   /*
    * If the user opened in non-blocking mode, then check to see if we
    * would block, and if so return EAGAIN.
    *
    * Otherwise, sleep until a request shows up on the outQ.
    */
   if (file->f_flags & O_NONBLOCK) {
      if (list_empty(&si->outQ)) {
         spin_unlock(&hgfsBigLock);
         return -EAGAIN;
      }
   } else {
      HgfsWaitRequest(si);
   }

   /*
    * Take the request off the outQ.
    */
   if ((si->sb) && (!list_empty(&si->outQ))) {
      req = list_entry(si->outQ.next, HgfsReq, list);
      list_del_init(&req->list);
      LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevRead: woke up, finished getting req\n"));
   } else {
      req = NULL;
   }
   spin_unlock(&hgfsBigLock);

   if (!si->sb) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRead: superblock is null\n"));
      return -ENODEV;
   }

   if (!req) {
      /*
       * We got interrupted while waiting for a request.
       */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRead: interrupted while waiting\n"));
      return -ERESTARTSYS;
   }

   /*
    * Send the packet to the server user process.
    */
   result = HgfsSendPacketToUser(req->packet, req->packetSize, buf, count);

   /*
    * Make sure the packet was sent correctly.
    */
   spin_lock(&hgfsBigLock);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRead: error sending msg to server\n"));
      req->error = -EPROTO;
      req->stateFile = HGFS_REQ_STATE_FINISHED;
      HgfsWakeWaitingClient(req);
   } else {
      LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDevRead: msg sent to server successfully\n"));
      req->stateFile = file;
      req->error = 0;
   }

   spin_unlock(&hgfsBigLock);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsCopyInFromUser --
 *
 *    Copy bytes in from a user buffer to a kernel buffer. Increments the
 *    user buffer pointer and decrements the buffer size count so that
 *    it can be called from a loop to receive multiple packets if so
 *    desired. On return, *srcBufP points to the next uncopied byte in the
 *    user buffer and *dstSizeP contains the number of uncopied bytes in
 *    the user buffer.
 *
 * Results:
 *    Returns zero on success and a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline int
HgfsCopyInFromUser(void *dst,            // IN: Destination (kernel) buffer
                   size_t dstSize,       // IN: Size of kernel buffer
                   const char **srcBufP, // IN/OUT: Pointer to source (user) buffer
                   size_t *srcLengthP)   // IN/OUT: Pointer to size of user buffer
{
   size_t copyIn;

   ASSERT(dst);
   ASSERT(srcBufP);
   ASSERT(srcLengthP);

   /*
   if (dstSize < *srcLengthP) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsCopyInFromUser: destination "
              "buffer too small\n"));
      return -EIO;
   }
   */

   /* We want to copy in whichever is smaller, the dstSize or srcSize. */
   copyIn = (dstSize < *srcLengthP) ? dstSize : *srcLengthP;

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsCopyInFromUser: about to copy_from_user\n"));
   if (copy_from_user(dst, *srcBufP, copyIn)) {
      return -EFAULT;
   }

   *srcBufP += copyIn;
   *srcLengthP -= copyIn;

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsCopyInFromUser: copied %Zu bytes from user\n",
           copyIn));

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetReplyFromUser --
 *
 *    Receive a reply from the user. Calls CopyInFromUser which does
 *    the raw copy from the user buffer.
 *
 * Results:
 *    Returns the number of bytes received on success or a negative error
 *    on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline int
HgfsGetReplyFromUser(HgfsReply *reply,    // OUT:  Kernel buffer to copy into
                     const char *userBuf, // IN:   User buffer to copy from
                     size_t count)        // IN:   Size of user buffer in bytes
{
   int error;
   size_t numBytesTried = count;

   ASSERT(reply);
   ASSERT(userBuf);

   /* XXX We already checked this in the calling function. Pick one. */
   if (count < sizeof *reply) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetReplyFromUser: write is short\n"));
      return -EIO;
   }

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsGetReplyFromUser: calling CopyIn with reply,"
           "%Zu, userBuf, %Zu\n", sizeof *reply, count));

   error = HgfsCopyInFromUser(reply, sizeof *reply, &userBuf, &count);
   if (error) {
      return error;
   }

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsGetReplyFromUser: finished, read %Zu bytes,"
           " %Zu left\n", (numBytesTried - count), count));

   return numBytesTried - count;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetPacketRemainderFromUser --
 *
 *    Get the rest of a packet from the user. Calls CopyInFromUser
 *    which does the raw copy from the user buffer.
 *
 * Results:
 *    Returns the number of bytes received on success or a negative
 *    error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline int
HgfsGetPacketRemainderFromUser(char *packetBuf,     // OUT:  Kernel buffer to copy into
                               size_t bufSize,      // IN:   Room left in kernel buffer
                               const char *userBuf, // IN:   User buffer to copy from
                               size_t count)        // IN:   Size of user buffer in bytes
{
   int error;
   size_t numBytesTried = count;

   ASSERT(packetBuf);
   ASSERT(userBuf);

   error = HgfsCopyInFromUser(packetBuf, bufSize, &userBuf, &count);
   if (error) {
      return error;
   }

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsGetPacketRemainderFromUser: finished, "
           "read %Zu bytes, %Zu left\n", (numBytesTried - count), count));

   return numBytesTried - count;
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsHandle2Req --
 *
 *    Retrieve the request a handle refers to
 *
 * Results:
 *    The request if the handle is valid (i.e. it refers to a request we last
 *     sent to file struct 'file')
 *    NULL if the handle is invalid
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static INLINE HgfsReq *
HgfsHandle2Req(HgfsHandle handle,       // IN: Packet to examine
               struct file const *file) // IN: File pointer associated with
                                        //     this write
{
   HgfsReq *req;

   ASSERT(file);

   /*
    * Handle is user-provided, so this test must be carefully
    * written to prevent wraparounds.
    */
   if (handle >= HGFS_MAX_OUTSTANDING_REQS) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsValidateReply: handle out of range!\n"));
      return NULL;
   }

   req = &requestPool[handle];

   if (req->stateFile != file) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsValidateReply: wrong fd for reply %u!\n",
              handle));
      return NULL;
   }

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsValidateReply: incoming packet handle %u valid\n",
           handle));

   return req;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDevWrite --
 *
 *    Called when the server process does a write on our /proc
 *    device. Copy a packet in from the user process, validate that it
 *    is a valid reply to a request that we sent, and then copy the
 *    packet into that request object and set its state to "finished".
 *
 * Results:
 *    Returns the number of bytes copied from the user buffer on
 *    success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static ssize_t
HgfsDevWrite(struct file *file, // IN: File pointer of file user is writing to
             const char *buf,   // IN: User buffer to copy from
             size_t count,      // IN: Number of bytes to copy
             loff_t *offset)    // IN: Offset in the file to write to. ignored.
{
   HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsReply reply;
   ssize_t result;
   size_t totalCopied;

   ASSERT(file);
   ASSERT(buf);
   ASSERT(offset);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: called to write %Zu bytes\n",
           count));

   si = HGFS_DEV_TO_COMMON(file);
   ASSERT(si);

   if (!si->sb) {
      return -EIO;
   }

   /* Server has to send us at least the size of a reply. */
   if (count < sizeof reply) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: write from user is short\n"));
      return -EPROTO;
   }

   /* Copy the reply in from the user buffer. */
   result = HgfsGetReplyFromUser(&reply, buf, count);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: failed getting reply from server\n"));
      return result;
   }

   if (result != sizeof reply) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: wrong size copied from user\n"));
      return -EPROTO;
   }

   totalCopied = sizeof reply;

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: got reply from server, id %u\n",
           reply.id));

   req = HgfsHandle2Req(reply.id, file);
   if (!req) {
      /* If the server sent us an invalid packet, return an error. */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: server sent invalid packet\n"));
      return -EPROTO;
   }

   spin_lock(&hgfsBigLock); /* XXX is it okay to have this whole thing locked? */
   if (!HgfsClientWaitingOnRequest(req)) {
      /*
       * If no client is waiting on this request, it means the client
       * process caught a signal and returned already, so we free the
       * request. HgfsFreeRequest has to be called with the big lock held.
       */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: no client waiting for "
              "reply, freeing request\n"));
      HgfsFreeRequest(req);
   } else {
      /*
       * Otherwise, copy the response into the request, copy the rest
       * of the packet into the request, set the state to "finished",
       * and wake the client process.
       */
      LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: request is active, "
              "getting rest of message\n"));
      memcpy(req->packet, &reply, sizeof reply);

      LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: calling Remainder with "
              "(packet+total), %Zu, (buf+total), %Zu\n",
              (HGFS_PACKET_MAX - sizeof reply), (count - sizeof reply)));

      result = HgfsGetPacketRemainderFromUser(req->packet + sizeof reply,
                                              HGFS_PACKET_MAX - sizeof reply,
                                              buf + sizeof reply,
                                              count - sizeof reply);
      if (result < 0) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: failed to copy remainder "
                 "of packet\n"));
         req->error = result;
      } else {
         totalCopied += result;
      }

      req->packetSize = totalCopied;
      req->stateFile = HGFS_REQ_STATE_FINISHED;
      HgfsWakeWaitingClient(req);
   }
   spin_unlock(&hgfsBigLock);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevWrite: finished\n"));
   return totalCopied;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDevPoll --
 *
 *    Called when a process (usually the pserver) calls poll or select
 *    on the hgfs device file.
 *
 * Results:
 *    Returns poll flags indicating whether the file is readable
 *    and/or writeable. Readability means that a request is waiting
 *    in the driver.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static unsigned int
HgfsDevPoll(struct file *file, // IN: File to poll
            poll_table *wait)  // IN: Poll table
{
   HgfsSuperInfo *si;
   uint32 events = POLLWRNORM | POLLOUT;

   ASSERT(file);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevPoll: was called\n"));

   si = HGFS_DEV_TO_COMMON(file);
   ASSERT(si);

   if (!si->sb) {
      return -EIO;
   }

   poll_wait(file, &si->waitQ, wait);

   spin_lock(&hgfsBigLock);
   if (!list_empty(&si->outQ)) {
      events |= POLLRDNORM | POLLIN;
   }
   spin_unlock(&hgfsBigLock);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevPoll: finished\n"));
   return events;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsCancelRequest --
 *
 *    Cancel a request and deal with waking up a client waiting on it.
 *    
 *    Must be called with the big lock held.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static void
HgfsCancelRequest(HgfsReq *req)
{
   ASSERT(req);

   /*
    * Set this request's error and state fields to cause its woken
    * client to return an error. Then wake the sleeping client (if
    * there is one) or free the request directly (if there isn't).
    */

   req->error = -ENOTCONN;
   req->stateFile = HGFS_REQ_STATE_FINISHED;

   if (HgfsClientWaitingOnRequest(req)) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsCancelRequest: Waking client\n"));
      HgfsWakeWaitingClient(req);
   } else {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsCancelRequest: No waiting client\n"));
      HgfsFreeRequest(req);
   }
}

/*
 *----------------------------------------------------------------------
 *
 * HgfsDevRelease --
 *
 *    Called when the last reference to this hgfs proc device file has
 *    been closed.
 *
 * Results:
 *    Returns zero on success, negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsDevRelease(struct inode *inode, // IN: Inode of released file
               struct file *file)   // IN: Released file
{
   HgfsSuperInfo *si;

   ASSERT(inode);
   ASSERT(file);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsDevRelease: called\n"));

   si = HGFS_DEV_TO_COMMON(file);
   ASSERT(si);

   spin_lock(&hgfsBigLock);

   ASSERT(si->userFile);
   /* Tell the filesystem side of the driver that the proc side is gone. */
   si->userFile = NULL;

   if (si->sb) {
      HgfsReq *req;

      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRelease: fs mounted\n"));
      /*
       * The state of the whole driver is 1 or 2, and we need to
       * transition to state 4 or 3.
       *
       * To do this, cancel all requests that are either waiting on
       * the outQ or have already been sent but whose replies have not
       * yet been received.
       */

      /*
       * Cancel requests waiting on the outQ. We do this first because
       * we can't distinguish between requests on the outQ and
       * requests on the free list when we walk the array. [bac]
       */
      while (!list_empty(&si->outQ)) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRelease: Removing waiting req\n"));
         req = list_entry(si->outQ.next, HgfsReq, list);
         list_del_init(&req->list);
         HgfsCancelRequest(req);
      }

      /*
       * Walk through the entire array of requests and cancel any
       * which have been sent and are still waiting for a reply.
       */
      for (req = &requestPool[0];
           req < &requestPool[HGFS_MAX_OUTSTANDING_REQS];
           req++) {
         if (req->stateFile != HGFS_REQ_STATE_INACTIVE &&
             req->stateFile != HGFS_REQ_STATE_FINISHED) {
            LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRelease: Removing pending req\n"));
            HgfsCancelRequest(req);
         }
      }

      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRelease: Done removing reqs\n"));
      spin_unlock(&hgfsBigLock);
      return 0;
   }

   /*
    * The whole driver is in the initial state. Transition to the final
    * state.
    */
   LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDevRelease: Freeing super info\n"));
   kfree(si);

   spin_unlock(&hgfsBigLock);
   return 0;
}

/* Proc device file operations */
struct file_operations HgfsProcDevOperations = {
   owner:   THIS_MODULE,
   read:    HgfsDevRead,
   write:   HgfsDevWrite,
   poll:    HgfsDevPoll,
   open:    HgfsDevOpen,
   release: HgfsDevRelease,
};


/*
 *----------------------------------------------------------------------
 *
 * HgfsCleanupProcDevice --
 *
 *    Removes the two VMware hgfs proc entries.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

void
HgfsCleanupProcDevice(void)
{
   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsCleanupProcDevice: was called\n"));

   if (hgfsProcFsEntry) {
      remove_proc_entry(HGFS_DEVICE_NAME, hgfsProcFsEntry);
      remove_proc_entry(HGFS_NAME, proc_root_fs);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsSetupProcDevice --
 *
 *    Set up the VMware hgfs entries, and initialize the global data
 *    structures.
 *
 * Results:
 *    Zero on success, negative error on failure
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

int
HgfsSetupProcDevice(void)
{
   int result;
   unsigned int i;
   struct proc_dir_entry *hgfsProcDevEntry;

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsSetupProcDevice: entered\n"));

   hgfsProcFsEntry = NULL;

   result = -EIO;
   hgfsProcFsEntry = proc_mkdir(HGFS_NAME, proc_root_fs);
   if (!hgfsProcFsEntry) {
      LOG(4, (KERN_DEBUG "VMware hgfs: failed to create entry in /proc/fs\n"));
      goto error;
   }

   hgfsProcFsEntry->owner = THIS_MODULE;
   hgfsProcDevEntry = create_proc_entry(HGFS_DEVICE_NAME, S_IFSOCK | 0600,
                                        hgfsProcFsEntry);
   if (!hgfsProcFsEntry) {
      LOG(4, (KERN_DEBUG "VMware hgfs: failed to create device in "
              "/proc/fs/%s/\n", HGFS_NAME));
      goto error;
   }

   hgfsProcDevEntry->owner = THIS_MODULE;
   hgfsProcDevEntry->proc_fops = &HgfsProcDevOperations;

   spin_lock(&hgfsBigLock);

   /*
    * Initialize global data structures.
    */
   INIT_LIST_HEAD(&reqFreeList);
   sema_init(&requestSem, HGFS_MAX_OUTSTANDING_REQS);

   /*
    * Set the id of each request, and put them all
    * on the free list.
    */
   for (i = 0; i < HGFS_MAX_OUTSTANDING_REQS; i++) {
      requestPool[i].id = i;
      requestPool[i].stateFile = HGFS_REQ_STATE_INACTIVE;
      INIT_LIST_HEAD(&requestPool[i].list);
      list_add_tail(&requestPool[i].list, &reqFreeList);
   }

   spin_unlock(&hgfsBigLock);
   return 0;

error:
   HgfsCleanupProcDevice();
   return result;
}
